/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlCurveCV - Encapsulates methods common to curve CVs.
//
//	.SECTION Description
//
//		AlCurveCV is the class used to access and manipulate
//		curve CV's (also referred to as Control Points).
//		There is one AlCurveCV object for each CV of an AlCurve as you see
//		it in the interactive Alias package.  You cannot create or delete
//		an AlCurveCV.  AlCurveCV's are only created through AlCurve's.
//
//		There are methods to query and set the multiplicity of a CV,
//		and methods to query the world position and the "unaffected"
//		position of a CV.  The "unaffected" position of a CV is its
//		position BEFORE any transformations (from its parent dag node
//		or from affecting clusters) have been applied to it.
//		There is also a method to access the clusters that affect a CV.
//		Methods setWorldPosition(*) are used for setting either the
//		unaffected or affected(clusters) position of a CV.
//
//		To print out all CV's of a curve:
//
//	.nf
// %@ AlCurve     curve;
// %@ AlCurveCV  *cvPtr;
// %@ int         i = 0;
// %@ double      x, y, z, w;
// %@
// %@ if ( cvPtr = curve->firstCV() )
// %@ {
// %@%@ do {
// %@%@%@ cvPtr->worldPosition( x, y, z, w );
// %@%@%@ printf("CV %d is at %G, %G, %G with weight %G.\n", i, x, y, z, w );
// %@%@%@ i++;
// %@%@ } while( cvPtr->nextD() == sSuccess );
// %@ }
// %@ delete cvPtr;
// %@
// %@ Example Result:
// %@          CV 1 is at 0.0, 1.0, 0.0 with weight 1.0
// %@          CV 2 is at 0.0, 2.0, 0.0 with weight 1.0
// %@          CV 3 is at 0.0, 3.0, 0.0 with weight 1.0
// %@          CV 4 is at 0.0, 4.0, 0.0 with weight 1.0
// %@
//	.fi
//
//	Multiplicity:
//
//		When an AlCurveCV has a multiplicity > 1, CV's
//		are internally piled up at that position.  In this case, an AlCurveCV
//		actually represents more than one CV internally.  To access ALL CV's,
//		use the methods in the AlCurve class to query values "including
//		multiplicity", or use the example code below.  By representing
//		AlCurveCV's this way, then if an AlCurveCV is moved, the corresponding
//		multiple CV's at that position are moved as well, acting exactly
//		like the Alias interactive package.
//
//		For example, to print all internal CV's of a curve with 4 AlCurveCV's
//		where the second AlCurveCV has multiplicity = 2, you would do this:
//
//	.nf
// %@ AlCurve curve;
// %@ AlCurveCV *cvPtr;
// %@ int i,j = 0;
// %@ double x, y, z, w;
// %@
// %@ if ( cvPtr = curve->firstCV() )
// %@ {
// %@%@ do {
// %@%@%@ for( j = 0; j <= cvPtr->multiplicity(); j++, i++ )
// %@%@%@ {
// %@%@%@%@ cvPtr->worldPosition( x, y, z, w );
// %@%@%@%@ printf("CV %d is at %G, %G, %G with weight %G.\n", i, x, y, z, w );
// %@%@%@ }
// %@%@%@ i++;
// %@%@ } while( cvPtr->nextD() == sSuccess );
// %@ }
// %@ delete cvPtr;
// %@
// %@      Example Result:
// %@          CV 1 is at 0.0, 1.0, 0.0 with weight 1.0
// %@          CV 2 is at 0.0, 2.0, 0.0 with weight 1.0
// %@          CV 3 is at 0.0, 2.0, 0.0 with weight 1.0
// %@          CV 4 is at 0.0, 3.0, 0.0 with weight 1.0
// %@          CV 5 is at 0.0, 4.0, 0.0 with weight 1.0
//	.fi
//
//	Creation:
//
//		Note that certain routines require that the given CV be installed
//		in the Dag (i.e., that the curve belongs to a AlCurveNode).  If
//		any of these are called on a CV that's not part of the Dag, they
//		will simply fail.
//
//		Routines that require the CV to be in the Dag:
//	.nf
//			next
//			nextD
//			worldPosition
//			setMultiplicity
//			setUnaffectedPosition
//			setWorldPosition
//	.fi
//


#ifndef _AlCurveCV
#define _AlCurveCV

#include <AlObject.h>
#include <AlClusterable.h>
#include <AlAnimatable.h>
#include <AlSettable.h>
#include <AlPickable.h>
#include <AlTM.h>

struct ag_cnode;
struct Dag_node;
struct Spline_surface;

class STUDIOAPI_DECL AlCurveCV : public AlObject , public AlClusterable , public AlAnimatable , public AlSettable , public AlPickable
{
	friend					class AlFriend;
public:

	virtual					~AlCurveCV();
	virtual AlObject*		copyWrapper() const;

	AlObjectType		type() const;

	AlCurveCV*			next() const;
	AlCurveCV*			prev() const;
	statusCode			nextD();
	statusCode			prevD();

	int					index() const;
	int					multiplicity() const;
	statusCode			worldPosition( double&, double&, double&, double& ) const;
	statusCode			affectedPosition( const AlTM&, double&, double&, double&, double& ) const;
	statusCode			unaffectedPosition( double&, double&, double&, double& ) const;
	statusCode			setMultiplicity( int );
	statusCode			setWorldPosition( double, double, double, double, API_BOOLEAN );
	statusCode			setWorldPosition( double, double, double, double, AlTM );
	statusCode			setWorldPosition( double, double, double, API_BOOLEAN );
	statusCode			setWorldPosition( double, double, double, AlTM );
	statusCode			setUnaffectedPosition( double, double, double, double );

	AlCurve*			curve() const;

	statusCode			blindData( int, long&, const char *& );
	statusCode 			setBlindData( int, long, const char * );
	statusCode 			removeBlindData( int );

	statusCode			doUpdates( API_BOOLEAN newState = TRUE );

protected:
						AlCurveCV( Spline_surface* );

private:
	API_BOOLEAN				updateOn;
	API_BOOLEAN				updateNeeded;
	void				updatePerform();

	virtual	API_BOOLEAN 	extractType( int&, void*&, void*& ) const;

	static void			initMessages();
	static void			finiMessages();

	Spline_surface*		fParent;
};
#endif
