/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/

//
//	.NAME AlCluster - Encapsulates creation, deletion and manipulation of clusters.
//
//	.SECTION Description
//
//		This class encapsulates the functionality for creating,
//		manipulating and deleting a cluster.  A cluster is a group of
//		dag nodes and/or curve and surface control points, which
//		are grouped as such for deformation purposes.  Every cluster
//		has a cluster dag node which is in the universe's dag.  Adding and
//		removing nodes and control points to and from a cluster
//		does not affect the topology of the universe's dag.
//		Transforming the cluster dag node affects the transformations
//		of the objects in the cluster.
//
//		Empty clusters are allowed.  An object can be in more
//		than one cluster at a time, provided that those clusters are
//		of type kMultiCluster.  When an object is added to
//		a cluster, it is given a weight that indicates how much
//		of the cluster's leaf transformation is applied to the object.
//		The default weight is 100%.  If a dag node is added to a
//		cluster the percentages of each individual CV may be manipulated
//		separately without actually adding the CVs themselves to the cluster.
//
//		To create a cluster, the user must instantiate and call
//		create on an AlCluster object.  This also creates an AlClusterNode
//		which gets attached to the AlCluster and which is inserted
//		into the universe's dag.  The user may not instantiate an
//		AlClusterNode or an AlClusterMember directly.
//
//		There are two ways to delete a cluster object.  When a cluster
//		is deleted, its attached cluster node is deleted.  Alternatively,
//		when AlClusterNode::deleteObject() is used, its cluster is deleted.
//		The dag nodes and control points in a cluster are not deleted,
//		however the AlClusterMember objects that represented the "in a
//		cluster" relation are invalidated.
//
//		Clusters don't have names.  Any attempts to query for a name
//		will return NULL.
//

#ifndef _AlCluster
#define _AlCluster

#include <AlObject.h>
#include <AlModel.h>

class AlIterator;

extern "C" {
	struct Cl_Cluster;
	struct Cl_Subcluster;
	struct Cl_Cluster_node;
	struct CL_Pw;
}

class STUDIOAPI_DECL  AlCluster : public AlObject 
{
	friend class		AlFriend;

public:
	enum PercentEffectType { kInvalid, kLeafNode, kJoint };

						AlCluster();
	virtual				~AlCluster();
	virtual statusCode	deleteObject();
	virtual AlObject*	copyWrapper() const;

	statusCode			create();

	AlObjectType		type() const;

	AlCluster*			nextCluster() const;
	statusCode			nextClusterD();

	AlCluster*			prevCluster() const;
	statusCode			prevClusterD();

	AlClusterNode*		clusterNode() const;
	API_BOOLEAN				isEmpty() const;

	int					numberOfMembers() const;
	AlClusterMember*	firstMember() const;

	statusCode			applyIteratorToMembers( AlIterator*, int& ) const;

	statusCode 			clusterRestrict( AlClusterRestrict& ) const;
	statusCode			setClusterRestrict( AlClusterRestrict );

	PercentEffectType	percentEffectType() const;
	statusCode			setPercentEffectType( PercentEffectType );

	statusCode			originalTRS( double[3], double[3], double[3] ) const;

private:
	static void			initMessages();
	static void			finiMessages();

	virtual API_BOOLEAN		extractType( int&, void *&, void *& ) const;
};

#endif
