/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//	.NAME AlCloud - Interface to Alias clouds.
//
//	.SECTION Description
//
//		AlCloud is the interface to clouds and their data in Alias.
//		This class allows a developer to perform operations
//		such as creating, naming, controlling visibility,
//		setting transformations, picking, merging, tesselating and subsetting
//		a cloud of points. In addition, points can be added to
//		a cloud that was created with the API.  Data access methods
//		to most of the settings are also provided.  For example
//		you can find the current translation, scale or pick state etc.
//
//		The number of points in a cloud and the actual points can
//		be retrieved with the class.  Please see the points() routine
//		for a description of the data structure used to retrieve points.
//		The same data structure is used in the addPoints() method.	
//
//		Importing a cloud of points is performed by the method
//		AlUniverse::importCloudFile().  Walking the cloud list
//		is performed with AlUniverse::firstCloud() in conjunction
//		with AlCloud::nextCloud().  Also AlUniverse::applyIteratorToClouds()
//		can be used to walk all cloud elements.
//
//		Notes: 
//
//		1. Tesselation of clouds will only work with CyberWare files.  As
//		a result, clouds created with the API cannot be tesselated.
//
//		2. Clouds are not in the Alias dag tree.
//
//		3. A cloud will not be an item on the Alias pick list.  Therefore,
//		AlCloud pick methods must be used instead of AlPickList methods.
//
//		4. To add arbitrary points to a cloud, you must create a cloud
//		with the new points and merge it into the existing cloud that you
//		wish to grow.  The cloud that is merged in will be
//		delete automatically in the merge process.  The new extended cloud will
//		have the outer bounding box of the two clouds.  
//
//		5. AlUniverse::importCloudFile() has a parameter which controls
//		the sampling factor when the cloud file is imported.
//  	
//

#ifndef _AlCloud
#define  _AlCloud

#include <AlObject.h>

class STUDIOAPI_DECL  AlCloud :	public AlObject
{
	friend class			AlFriend;
public:

	enum Subset 
	{
		kSubsetInside,
		kSubsetOutside,
		kSubsetBoth
	};

							AlCloud();
	virtual					~AlCloud();

	virtual AlObjectType	type() const;
	virtual AlObject		*copyWrapper() const;

	virtual statusCode		deleteObject();
	statusCode				create();

    virtual const char*     name() const;
    virtual statusCode      setName( const char* );

	API_BOOLEAN					visible();
	statusCode				setVisibility( API_BOOLEAN );

	statusCode				translation( double&, double&, double& );
	statusCode				setTranslation( double, double, double );
	statusCode				rotation( double&, double&, double& );
	statusCode				setRotation( double, double, double );
	statusCode				scale( double&, double&, double& );
	statusCode				setScale( double, double, double );

	statusCode				boundingBox( double&, double&, double&, double&, double&, double& ); 

	statusCode				subset( Subset, AlCloud **, double, double, double, double, double, double );

	statusCode				tesselate( int, int, AlPolyset *&); 
	statusCode				tessellate( double, double, double, int, AlMesh *&); 
	statusCode				merge( AlCloud * );

	int 					numberOfPoints() const;
	statusCode				points( int, float[] );
	statusCode				addPoints(double, double, double, double, double, double, int, float[] );

	statusCode				pick();
	statusCode				unpick();
	API_BOOLEAN					isPicked();

	AlCloud 				*nextCloud() const;
	statusCode				nextCloudD( AlCloud * ) const;

private:

	static void initMessages();
	static void finiMessages();
};

#endif // _AlCloud
