/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
//
//	.NAME AlBlendPoint	-  interface to Alias blend points
//
//	.SECTION Description
//
//		Alias Blend Curves contain of Alias Blend Points.  Blend Points
//		have constraints and curvature types as described in the Alias
//		manuals.  Methods for getting and setting the tangency, position,
//		constraint continuity and the geometry the point is associated
//		with are included.
//
//		Notes:
//		1. To find the Alias Blend Curve this point is on, use
//		the blendCurve() method.  
//		2. To find the Alias Curve that this point is on find the 
//		blend curve, the call the AlBlendCurve::curveNode() method.	
//		3. setPoint() will automatically disassociate a blend point
//		from any geometry it is attached to.
//		4. Do not rely on the param() value staying the same.  It will
//		be modified everytime the curve is also modified.
//		5. Attaching a point to a cloud is not currently supported. 
//		6. Some of the methods will only work if we have the proper
//		context.  For example, the interpolation direction might have
//		to be a certain type to make a method work. 
//		7. Deleting blend points can cause the blend curve to also be
//		deleted.  If there are less than two points left, then the
//		blend curve will be deleted automatically by the deleteObject()
//		method.  
//

#ifndef _AlBlendPoint
#define  _AlBlendPoint

#include <AlObject.h>

class AlSurface;
class AlCurve;

class STUDIOAPI_DECL AlBlendPoint :	public AlObject
{
	friend class			AlFriend;
	friend class			AlBlendCurve;
public:

	enum AlConstraintContinuityType
	{
		kG0,
		kG1,
		kG2,
		kG3,
		kG4
	};

	enum AlCurvatureType
	{
		kGeometricCurv,
		kParametric
	};

	enum AlInterpolationDirectionType
	{
		kLocation,
		kWorldSpace,
		kGeometricInterp
	};
	
	enum AlDirectionType 
	{
		kRay,
		kParallel
	};
	
							AlBlendPoint();
	virtual					~AlBlendPoint();

	virtual statusCode		deleteObject();
	virtual AlObject*		copyWrapper() const;

	virtual AlObjectType	type() const;

	statusCode				setPoint( double , double , double  );
	statusCode				point( double& , double& , double&  );

	statusCode 				param( double& );

	statusCode				tangent( double&, double&, double& );
	statusCode				setTangent( double, double, double );
	
	statusCode				interpolationDirectionType( AlInterpolationDirectionType& );
	statusCode				setInterpolationDirectionType( AlInterpolationDirectionType );
	
	statusCode				directionType( AlDirectionType& );
	statusCode				setDirectionType( AlDirectionType );
	
	statusCode				curvatureType( AlCurvatureType& );
	statusCode				setCurvatureType( AlCurvatureType );

	statusCode				constraintContinuityType( AlConstraintContinuityType& );
	statusCode				setConstraintContinuityType( AlConstraintContinuityType );

	statusCode				attachConstraint( AlSurface *, double, double );
	statusCode				attachConstraint( AlCurve *, double );
	statusCode				attachConstraint( AlCurveOnSurface *, double );

	statusCode				disassociateConstraint();
	
	statusCode				attachedTo( AlObject *&, double&, double& );
	
	API_BOOLEAN					isSlave();
	statusCode				makeMaster();

	AlBlendCurve*			blendCurve();

	AlBlendPoint*			next();
	statusCode				nextD( AlBlendPoint * );

	AlBlendPoint*			prev();
	statusCode				prevD( AlBlendPoint * );

private:

	static void initMessages();
	static void finiMessages();
};

#endif // _AlBlendPoint
