/********************************************************************
 * (C) Copyright 2006 by Autodesk, Inc. All Rights Reserved. By using
 * this code,  you  are  agreeing  to the terms and conditions of the
 * License  Agreement  included  in  the documentation for this code.
 * AUTODESK  MAKES  NO  WARRANTIES,  EXPRESS  OR  IMPLIED,  AS TO THE
 * CORRECTNESS OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE
 * IT.  AUTODESK PROVIDES THE CODE ON AN 'AS-IS' BASIS AND EXPLICITLY
 * DISCLAIMS  ANY  LIABILITY,  INCLUDING CONSEQUENTIAL AND INCIDENTAL
 * DAMAGES  FOR ERRORS, OMISSIONS, AND  OTHER  PROBLEMS IN THE  CODE.
 *
 * Use, duplication,  or disclosure by the U.S. Government is subject
 * to  restrictions  set forth  in FAR 52.227-19 (Commercial Computer
 * Software Restricted Rights) as well as DFAR 252.227-7013(c)(1)(ii)
 * (Rights  in Technical Data and Computer Software),  as applicable.
 *******************************************************************/
#include <stdio.h>
#include <AlCurve.h>
#include <AlCurveCV.h>
#include <AlCurveNode.h>
#include <AlCurveOnSurface.h>
#include <AlDagNode.h>
#include <AlFace.h>
#include <AlFaceNode.h>
#include <AlGroupNode.h>
#include <AlShellNode.h>
#include <AlShell.h>
#include <AlSurface.h>
#include <AlSurfaceCV.h>
#include <AlSurfaceNode.h>
#include <AlCamera.h>
#include <AlOrthographicCamera.h>
#include <AlPerspectiveCamera.h>
#include <AlCameraNode.h>
#include <AlCloud.h>
#include <AlCluster.h>
#include <AlClusterNode.h>
#include <AlClusterMember.h>
#include <AlPointLight.h>
#include <AlAmbientLight.h>
#include <AlDirectionLight.h>
#include <AlSpotLight.h>
#include <AlLinearLight.h>
#include <AlAreaLight.h>
#include <AlLightNode.h>
#include <AlLight.h>
#include <AlTrimRegion.h>
#include <AlTrimBoundary.h>
#include <AlTrimCurve.h>

#include <AlSet.h>
#include <AlSetMember.h>
#include <AlUniverse.h>

#include <AlChannel.h>
#include <AlAction.h>
#include <AlParamAction.h>
#include <AlMotionAction.h>
#include <AlKeyframe.h>

#include <AlLinkItem.h>
#include <AlMappedFieldItem.h>
#include <AlShadingFieldItem.h>
#include <AlShader.h>
#include <AlTexture.h>
#include <AlTextureNode.h>
#include <AlEnvironment.h>

#include <AlPolysetNode.h>
#include <AlPolyset.h>
#include <AlPolygon.h>
#include <AlPolysetVertex.h>

#include <AlMeshNode.h>
#include <AlMesh.h>

#include <AlAttributes.h>
#include <AlArcAttributes.h>
#include <AlLineAttributes.h>
#include <AlCurveAttributes.h>
#include <AlConicAttributes.h>
#include <AlPlaneAttributes.h>
#include <AlRevSurfAttributes.h>
#include <AlJoint.h>
#include <AlIKHandle.h>
#include <AlAimConstraint.h>
#include <AlPointConstraint.h>
#include <AlOrientationConstraint.h>
#include <AlBlendCurve.h>
#include <AlBlendPoint.h>

#include <AlLocator.h>
#include <AlPoint.h>

#include <AlCurveOnSurfacePoint.h>
#include <AlCurvePoint.h>
#include <AlSpacePoint.h>
#include <AlSurfacePoint.h>

#include <AlAngleLocator.h>
#include <AlAnnotationLocator.h>
#include <AlDeviationLocator.h>
#include <AlDistanceLocator.h>
#include <AlMinmaxLocator.h>
#include <AlRadialLocator.h>

#include <AlConstructionVector.h>
#include <AlConstructionPlane.h>

#include <AlLayer.h>

#include <AlLiveData.h>

#include <AlPrint.h>
#include <AlRender.h>
#include <AlRenderInfo.h>
#include <AlIterator.h>
#include <AlList.h>

#ifndef FALSE
#define FALSE 0
#define TRUE 1
#endif

typedef double double3[3];
typedef double double4[4];

static int		indentation = 0;
static char		indentationBuffer[256] = "";
static AlOutputType	print_outputType = kStdout;
static FILE *PRINTFILE = NULL;

void setPrintFile( AlOutputType ot )
{
	print_outputType = ot;
}

void setPrintFile( Xconst char *fileName )
{
	// First make sure the file does not exist.
	PRINTFILE = fopen( fileName, "r" );
	if (  PRINTFILE != NULL )
	{
		fprintf(stderr,"Error: setPrintFile() output file exists: %s.\n",fileName);
		fclose( PRINTFILE );
		PRINTFILE = NULL;
		return;
	}
	
	PRINTFILE = fopen( fileName, "w" );
}

void printFile( const char *fmt, ... )
{
	va_list ap;

	va_start( ap, fmt );
	if ( PRINTFILE != NULL )
		vfprintf( PRINTFILE, fmt, ap );
	else 
		AlVprintf( print_outputType, fmt, ap );
	va_end( ap );
}

void endPrintFile()
{
	if ( PRINTFILE != NULL )
		fclose( PRINTFILE );
}

int getIndentation( void )
{
	return indentation;
}

char* indentSpace( void )
{
	return indentationBuffer;
}

void setIndentation( int space )
{
	if( space < 0 )
	{
		if( indentation + space < 0 )
			indentation = 0;
		else
			indentation += space;
	}
	else
	{
		if( indentation + space > 255 )
			space = 255 - indentation;
		for( int i = 0; i < space; i++ )
			indentationBuffer[indentation+i] = ' ';
		indentation += space;
	}
	indentationBuffer[indentation] = '\0';
}

#define INDENT		setIndentation( 4 );
#define UNINDENT	setIndentation( -4 );

void printCVs( const double4 cvs[], int count )
{
	while( count --> 0 )
	{
		printFile( "%s%g %g %g %g\n", indentSpace(),
					cvs[0][0], cvs[0][1], cvs[0][2], cvs[0][3] );
		cvs++;
	}
}

void printCVs( const double3 cvs[], int count )
{
	while( count --> 0 )
	{
		printFile( "%s%g %g %g\n", indentSpace(),
					cvs[0][0], cvs[0][1], cvs[0][2] );
		cvs++;
	}
}

void printObjectName( const AlObject *object )
{
	if ( NULL == object ) return;

	const char*	className;
	if ( NULL != (className = determineObjectClassName( object->type() )) )
	{
		printFile( "%s%s(", indentSpace(), className );
		const char*	name = object->name();
		if ( NULL != name )
		{
			printFile( "%s", name );
		}
		printFile( ")\n" );
	}
}

void printChannelInfo( Xconst AlChannel *channel )
{
	AlObject	*item		= channel->animatedItem();
	const char	*field_name	= channel->parameterName();
	int			field;
	
	const char *objectClassName = "unknownClassName";
	if ( item != NULL )
		objectClassName = determineObjectClassName(item->type() );	

	printFile( "%schannel: (", indentSpace());
	printFile( "%s, ", objectClassName );
	field = channel->parameter();

	printFile( "%s [%d] )\n", field_name, field);
	delete item;
}

void printAlObject( Xconst AlObject *object, AlTM& tm )
{
	if ( NULL == object ) return;

	switch ( object->type() )
	{
	  case kCameraType:
	  case kPerspectiveCameraType:
	  case kOrthographicCameraType:
		printAlCamera( object->asCameraPtr(), tm );
		break;

	  case kCameraEyeType:
	  case kCameraViewType:
	  case kCameraUpType:
		printAlCameraNode( object->asCameraNodePtr(), tm );
		break;

	  case kClusterMemberType:
		break;

	  case kClusterType:
		break;

	  case kClusterNodeType:
		printAlClusterNode( object->asClusterNodePtr(), tm );
		break;

	  case kCurveType:
		printAlCurve( object->asCurvePtr(), tm );
		break;
	  case kCurveNodeType:
		printAlCurveNode( object->asCurveNodePtr(), tm );
		break;

	  case kCurveOnSurfaceType:
		printAlCurveOnSurface( object->asCurveOnSurfacePtr(), tm );
		break;

	  case kFaceType:
		printAlFace( object->asFacePtr(), tm );
		break;

	  case kFaceNodeType:
		printAlFaceNode( object->asFaceNodePtr(), tm );
		break;

	  case kDagNodeType:
		printAlDagNode( object->asDagNodePtr(), tm );
		break;

	  case kGroupNodeType:
		printAlGroupNode( object->asGroupNodePtr(), tm );
		break;

	  case kLightLookAtNodeType:
	  case kLightNodeType:
	  case kLightUpNodeType:
		printAlLightNode( object->asLightNodePtr(), tm );
		break;

	  case kLightType:
	  case kNonAmbientLightType:
		break;

	  case kAmbientLightType:
		printAlAmbientLight( object->asAmbientLightPtr(), tm );
		break;

	  case kPointLightType:
		printAlPointLight( object->asPointLightPtr(), tm );
		break;

	  case kDirectionLightType:
		printAlDirectionLight( object->asDirectionLightPtr(), tm );
		break;

	  case kSpotLightType:
		printAlSpotLight( object->asSpotLightPtr(), tm );
		break;

	  case kLinearLightType:
		printAlLinearLight( object->asLinearLightPtr(), tm );
		break;

	  case kAreaLightType:
		printAlAreaLight( object->asAreaLightPtr(), tm );
		break;

	  case kSetType:
		printAlSet( object->asSetPtr() );
		break;

	  case kSetMemberType:
		break;

	  case kActionType:
	  case kParamActionType:
	  case kMotionActionType:
		printAlAction(object->asActionPtr());
		break;

	  case kChannelType:
		printAlChannel(object->asChannelPtr());
		break;

	  case kShellNodeType:
		printAlShellNode( object->asShellNodePtr(), tm );
		break;

	  case kShellType:
		printAlShell( object->asShellPtr(), tm );
		break;

	  case kSurfaceNodeType:
		printAlSurfaceNode( object->asSurfaceNodePtr(), tm );
		break;

	  case kSurfaceType:
		{
			AlSurface* surface = object->asSurfacePtr();
			printAlSurface( surface, tm );
			if( surface->trimmed() )
				printAlSurfaceTrimRegions( surface, tm );
		}
		break;

	  case kPolysetNodeType:
		printAlPolysetNode( object->asPolysetNodePtr(), tm );
		break;

      case kMeshNodeType:
		printAlMeshNode( object->asMeshNodePtr(), tm );
		break;

	  case kTextureNodeType:
		printAlTextureNode( object->asTextureNodePtr(), tm );
		break;

	  default:
		break;
	}
}

void printAllAlObjects( Xconst AlObject *object, AlTM tm )
{
	if ( !object )
		return;
		
	AlObject *current = object->copyWrapper();
	
	while ( current != NULL )
	{
		// Print data about this node.
		//
		printAlObject( current, tm );

		// Recurse to the right or down.
		//
		if( NULL != current->asDagNodePtr() )
		{
			AlGroupNode*		groupNode = current->asGroupNodePtr();
			// Recurse down.
			//
			if ( NULL != groupNode )
			{
				INDENT
				AlTM	newtm = tm;
				AlObject *child = groupNode->childNode( newtm );
				printAllAlObjects( child, newtm );
				delete child;
				UNINDENT
			}

			// Recurse across.
			//
			AlObject *next = current->asDagNodePtr()->nextNode();
			delete current;
			current = next;
		}
	}
}

void printAttributes( Xconst AlAttributes* attributes )
{
	double	a, b, c, d, e, f;
	double	x, y, z;
	double	matrix[4][4];
	int		i;

	if ( attributes == NULL )
		return;

	printObjectName( attributes );

	INDENT

	switch ( attributes->type() )
	{
		case kAttributeType:
			break;

		case kArcAttributeType:
		{
			AlArcAttributes*	arcAttr = attributes->asArcAttributesPtr();
			if ( arcAttr == NULL )
				break;

			arcAttr->centerPoint(x, y, z);
			printFile( "%sCenter Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			arcAttr->startPoint(x, y, z);
			printFile( "%sStart Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			arcAttr->endPoint(x, y, z);
			printFile( "%sEnd Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			x = arcAttr->sweep();
			printFile( "%sSweep: %g\n", indentSpace(), x );

			x = arcAttr->radius();
			printFile( "%sRadius: %g\n", indentSpace(), x );

			break;
		}

		case kLineAttributeType:
		{
			AlLineAttributes*	lineAttr = attributes->asLineAttributesPtr();
			if ( lineAttr == NULL )
				break;

			lineAttr->startPoint(x, y, z);
			printFile( "%sStart Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			lineAttr->endPoint(x, y, z);
			printFile( "%sEnd Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			break;
		}

		case kCurveAttributeType:
		{
			double*	knots;
			double4	*cvs;
			int		numSpans;
			int		numKnots;
			int		numCvs;
			int*	multiplicity;

			AlCurveAttributes*	curve = attributes->asCurveAttributesPtr();
			if ( curve == NULL )
				break;

			//	Print out degree, form, number of spans, number of knots, etc.
			//	for this curve
			//
			printFile( "%sdegree = %d\n",
				indentSpace(), curve->degree() );

			printFile( "%sform   = %s\n", indentSpace(), determineFormTypeName(curve->form()) );

			numSpans = curve->numberOfSpans();
			numKnots = curve->numberOfKnots();
			numCvs   = curve->numberOfCVs();
			printFile( "%s# of Spans   = %d\n",
				indentSpace(), numSpans);
			printFile( "%s# of Knots   = %d\n",
				indentSpace(), numKnots);
			printFile( "%s# of CVs     = %d\n",
				indentSpace(), numCvs);

			if ( numKnots > 0 && numCvs > 0 )
			{
				knots = new double [ numKnots ];
				cvs   = new double4 [numCvs];
				multiplicity   = new int [ numCvs ];

				curve->knotVector( knots );
				curve->CVsUnaffectedPosition( cvs, multiplicity );

				printFile( "%sknots  = ", indentSpace() );
				for ( i = 0; i < numKnots; i++ )
				{
					printFile( " %g", knots[i] );
				}
				printFile( "\n" );

				printFile( "%smult  = ", indentSpace() );
				for ( i = 0; i < numCvs; i++ )
				{
					printFile( " %d", multiplicity[i] );
				}
				printFile( "\n" );

				printFile( "%sList of CV's =\n", indentSpace() );
				INDENT

				printCVs( cvs, numCvs );
			}
			UNINDENT
			break;
		}

		case kPlaneAttributeType:
		{
			AlPlaneAttributes*	planeAttr = attributes->asPlaneAttributesPtr();
			if ( planeAttr == NULL )
				break;

			planeAttr->coefficients( a, b, c, d );
			planeAttr->centerPoint(x, y, z);
			printFile( "%sCenter Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			printFile( "%sCoefficients: ( %g %g %g %g )\n",
				indentSpace(), a, b, c, d );

			break;
		}

		case kConicAttributeType:
		{
			AlConicAttributes*	conicAttr = attributes->asConicAttributesPtr();
			if ( conicAttr == NULL )
				break;

			conicAttr->coefficients( a, b, c, d, e, f );
			printFile( "%sCoefficients: ( %g %g %g %g %g %g )\n",
				indentSpace(), a, b, c, d, e, f );

			z = conicAttr->zDisplacement();
			printFile( "%sZ Displacement: %g\n", indentSpace(), z );

			conicAttr->startPoint( x, y );
			printFile( "%sStart Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			conicAttr->endPoint(x, y);
			printFile( "%sEnd Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			conicAttr->centerPoint(x, y);
			printFile( "%sCenter Point: ( %g, %g, %g )\n",
				indentSpace(), x, y, z );

			conicAttr->transform( matrix );
			printFile( "%sTransform is: \n", indentSpace());
			for( i  = 0; i < 4; i ++ )
			{
				printFile( "%s%g, %g, %g, %g\n", indentSpace(),
					matrix[i][0], matrix[i][1], matrix[i][2], matrix[i][3]);
			}

			printFile( "%sForm: ", indentSpace());
			switch ( conicAttr->form() )
			{
				case kNoConic:
					printFile( "Not a conic\n");
					break;
				case kEllipse:
					printFile( "Ellipse\n");
					break;
				case kParabola:
					printFile( "Parabola\n");
					break;
				case kHyperbola:
					printFile( "Hyperbola\n");
					break;
				case kCircle:
					printFile( "Circle\n");
					break;
				case kLine:
					printFile( "Line\n");
					break;
				default:
					printFile( "UNKNOWN\n");
					break;
			}
			break;
		}

		case kRevSurfAttributeType:
		{
			AlRevSurfAttributes* rsAttr = attributes->asRevSurfAttributesPtr();
			if ( rsAttr == NULL )
				break;

			x = rsAttr->startAngle();
			y = rsAttr->endAngle();
			printFile( "%sStart Angle: %g\n", indentSpace(), x);
			printFile( "%sEnd Angle: %g\n", indentSpace(), y);

			break;
		}

		default:
			break;
	}

	UNINDENT

	AlAttributes *next = attributes->nextAttribute();
	printAttributes( next );
	delete next;
}

void printAlCurve( Xconst AlCurve *curve, AlTM& tm  )
{
	if ( NULL == curve || NULL == curve->asCurvePtr() )
		return;

	int		 i;
	int		 numKnots;
	double	*knots;
	int		 numCvs;
	int		 numSpans;
	double4	*cvs;
	double4	*altcvs;
	int		*multiplicity;

	printObjectName( curve );

	INDENT

	printAttributes( curve->firstAttribute() );

	//	Print out degree, form, number of spans, number of knots, etc. for
	//	this curve
	//
	printFile( "%sdegree = %d\n", indentSpace(), curve->degree() );
	printFile( "%sform   = %s\n", indentSpace(), determineFormTypeName( curve->form()) );

	numSpans = curve->numberOfSpans();
	numKnots = curve->numberOfKnots();
	numCvs   = curve->numberOfCVs();
	printFile( "%s# of Spans   = %d\n", indentSpace(), numSpans);
	printFile( "%s# of Knots   = %d\n", indentSpace(), numKnots);
	printFile( "%s# of CVs     = %d\n", indentSpace(), numCvs);

	if ( numCvs > 0 && numKnots > 0 )
	{
		knots = new double [ numKnots ];
		cvs   = new double4 [numCvs];
		altcvs= new double4 [numCvs];
		multiplicity   = new int [ numCvs ];

		curve->knotVector( knots );
		curve->CVsWorldPosition( cvs, multiplicity );
		curve->CVsAffectedPosition( tm, altcvs, multiplicity );

		printFile( "%sknots  = ", indentSpace() );
		for ( i = 0; i < numKnots; i++ )
			printFile( " %g", knots[i] );

		printFile( "\n" );

		printFile( "%smult  = ", indentSpace() );
		for ( i = 0; i < numCvs; i++ )
			printFile( " %d", multiplicity[i] );

		printFile( "\n" );

		printFile( "%sList of CV's (CVsWorldPosition) =\n", indentSpace() );
		INDENT
		printCVs( cvs, numCvs );
		UNINDENT

		printFile( "%sList of CV's (CVsAffectedPosition) =\n", indentSpace() );
		INDENT
		printCVs( altcvs, numCvs );
		UNINDENT

		// Traverse the list of AlCurveCV's
		//
		printFile( "%sTraversing CV's Forward: \n", indentSpace() );

		AlCurveCV *CV = curve->firstCV();
		if( CV != NULL )
			{
				do {
					printAlCurveCV(CV, tm);
				} while( sSuccess == CV->nextD() );
				delete CV;
			}

		delete[] cvs;
		delete[] altcvs;
		delete knots;
		delete multiplicity;

#ifdef COPIOUS_OUTPUT
		// The following prints out a LOT of information. Add -DCOPIOUS_OUTPUT
		// to the compile line if you want to see it.

		// Print out world position and unaffected position including multiples
		//
		numCvs   = curve->numberOfCVsInclMultiples();
		cvs      = new double4 [ numCvs ];

		curve->CVsWorldPositionInclMultiples( cvs );
		printFile( "%s%d CV's including muliples =\n",
				indentSpace(), numCvs );
		INDENT
		printCVs( cvs, numCvs );
		UNINDENT
		delete cvs;

		cvs      = new double4 [numCvs];

		curve->CVsUnaffectedPositionInclMultiples( cvs );
		printFile( "%s%d unaffected CV's including muliples =\n",
				indentSpace(), numCvs );
		INDENT
		printCVs( cvs, numCvs );
		UNINDENT
		delete cvs;
#endif	// COPIOUS_OUTPUT
	}

	// If this curve is a face, print out the face's render information
	//
	AlRenderInfo renderInfo;
	if( (curve->asFacePtr() != NULL ) &&
		( curve->asFacePtr()->renderInfo( renderInfo ) == sSuccess ) )
	{
		printFile( "%sFace is %sdouble sided.\n", indentSpace(),
				(renderInfo.doubleSided ? "" : "NOT "));
		printFile( "%sFace is %sopposite.\n", indentSpace(),
				(renderInfo.opposite ? "" : "NOT "));
		printFile( "%sFace will %scast shadows.\n", indentSpace(),
				(renderInfo.castsShadow ? "" : "NOT "));
	}

	// If this curve is a face, print out its normal.
	//
	if( curve->asFacePtr() != NULL )
	{
		double norm[3];
		curve->asFacePtr()->normal(norm);
		printFile( "%sFace normal [ %g, %g, %g ].\n",
			indentSpace(), norm[0], norm[1], norm[2]);
	}

	UNINDENT

	// If this curve is a face, print out the next face
	//
	if (curve->asFacePtr() != NULL)
	{
		AlFace *nextFace = curve->asFacePtr()->nextFace();
		printAlFace( nextFace, tm );
		delete nextFace;
	}
}

void printAlCurveNode( Xconst AlCurveNode *curveNode, AlTM& tm )
{
	if ( NULL == curveNode ) return;

	printAlDagNode( curveNode, tm );

	INDENT
	AlTM	newtm = tm;

	AlCurve *curve = curveNode->curve(newtm);
	printAlCurve( curve, newtm );
	delete curve;
	UNINDENT
}

void printAlCurveCV( Xconst AlCurveCV *curveCV, AlTM& tm )
{
	double	x, y, z, w;
	double	ux, uy, uz, uw;
	double	ax, ay, az, aw;
	int		multi;

	printObjectName(curveCV);

	multi = curveCV->multiplicity();
	curveCV->worldPosition( x, y, z, w );
	curveCV->unaffectedPosition( ux, uy, uz, uw );
	curveCV->affectedPosition( tm, ax, ay, az, aw );

	INDENT

	printFile(
		"%s(%g %g %g %g) Multiplicity: %d\n", indentSpace(), x, y, z, w, multi);
	printFile(
		"%sUnaffected position: (%g %g %g %g)\n", indentSpace(),ux, uy, uz, uw);
	printFile(
		"%sAffected position: (%g %g %g %g)\n", indentSpace(),ax, ay, az, aw);

	// Print the clusters that this CV is in (if any)
	//
	AlCluster *cluster = curveCV->firstCluster();
	if (cluster != NULL)
	{
		printFile( "%sCLUSTERS this CV is in:\n", indentSpace());
		INDENT
		do {
			printFile( "%scluster: %s\n",
						indentSpace(), cluster->clusterNode()->name());
		} while (sSuccess == curveCV->nextClusterD(cluster));
		UNINDENT
		delete cluster;
	}

	// Print the animation on this CV (if any)
	//
	AlChannel *channel = curveCV->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this CV:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while (sSuccess == curveCV->nextChannelD(channel));
		UNINDENT
		delete channel;
	}
	UNINDENT
}

void printAlFace( Xconst AlFace *face, AlTM& tm )
{
	if( NULL == face ) return;
	printAlCurve( face, tm );
}

void printAlFaceNode( Xconst AlFaceNode *faceNode, AlTM& tm )
{
	if( NULL == faceNode ) return;
	printAlCurveNode( faceNode, tm );
}

void printAlDagNode( Xconst AlDagNode *dagNode, AlTM& tm )
{
	if ( NULL == dagNode ) return;

	printObjectName( dagNode );

	INDENT

	//  Print the display state of the dag Node
	//
	const char *name = dagNode->name();

	if( dagNode->isDisplayModeSet( kDisplayModeInvisible ) ) {
		printFile( "%sDag Node \"%s\" is invisible\n",
												indentSpace(), name );
	}
	if( dagNode->isDisplayModeSet( kDisplayModeBoundingBox ) ) {
		printFile( "%sDag Node \"%s\" is bounding boxed\n",
												indentSpace(), name);
	}
	if( dagNode->isDisplayModeSet( kDisplayModeTemplate ) ) {
		printFile( "%sDag Node \"%s\" is templated\n",
											indentSpace(), name );
	}

	// If the dag node is bounding boxed, print out the corners of the
	// bounding box
	//
	if( dagNode->isDisplayModeSet( kDisplayModeBoundingBox ) ) {
		double corners[8][4];
		dagNode->boundingBox( corners );
		printFile( "%sBounding box is:\n", indentSpace() );

		for ( int i = 0; i < 8; i++ ) {
					printFile( "%s   %g, %g, %g\n", indentSpace(),
											corners[i][0],
											corners[i][1],
											corners[i][2]);
		}
	}

	// Print the clusters that this dag node is in (if any)
	//
	AlCluster *cluster = dagNode->firstCluster();
	if (cluster != NULL)
	{
		printFile( "%sCLUSTERS this dag node is in:\n", indentSpace());
		INDENT
		do {
			AlClusterNode * cn = cluster->clusterNode();
			const char * clusterNodeName = cn->name();

			printFile( "%scluster: %s\n",
						indentSpace(), clusterNodeName);

			delete cn;
		} while ( sSuccess == dagNode->nextClusterD(cluster));
		UNINDENT
		delete cluster;
	}

	// Print the animation on this dag node (if any)
	//
	AlChannel *channel = dagNode->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this dag node:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while (sSuccess == dagNode->nextChannelD(channel));
		UNINDENT
		delete channel;
	}

	// Print the local and global transformation matrices
	//
	double matrix[4][4];
	int    i;
	dagNode->localTransformationMatrix( matrix );
	printFile( "%sLocal Transform is: \n", indentSpace());
	for( i  = 0; i < 4; i ++ ) {
		printFile( "%s%g, %g, %g, %g\n", indentSpace(),
						matrix[i][0], matrix[i][1], matrix[i][2], matrix[i][3]);
	}
	dagNode->globalTransformationMatrix( matrix );
	printFile( "%sGlobal Transform is: \n", indentSpace());
	for( i  = 0; i < 4; i ++ ) {
		printFile( "%s%g, %g, %g, %g\n", indentSpace(),
						matrix[i][0], matrix[i][1], matrix[i][2], matrix[i][3]);
	}
	dagNode->affectedTransformationMatrix( tm, matrix );
	printFile( "%sAffected Transform is: \n", indentSpace());
	for( i  = 0; i < 4; i ++ ) {
		printFile( "%s%g, %g, %g, %g\n", indentSpace(),
						matrix[i][0], matrix[i][1], matrix[i][2], matrix[i][3]);
	}

	AlJoint*	joint= dagNode->joint();
	if ( joint== NULL)
		printFile( "%sDag node is not a joint.\n", indentSpace());
	else
	{
		printAlJoint(joint, tm);
		delete joint;
	}

	AlConstraint* con = dagNode->firstConstraint();
	AlConstraint* old;
	if( con ) {
		while( con ) {
			printAlConstraint( con );
			old = con;
			con = con->next();
			delete old;
		}
	} else {
		printFile( "%sDag node is not constrained.\n", indentSpace() );
	}

	UNINDENT
}

void printAlJoint( Xconst AlJoint *joint, AlTM& tm )
{
	if ( NULL == joint )
		return;

	printObjectName( joint );

	INDENT

	boolean	transbool[3], rotbool[3];
	joint->useTransforms(transbool, rotbool);
	printFile( "%s%sse X translation.\n", indentSpace(),
		(transbool[0] ? "U" : "Don't u"));
	printFile( "%s%sse Y translation.\n", indentSpace(),
		(transbool[1] ? "U" : "Don't u"));
	printFile( "%s%sse Z translation.\n", indentSpace(),
		(transbool[2] ? "U" : "Don't u"));
	printFile( "%s%sse X rotation.\n", indentSpace(),
		(rotbool[0] ? "U" : "Don't u"));
	printFile( "%s%sse Y rotation.\n", indentSpace(),
		(rotbool[1] ? "U" : "Don't u"));
	printFile( "%s%sse Z rotation.\n", indentSpace(),
		(rotbool[2] ? "U" : "Don't u"));

	joint->useLimits(transbool, rotbool);
	printFile( "%s%sse X translation limit.\n", indentSpace(),
		(transbool[0] ? "U" : "Don't u"));
	printFile( "%s%sse Y translation limit.\n", indentSpace(),
		(transbool[1] ? "U" : "Don't u"));
	printFile( "%s%sse Z translation limit.\n", indentSpace(),
		(transbool[2] ? "U" : "Don't u"));
	printFile( "%s%sse X rotation limit.\n", indentSpace(),
		(rotbool[0] ? "U" : "Don't u"));
	printFile( "%s%sse Y rotation limit.\n", indentSpace(),
		(rotbool[1] ? "U" : "Don't u"));
	printFile( "%s%sse Z rotation limit.\n", indentSpace(),
		(rotbool[2] ? "U" : "Don't u"));

	double mins[3], maxs[3];
	int stiffness[3];
	joint->rotation(mins, maxs, stiffness);
	printFile( "%sX rotation min = %g, max = %g, stiffness = %d\n",
		indentSpace(), mins[0], maxs[0], stiffness[0]);
	printFile( "%sY rotation min = %g, max = %g, stiffness = %d\n",
		indentSpace(), mins[1], maxs[1], stiffness[1]);
	printFile( "%sZ rotation min = %g, max = %g, stiffness = %d\n",
		indentSpace(), mins[2], maxs[2], stiffness[2]);

	joint->translation(mins, maxs, stiffness);
	printFile( "%sX translation min = %g, max = %g, stiffness = %d\n",
		indentSpace(), mins[0], maxs[0], stiffness[0]);
	printFile( "%sY translation min = %g, max = %g, stiffness = %d\n",
		indentSpace(), mins[1], maxs[1], stiffness[1]);
	printFile( "%sZ translation min = %g, max = %g, stiffness = %d\n",
		indentSpace(), mins[2], maxs[2], stiffness[2]);

	printFile( "%sJoint %s a character joint\n", indentSpace(),
		(joint->charJoint() ? "IS" : "is NOT"));
	if (joint->charJoint())
	{
		INDENT

		double upper, lower;
		AlJointInterpolationType type;
		joint->charJointLimits(upper, lower, type);
		printFile( "%sBounds: upper = %g, lower = %g\n", indentSpace(),
							upper, lower);
		printFile( "%sInterpolation type: ", indentSpace());
		switch(type)
		{
			case kJointInterpLinear:
				printFile( "LINEAR\n");
				break;
			case kJointInterpExponential:
				printFile( "EXPONENTIAL\n");
				break;
			case kJointInterpSine:
				printFile( "SINE\n");
				break;
			default:
				printFile( "Unknown\n");
				break;
		}

		printFile( "%sCharacter joint bulging: %s\n", indentSpace(),
							(joint->useBulge() ? "ON" : "OFF"));

		int code, orient, around, along;
		double scale, range;
		AlBulgeSections sections;
		AlDagNodeFields parameter;
		AlBulgeRelateTo which_node;
		char node[100];
		joint->bulgeAttributes(&code, &orient, &scale, &sections,
								   &around, &along, &parameter, &which_node,
								   node, &range);
		printFile( "%sBulge code: %d\n", indentSpace(), code);
		printFile( "%sOrientation axis: %c\n", indentSpace(),
					(orient == 0 ? 'X' : ((orient == 1) ? 'Y' :
					(orient == 2 ? 'Z' : '?'))));
		printFile( "%sBulge scale: %g\n", indentSpace(), scale);
		printFile( "%sBulge sections: %s\n", indentSpace(),
				(sections == kBulgeDefinition ? "DEFINITION" :
				(sections == kBulgeEvenlySpaced ? "EVENLY SPACED" : "?")));
		printFile( "%sSections, around = %d, along = %d\n",
									indentSpace(), around, along);
		printFile( "%sRelate to parameter: %s\n", indentSpace(),
			(parameter == kFLD_DAGNODE_XTRANSLATE ? "X_Translate" :
			(parameter == kFLD_DAGNODE_YTRANSLATE ? "Y_Translate" :
			(parameter == kFLD_DAGNODE_ZTRANSLATE ? "Z_Translate" :
			(parameter == kFLD_DAGNODE_XROTATE ? "X_Rotate" :
			(parameter == kFLD_DAGNODE_YROTATE ? "Y_Rotate" :
			(parameter == kFLD_DAGNODE_ZROTATE ? "Z_Rotate" :
			(parameter == kFLD_DAGNODE_XSCALE ? "X_Scale" :
			(parameter == kFLD_DAGNODE_YSCALE ? "Y_Scale" :
			(parameter == kFLD_DAGNODE_ZSCALE ? "Z_Scale" :
			(parameter == kFLD_DAGNODE_VISIBILITY ? "Visibility" :
			"Unknown")))))))))));
		printFile( "%sRelate to %s", indentSpace(),
				(which_node == kBulgeLowerJoint ? "LOWER JOINT" :
				(which_node == kBulgeOtherNode ? "OTHER NODE: " : "Unknown")));
		printFile( "%s\n",
				(which_node == kBulgeOtherNode ? node : ""));
		printFile( "%sParameter bulge range: %g\n", indentSpace(),
																	range);

		UNINDENT
	}

	// Deal with the handles.
	AlIKHandle * ikh = joint->endEffectorHandle();
	if( ikh ) {
		printAlIKHandle( ikh, tm );
		delete ikh;
	} else {
		printFile( "%sJoint is not an end effector.\n", indentSpace() );
	}

	UNINDENT
}

void printAlIKHandle( Xconst AlIKHandle* ikh, AlTM& )
{
	if( NULL == ikh ) {
		return;
	}
	printObjectName( ikh );
	INDENT

	// Print ik handle solver type
	//
	AlIKHandleSolverType st = ikh->solverType();
	printFile( "%sSolver type: %s.\n", indentSpace(),
								(st == kSingleChain ? "kSingleChain" :
								 st == kMultiSolver ? "kMultiSolver" :
								 st == kSplineSolver ? "kSplineSolver" :
								 "unknow"));

	// Print ik handle root and end joint nodes
	//
	AlDagNode* n = ikh->rootNode();
	printFile("%sRoot Joint: %s.\n", indentSpace(), n ? n->name():"NULL");
	if( n ) delete n;

	n = ikh->endEffectorNode();
	printFile("%sEnd joint: %s.\n", indentSpace(), n? n->name() : "NULL");
	if( n ) delete n;

	// Print ik handle goal type (only effects if it is not a spline handle)
	//
	if (st != kSplineSolver) {
		AlIKHandleGoalType	gt = ikh->goalType();
		printFile( "%sGoal type: %s.\n", indentSpace(),
							(gt == kPositionGoal ? "kPositionGoal" :
							 gt == kOrientationGoal ? "kOrientationGoal" :
							 gt == kBothGoal ? "kBothGoal" :
							 "unknow"));
	}

	// Print weight for multi handle
	//
	if (st == kMultiSolver)	{
		printFile( "%sWeight: %g.\n", indentSpace(), ikh->weight() );
	}

	// Print information only used by single-chain solver
	//
	if (st == kSingleChain) {
		printFile( "%sOrientaion Control: %s.\n", indentSpace(),
							ikh->worldOrientation() ? "world":"local");

		AlIKHandleRotationOrder	ro	= ikh->rotationOrder();
		printFile( "%sRotation Order: %s.\n", indentSpace(),
							(ro == kRotationOrder_XYZ ? "kRotationOrder_XYZ" :
							 ro == kRotationOrder_YXZ ? "kRotationOrder_YXZ" :
							 ro == kRotationOrder_ZXY ? "kRotationOrder_ZXY" :
							 "unknow"));

		double	restRot[3];
		ikh->restRotation(restRot);
		printFile( "%sRest Rotation: <%g, %g, %g>.\n", indentSpace(),
									restRot[0], restRot[1], restRot[2]);
	}

	// Print information only used by spline handle
	//
	if (st == kSplineSolver) {
		AlIKHandlePositionType	pt = ikh->positionType();
		printFile( "%sSpline Position Type: %s\n", indentSpace(),
							(pt == kParameter ? "kParameter" :
							 pt == kArcLength ? "kArcLength" :
							 "unknow"));

		AlIKHandleTwistType		tt = ikh->twistType();
		printFile( "%sSpline Twist Type: %s\n", indentSpace(),
							(tt == kTwist_Linear ? "kTwist_Linear" :
							 tt == kTwist_FOSI ? "kTwist_FOSI" :
							 tt == kTwist_SOFI ? "kTwist_SOFI" :
							 tt == kTwist_EOEI ? "kTwist_EOEI" :
							 "unknow"));

		AlCurveNode* crv = ikh->curveNode();
		printFile("%sSpline Curve: %s.\n", indentSpace(),
									(crv? crv->name() : "NULL"));
		if (crv) delete crv;

		printFile("%sIs One joine: %s.\n", indentSpace(),
									(ikh->oneJointHandle() ? "yes" : "no"));
	}

	UNINDENT
}

void printAlConstraint(Xconst AlConstraint *constraint)
{
	if ( NULL == constraint )
		return;

	printObjectName( constraint );

	INDENT

	printFile( "%son = %s\n", indentSpace(),
		(constraint->on()) ? "TRUE" : "FALSE");
	printFile( "%sweight = %g\n", indentSpace(), constraint->weight());

	AlDagNode*	dagNode = constraint->sourceNode();
	if ( NULL != dagNode )
	{
		printFile( "%sSource node = %s\n", indentSpace(),
			dagNode->name());
		delete dagNode;
	}

	dagNode = constraint->destinationNode();
	if ( NULL != dagNode )
	{
		printFile( "%sDestination node = %s\n", indentSpace(),
			dagNode->name());
		delete dagNode;
	}

	if ( constraint->asAimConstraintPtr() != NULL )
	{
		AlAimConstraint*	aim = constraint->asAimConstraintPtr();
		char*				axisText;
		switch ( aim->aimAxis() )
		{
			case kAimConstraint_X:
				axisText = "X";
				break;
			case kAimConstraint_Y:
				axisText = "Y";
				break;
			case kAimConstraint_Z:
				axisText = "Z";
				break;
			default:
				axisText = "UNKNOWN";
				break;
		}
		printFile( "%sAim axis = %s\n", indentSpace(), axisText);
		switch ( aim->upAxis() )
		{
			case kAimConstraint_X:
				axisText = "X";
				break;
			case kAimConstraint_Y:
				axisText = "Y";
				break;
			case kAimConstraint_Z:
				axisText = "Z";
				break;
			default:
				axisText = "UNKNOWN";
				break;
		}
		printFile( "%sUp axis = %s\n", indentSpace(), axisText);
	}
	else if ( constraint->asPointConstraintPtr() != NULL )
	{
		AlPointConstraint*	point = constraint->asPointConstraintPtr();
		double d[3];
		point->point( d );
		printFile( "%spoint = %g %g %g.\n", indentSpace(), d[0], d[1], d[2] );
	}
	else
	{
		AlOrientationConstraint*	orient = constraint->asOrientationConstraintPtr();
		double d[3];
		orient->goalX( d );
		printFile( "%sX goal = %g %g %g.\n", indentSpace(), d[0], d[1], d[2] );
		orient->goalY( d );
		printFile( "%sY goal = %g %g %g.\n", indentSpace(), d[0], d[1], d[2] );
		orient->goalZ( d );
		printFile( "%sZ goal = %g %g %g.\n", indentSpace(), d[0], d[1], d[2] );
	}

	UNINDENT
}

void printAlGroupNode( Xconst AlGroupNode *groupNode, AlTM& tm )
{
	if ( NULL == groupNode ) return;
	printAlDagNode( groupNode, tm );
}


void printAlShellNode( Xconst AlShellNode *shellNode, AlTM& tm )
{
	if ( NULL == shellNode ) return;

	printAlDagNode( shellNode, tm );

	INDENT
	AlTM newtm = tm;
	AlShell *shell = shellNode->shell(newtm);
	printAlShell( shell, newtm );
	delete shell;
	UNINDENT
}

void printAlShell( Xconst AlShell *shell, AlTM& tm )
{
	if( shell == (AlShell*) 0 )
		return;

	printObjectName( shell );
	INDENT

	// Print out the render info on this shell
	//
	AlRenderInfo renderInfo;
	if( sSuccess == shell->renderInfo( renderInfo ) )
	{
		printFile( "%sshell is %sDOUBLE SIDED.\n", indentSpace(),
			(renderInfo.doubleSided ? "" : "not "));
		printFile( "%sshell is %sopposite.\n", indentSpace(),
			(renderInfo.opposite ? "" : "not "));
		printFile( "%sshell will %scast shadows.\n", indentSpace(),
			(renderInfo.castsShadow ? "" : "not "));
	}

	AlShader*	shader = shell->firstShader();
	if ( shader != (AlShader*)NULL )
	{
		printFile( "%sShaders on this shell:\n", indentSpace());
		INDENT
		do {
			const char *shaderName = shader->name();
			printFile( "%s%s\n", indentSpace(), shaderName );
		} while( sSuccess == shell->nextShaderD( shader ) );
		UNINDENT
		delete shader;
	}

	AlTrimRegion* memberSurface = shell->firstTrimRegion();
	if ( memberSurface != (AlTrimRegion*) 0 )
	{
		printFile( "%sSurfaces that are members of this shell:\n",
			indentSpace());
		INDENT
		do {
			printAlTrimRegion( memberSurface, tm );
		} while( sSuccess == memberSurface->nextRegionD() );
		delete memberSurface;
		UNINDENT
	}
	UNINDENT
}

void printAlSurface( Xconst AlSurface *surface, AlTM& tm )
{
	if ( NULL == surface ) return;

	int		 i;
	int		 numUKnots;
	int		 numVKnots;
	double	*uKnots;
	double	*vKnots;
	int		 numUCvs;
	int		 numVCvs;
	int		 numUSpans;
	int		 numVSpans;
	double4	*cvs;
	double4	*altcvs;
	int		*uMult;
	int		*vMult;

	printObjectName( surface );
	INDENT

	AlAttributes *attribute = surface->firstAttribute();
	printAttributes( attribute );
	delete attribute;

	// Print out the render info on this surface
	//
	AlRenderInfo renderInfo;
	if( sSuccess == surface->renderInfo( renderInfo ) )
	{
		printFile( "%sSurface is %sDOUBLE SIDED.\n", indentSpace(),
				(renderInfo.doubleSided ? "" : "not "));
		printFile( "%sSurface is %sopposite.\n", indentSpace(),
				(renderInfo.opposite ? "" : "not "));
		printFile( "%sSurface will %scast shadows.\n", indentSpace(),
				(renderInfo.castsShadow ? "" : "not "));
	}
	printFile( "%sSurface is %saffected by viewframe.\n",
		indentSpace(), (surface->isConstructionHistoryResultingSurface()? "" : "not "));

	AlShader*	shader = surface->firstShader();
	if ( shader != NULL )
	{
		printFile( "%sShaders on this surface:\n", indentSpace());
		INDENT
		do {
			const char *shaderName = shader->name();
			printFile( "%s%s\n", indentSpace(), shaderName );
		} while( sSuccess == surface->nextShaderD( shader ) );
		UNINDENT
		delete shader;
	}

	printFile( "%sSurface is %strimmed.\n", indentSpace(),
		(surface->trimmed() ? "" : "not "));

	printFile( "%sU degree = %d\n", indentSpace(), surface->uDegree());
	printFile( "%sV degree = %d\n", indentSpace(), surface->vDegree());
	printFile( "%sU form   = %s\n", indentSpace(), determineFormTypeName(surface->uForm() ));
	printFile( "%sV form   = %s\n", indentSpace(), determineFormTypeName( surface->vForm()) );

	numUSpans = surface->uNumberOfSpans();
	numVSpans = surface->vNumberOfSpans();
	numUKnots = surface->realuNumberOfKnots();
	numVKnots = surface->realvNumberOfKnots();
	numUCvs   = surface->uNumberOfCVs();
	numVCvs   = surface->vNumberOfCVs();

	printFile( "%s# of Spans U: %d V: %d\n",
						indentSpace(), numUSpans, numVSpans);
	printFile( "%s# of Knots U: %d V: %d\n",
						indentSpace(), numUKnots, numVKnots);
	printFile( "%s# of CVs   U: %d V: %d\n",
						indentSpace(), numUCvs, numVCvs);

	if ( numUKnots > 0 && numVKnots > 0 && numUCvs > 0 && numVCvs > 0 )
	{
		// Print out the knot vectors
		//
		uKnots = new double [ numUKnots ];
		vKnots = new double [ numVKnots ];
		surface->realuKnotVector( uKnots );
		surface->realvKnotVector( vKnots );

		printFile( "%sU knots  =", indentSpace() );
		for ( i = 0; i < numUKnots; i++ )
		{
			printFile( " %g", uKnots[i] );

		}
		printFile( "\n" );

		printFile( "%sV knots  =", indentSpace() );
		for ( i = 0; i < numVKnots; i++ )
		{
			printFile( " %g", vKnots[i] );
		}
		printFile( "\n" );

		delete uKnots;
		delete vKnots;

		// Print out the multiplicities
		//
		cvs   = new double4 [ numUCvs * numVCvs];
		altcvs= new double4 [ numUCvs * numVCvs];
		uMult   = new int [ numUCvs ];
		vMult   = new int [ numVCvs ];
		surface->CVsWorldPosition( &cvs[0][0], uMult, vMult );
		surface->CVsAffectedPosition( tm, &altcvs[0][0], uMult, vMult );

		printFile( "%sU multiplicities  =", indentSpace() );
		for ( i = 0; i < numUCvs; i++ )
		{
			printFile( " %d", uMult[i] );
		}
		printFile( "\n" );
		printFile( "%sV multiplicities  =", indentSpace() );
		for ( i = 0; i < numVCvs; i++ )
		{
			printFile( " %d", vMult[i] );
		}
		printFile( "\n" );

		delete uMult;
		delete vMult;

		// Print out the CV's
		//

		printFile( "%sList of CV's (CVsWorldPosition) =\n", indentSpace() );
		INDENT
		printCVs( cvs, numUCvs * numVCvs);
		UNINDENT
		delete[] cvs;

		printFile( "%sList of CV's (CVsAffectedPosition) =\n", indentSpace() );
		INDENT
		printCVs( altcvs, numUCvs * numVCvs);
		UNINDENT
		delete[] altcvs;

		// Traverse the list of AlSurfaceCV's
		//
		printFile( "%sTraversing CV's Forward: \n", indentSpace() );

		AlSurfaceCV *surfCVV = surface->firstCV();
		if( surfCVV != NULL )
			{
			do
				{
				AlObject *copy = surfCVV->copyWrapper();
				AlSurfaceCV *surfCVU = copy->asSurfaceCVPtr();
				if( surfCVU != NULL )
					{
					do
						{
						printAlSurfaceCV( surfCVU, tm );
						} while( sSuccess == surfCVU->nextInUD());
					delete surfCVU;
					}
				} while( sSuccess == surfCVV->nextInVD());
			delete surfCVV;
			}

#ifdef COPIOUS_OUTPUT
		// The following prints out a LOT of information. Add -DCOPIOUS_OUTPUT
		// to the compile line if you want to see it.

		// Print out world position of all CVs including multiples
		//
		numUCvs   = surface->uNumberOfCVsInclMultiples();
		numVCvs   = surface->vNumberOfCVsInclMultiples();
		cvs     = new double4 [ numUCvs * numVCvs ];

		surface->CVsWorldPositionInclMultiples( &cvs[0][0] );
		printFile( "%s List of %d CV's including multiples =\n",
				indentSpace(), numUCvs * numVCvs );

		// Print out the cvs matrix
		//
		INDENT
		printCVs( cvs, numUCvs * numVCvs);
		UNINDENT
		delete cvs;

		// Print out unaffected positions of all CVs including multiplies
		//
		cvs     = new double4 [ numUCvs * numVCvs ];
		surface->CVsUnaffectedPositionInclMultiples( &cvs[0][0] );
		printFile( "%sList of %d unaffected CV's including multiples =\n",
				indentSpace(), numUCvs * numVCvs );

		INDENT
		printCVs( cvs, numUCvs * numVCvs);
		UNINDENT
		delete cvs;
#endif	// COPIOUS_OUTPUT
	}


	printAlCurvesOnSurface( surface, tm );

	if( surface->trimmed() )
		printAlSurfaceTrimRegions( surface, tm );

	UNINDENT
}

void printAlCurvesOnSurface( Xconst AlSurface *surface, AlTM& tm )
{
	if( NULL == surface ) return;
	AlCurveOnSurface *cos = surface->firstCurveOnSurface();
	if( cos != NULL )
	{
		do {
			printAlCurveOnSurface( cos, tm );
		} while( sSuccess == cos->nextCurveOnSurfaceD() );
		delete cos;
	}
}

void printAlCurveOnSurface( Xconst AlCurveOnSurface *cos, AlTM& )
{
	if( NULL == cos ) return;

	double  pt[4];
	double  knot;
	int     i;

	printObjectName( cos );
	INDENT

	printFile( "%sdegree = %d\n", indentSpace(), cos->degree());
	printFile( "%sform   = %s\n", indentSpace(), determineFormTypeName(cos->form()) );
	printFile( "%snumberOfSpans = %d\n", indentSpace(),
							cos->numberOfSpans());
	printFile( "%snumberOfKnots = %d\n", indentSpace(),
							cos->numberOfKnots());
	printFile( "%snumberOfCVs = %d\n", indentSpace(),
							cos->numberOfControlPoints());

	printFile( "%sKnots = \n%s", indentSpace(), indentSpace() );
	for( i = 0; i < cos->numberOfKnots(); i ++ ) {
		knot = cos->knotValue( i );
		printFile( " %g ", knot );
	}
	printFile( "\n");

	printFile( "%sControl Points =\n", indentSpace());
	for( i = 0; i < cos->numberOfControlPoints(); i ++ ) {
		cos->controlPoint( i, pt );
		printFile( "%s %g, %g, %g, %g\n", indentSpace(),
								pt[0], pt[1], pt[2], pt[3]);
	}
	UNINDENT
}

void printAlSurfaceNode( Xconst AlSurfaceNode *surfaceNode, AlTM& tm )
{
	if ( NULL == surfaceNode ) return;

	printAlDagNode( surfaceNode, tm );

	INDENT
	printFile( "%sPatch precision = %d\n",
				indentSpace(), surfaceNode->patchPrecision() );

	AlTM	newtm = tm;
	AlSurface *surface = surfaceNode->surface( newtm );
	printAlSurface( surface, newtm );
	delete surface;

	UNINDENT
}

void printAlSurfaceCV( Xconst AlSurfaceCV *surfaceCV, AlTM& tm )
{
	double	x, y, z, w;
	double	ux, uy, uz, uw;
	double	ax, ay, az, aw;
	int		multiU, multiV;

	printObjectName(surfaceCV);

	multiU = surfaceCV->multiplicityInU();
	multiV = surfaceCV->multiplicityInV();
	surfaceCV->worldPosition( x, y, z, w );
	surfaceCV->unaffectedPosition( ux, uy, uz, uw );
	surfaceCV->affectedPosition( tm, ax, ay, az, aw );

	INDENT

	printFile(
		"%s(%g %g %g %g) Multiplicity: %d %d\n",
									indentSpace(), x, y, z, w, multiU, multiV);
	printFile(
		"%sUnaffected position: (%g %g %g %g)\n", indentSpace(),ux, uy, uz, uw);
	printFile(
		"%sAffected position: (%g %g %g %g)\n", indentSpace(),ax, ay, az, aw);

	// Print the clusters that this CV is in (if any)
	//
	AlCluster *cluster = surfaceCV->firstCluster();
	if (cluster != NULL)
	{
		printFile( "%sCLUSTERS this CV is in:\n", indentSpace());
		INDENT
		do {
			printFile( "%scluster: %s\n",
						indentSpace(), cluster->clusterNode()->name());
		} while ( sSuccess == surfaceCV->nextClusterD(cluster));
		UNINDENT
		delete cluster;
	}

	// Print the animation on this CV (if any)
	//
	AlChannel *channel = surfaceCV->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this CV:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while ( sSuccess == surfaceCV->nextChannelD(channel));
		UNINDENT
		delete channel;
	}

	UNINDENT
}

void printAlSurfaceTrimRegions ( Xconst AlSurface *surface, AlTM& tm )
{
	if( NULL == surface ) return;

	AlTrimRegion*	region = surface->firstTrimRegion();
	if( NULL != region )
	{
		do {
			printAlTrimRegion( region, tm );
		} while( sSuccess == region->nextRegionD());
		delete region;
	}
}

void printAlTrimRegion ( Xconst AlTrimRegion *region, AlTM& tm )
{
	if( NULL == region ) return;
	printFile( "%sAlTrimRegion\n", indentSpace() );

	AlTrimBoundary*	boundary = region->firstBoundary();
	if( boundary == (AlTrimBoundary*)NULL )
		return;

	INDENT
	printFile( "%s--Outer Boundary--\n", indentSpace() );
	printAlTrimBoundary( boundary, tm );

	// having printed the first one we advance to the next...
	while( boundary->nextBoundaryD() == sSuccess )
	{
		printFile( "%s--Inner Boundaries--\n", indentSpace() );
		printAlTrimBoundary( boundary, tm );
	}

	delete boundary;
	UNINDENT
}

void printAlTrimBoundary ( Xconst AlTrimBoundary *boundary, AlTM& tm )
{
	if( NULL == boundary ) return;

	printFile( "%sAlTrimBoundary\n", indentSpace() );
	INDENT

	AlTrimCurve*	curve = boundary->firstCurve();
	if( curve != (AlTrimCurve*)NULL )
	{
		do {
			printAlTrimCurve( curve, tm );
		} while( sSuccess == curve->nextCurveD());
		delete curve;
	}
	UNINDENT
}

void printAlTrimCurve ( Xconst AlTrimCurve *curve, AlTM& )
{
	if ( NULL == curve ) return;

	int		 i;
	int		 numKnots;
	int		 numCvs;
	int		 numSpans;
	double	*knots;
	double3	*cvs;

	printFile( "%sAlTrimCurve\n", indentSpace() );

	INDENT

	printFile( "%sdegree = %d\n", indentSpace(), curve->degree() );
	printFile( "%sform   = %s\n", indentSpace(), determineFormTypeName(curve->form()) );

	numKnots = curve->numberOfKnots();
	numCvs   = curve->numberOfCVs();
	numSpans = curve->numberOfSpans();
	printFile( "%s# of Spans   = %d\n", indentSpace(), numSpans);
	printFile( "%s# of Knots   = %d\n", indentSpace(), numKnots);
	printFile( "%s# of CVs     = %d\n", indentSpace(), numCvs);

	if ( numKnots > 0 && numCvs > 0 )
	{
		knots = new double [ numKnots ];
		cvs   = new double3[numCvs];

		curve->CVsUVPosition( knots, cvs );
		printFile( "%sknots  = ", indentSpace() );
		for ( i = 0; i < numKnots; i++ )
		{
			printFile( " %g", knots[i] );
		}
		printFile( "\n" );

		printFile( "%sList of CV's =\n", indentSpace() );
		INDENT
		printCVs( cvs, numCvs );	// 3 version
		UNINDENT

		UNINDENT

		delete[] cvs;
		delete knots;
	}
}

void printAlCamera( Xconst AlCamera *camera, AlTM& )
{
	if( !AlIsValid( camera ) ) return;

	AlPerspectiveCamera * pcam = camera->asPerspectiveCameraPtr();
	if( pcam ) {

		double ncp,fcp;
		printObjectName( pcam );
		INDENT
		printFile( "%sAngleOfView = %g\n",
				indentSpace(), pcam->angleOfView() );
		printFile( "%stwist      = %g\n",
				indentSpace(), pcam->twistAngle() );
		pcam->nearClippingPlane( ncp );
		printFile( "%snearClip   = %g\n",
				indentSpace(), ncp );
		pcam->farClippingPlane( fcp );
		printFile( "%sfarClip    = %g\n",
				indentSpace(), fcp );

		double x,y,z;
		pcam->worldEye( x,y,z);
		printFile( "%sWorld Eye  = %g %g %g\n", indentSpace(), x, y, z );
		pcam->worldView( x,y,z);
		printFile( "%sWorld View = %g %g %g\n", indentSpace(), x, y, z );
		pcam->worldUp( x,y,z);
		printFile( "%sWorld Up   = %g %g %g\n", indentSpace(), x, y, z );

		// Print the animation on this pcam (if any)
		//
		AlChannel *channel = pcam->firstChannel();
		if (channel != NULL)
		{
			printFile( "%sANIMATION on this pcam:\n", indentSpace());
			INDENT
			do {
				printChannelInfo( channel );
			} while ( sSuccess == pcam->nextChannelD(channel));
			UNINDENT
			delete channel;
		}
		UNINDENT
	} else {
		AlOrthographicCamera *ocam = camera->asOrthographicCameraPtr();
		if( !ocam ) return;
		double ncp,fcp;

		printObjectName( ocam );
		INDENT
		ocam->nearClippingPlane( ncp );
		printFile( "%snearClip   = %g\n",
				indentSpace(), ncp );
		ocam->farClippingPlane( fcp );
		printFile( "%sfarClip    = %g\n",
				indentSpace(), fcp );
		UNINDENT
	}
}

void printAlCameraNode( Xconst AlCameraNode *cameraNode, AlTM& tm )
{
	if( NULL == cameraNode ) return;

	switch( cameraNode->type())
	{
	case kCameraEyeType:
	{
		printAlDagNode( cameraNode, tm );
		INDENT
		AlTM	newtm = tm;
		AlCamera *camera = cameraNode->camera( newtm );
		printAlCamera( camera, newtm );
		delete camera;

		UNINDENT
		break;
	}

	  case kCameraUpType:
	  case kCameraViewType:
		printAlDagNode( cameraNode, tm);
		break;

	  default:
		break;
	}
}

void printAlLightGlow( const AlLight *light )
{
	if ( light == NULL )
		return;

	if ( light->type() == kAmbientLightType )
	{
		// Ambient light's do not support glow.
		return;
	}

	double value;
	char *typestr = NULL;

	light->parameter( kFLD_LIGHT_GLOW_TYPE, value );
	// convert glow type from double
	AlLightGlowType glowtype = (AlLightGlowType)int(value);
	switch ( glowtype )
	{
		case kLightGlowOff:
			typestr = "OFF";
			break;
		case kLightGlowLinear:
			typestr = "Linear";
			break;
		case kLightGlowExponential:
			typestr = "Exponential";
			break;
		case kLightGlowBall:
			typestr = "Ball";
			break;
		case kLightGlowSpectral:
			typestr = "Spectral";
			break;
		case kLightGlowRainbow:
			typestr = "Rainbow";
			break;
		default:
			typestr = "UNKNOWN";
			break;
	}
	printFile( "\n%sGlow Type: %s\n", indentSpace(), typestr);

	// Only print glow parameters if glow is not off.
	if ( glowtype != kLightGlowOff )
	{
		light->parameter( kFLD_LIGHT_GLOW_INTENSITY, value );
		printFile( "%sGlow Intensity    = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_GLOW_SPREAD, value );
		printFile( "%sGlow Spread       = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_GLOW_2DNOISE, value );
		printFile( "%sGlow 2D Noise     = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_GLOW_RADIAL_NOISE, value );
		printFile( "%sGlow Radial Noise = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_GLOW_STAR_LEVEL, value );
		printFile( "%sGlow Star Level   = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_GLOW_OPACITY, value );
		printFile( "%sGlow Opacity      = %g\n", indentSpace(), value);
	}

	light->parameter( kFLD_LIGHT_HALO_TYPE, value );
	// convert halo type from double
	AlLightHaloType halotype = (AlLightHaloType)int(value);
	switch ( halotype )
	{
		case kLightHaloOff:
			typestr = "OFF";
			break;
		case kLightHaloLinear:
			typestr = "Linear";
			break;
		case kLightHaloExponential:
			typestr = "Exponential";
			break;
		case kLightHaloBall:
			typestr = "Ball";
			break;
		case kLightHaloLensFlare:
			typestr = "Lens Flare";
			break;
		case kLightHaloRimHalo:
			typestr = "Rim Halo";
			break;
		default:
			typestr = "UNKNOWN";
			break;
	}
	printFile( "\n%sHalo Type: %s\n", indentSpace(), typestr);

	// Only print halo parameters if glow is not off.
	if ( halotype != kLightHaloOff )
	{
		light->parameter( kFLD_LIGHT_HALO_INTENSITY, value );
		printFile( "%sHalo Intensity    = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_HALO_SPREAD, value );
		printFile( "%sHalo Spread       = %g\n", indentSpace(), value);
	}

	light->parameter( kFLD_LIGHT_FOG_TYPE, value );
	// convert fog type and print.
	AlLightFogType fogtype = (AlLightFogType)int(value);
	switch ( fogtype )
	{
		case kLightFogOff:
			typestr = "OFF";
			break;
		case kLightFogLinear:
			typestr = "Linear";
			break;
		case kLightFogExponential:
			typestr = "Exponential";
			break;
		case kLightFogBall:
			typestr = "Ball";
			break;
		default:
			typestr = "UNKNOWN";
			break;
	}
	printFile( "\n%sFog Type: %s\n", indentSpace(), typestr);

	// Only print fog parameters if glow is not off.
	if ( fogtype != kLightFogOff )
	{
		light->parameter( kFLD_LIGHT_FOG_INTENSITY, value );
		printFile( "%sFog Intensity     = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_FOG_SPREAD, value );
		printFile( "%sFog Spread        = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_FOG_2DNOISE, value );
		printFile( "%sFog 2D Noise      = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_FOG_RADIAL_NOISE, value );
		printFile( "%sFog Radial Noise  = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_FOG_STAR_LEVEL, value );
		printFile( "%sFog Star Level    = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_FOG_OPACITY, value );
		printFile( "%sFog Opacity       = %g\n", indentSpace(), value);
	}

	// Additional parameters.
	if ( glowtype != kLightGlowOff || halotype != kLightHaloOff
		|| fogtype != kLightFogOff )
	{
		light->parameter( kFLD_LIGHT_RADIAL_FREQUENCY, value );
		printFile( "\n%sRadial Frequency  = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_STAR_POINTS, value );
		printFile( "%sStar Points       = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_ROTATION, value );
		printFile( "%sRotation          = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_NOISE_USCALE, value );
		printFile( "%sNoise U Scale     = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_NOISE_VSCALE, value );
		printFile( "%sNoise V Scale     = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_NOISE_UOFFSET, value );
		printFile( "%sNoise U Offset    = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_NOISE_VOFFSET, value );
		printFile( "%sNoise V Offset    = %g\n", indentSpace(), value);
		light->parameter( kFLD_LIGHT_NOISE_THRESHOLD, value );
		printFile( "%sNoise Threshold   = %g\n", indentSpace(), value);
	}
}

void printAlLight( Xconst AlLight *light, AlTM& tm )
{
	if( NULL == light ) return;

	printObjectName( light );

	INDENT

	double x, y, z;
	light->worldPosition( x, y, z );
	printFile( "%sLight Position: %g, %g, %g\n",
			indentSpace(), x, y, z );

	// Print out all the objects linked to this light
	//
	if( light->hasLinkedObjects() ) {
		printFile( "%sLINKED objects of this light:\n", indentSpace() );
		INDENT

		AlObject *nextPtr = NULL;
		for( AlObject *objPtr = light->firstLinkedObject();
			objPtr != NULL;
			objPtr = nextPtr )
		{
			const char *objName = objPtr->name();
			nextPtr = light->nextLinkedObject( objPtr );
			if ( objName == NULL )
				objName = "NULL";
			printFile( "%sObject = %s\n",indentSpace(),objName);
			delete objPtr;
		}
		UNINDENT
	}

	// Print the animation on this light (if any)
	//
	AlChannel *channel = light->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this light:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while ( sSuccess == light->nextChannelD(channel));
		UNINDENT
		delete channel;
	}

	// Print out the light parameters for this kind of light
	//
	switch( light->type() )
	{
	  case kAmbientLightType:
		AL_ASSERT( light->asAmbientLightPtr() != NULL,
					"Object is not an ambient light\n");
		printAlAmbientLight( light->asAmbientLightPtr(), tm );
		break;

	  case kPointLightType:
		AL_ASSERT( light->asPointLightPtr() != NULL,
					"Object is not an point light\n");
		printAlPointLight( light->asPointLightPtr(), tm );
		printAlLightGlow( light );
		break;

	  case kDirectionLightType:
		AL_ASSERT( light->asDirectionLightPtr() != NULL,
					"Object is not an directional light\n");
		printAlDirectionLight( light->asDirectionLightPtr(), tm );
		printAlLightGlow( light );
		break;

	  case kSpotLightType:
		AL_ASSERT( light->asSpotLightPtr() != NULL,
					"Object is not an spot light\n");
		printAlSpotLight( light->asSpotLightPtr(), tm );
		printAlLightGlow( light );
		break;

	  case kLinearLightType:
		AL_ASSERT( light->asLinearLightPtr() != NULL,
					"Object is not an linear light\n");
		printAlLinearLight( light->asLinearLightPtr(), tm );
		printAlLightGlow( light );
		break;

	  case kAreaLightType:
		AL_ASSERT( light->asAreaLightPtr() != NULL,
					"Object is not an area light\n");
		printAlAreaLight( light->asAreaLightPtr(), tm );
		printAlLightGlow( light );
		break;

	  default:
		break;
	}

	UNINDENT
}


void printAlAmbientLight( Xconst AlAmbientLight *light, AlTM& )
{
	if( NULL == light ) return;

	printFile( "%sShade factor = %g\n",
				indentSpace(), light->shadeFactor() );

	double r,g,b;
	light->color( r, g, b);
	printFile( "%sColour       = %g %g %g\n", indentSpace(), r, g, b );

	double intensity;
	intensity = light->intensity();
	printFile( "%sIntensity    = %g\n", indentSpace(), intensity );
}

void printAlPointLight( Xconst AlPointLight *light, AlTM& )
{
	if( NULL == light ) return;

	printFile( "%sIntensity = %g\n", indentSpace(),light->intensity());
	printFile( "%sDecay     = %d\n", indentSpace(), light->decay() );
	printFile( "%sShadows   = %d\n", indentSpace(), light->shadows() );

	double r,g,b;
	light->color( r, g, b);
	printFile( "%sColour    = %g %g %g\n", indentSpace(), r, g, b );
}

void printAlDirectionLight( Xconst AlDirectionLight *light, AlTM& )
{
	if( NULL == light ) return;

	printFile( "%sIntensity = %g\n", indentSpace(),light->intensity());
	printFile( "%sDecay     = %d\n", indentSpace(), light->decay() );
	printFile( "%sShadows   = %d\n", indentSpace(), light->shadows() );

	double r,g,b;
	light->color( r, g, b);
	printFile( "%sColour    = %g %g %g\n", indentSpace(), r, g, b );

	double x,y,z;
	light->direction( x, y, z);
	printFile( "%sDirection = %g %g %g\n", indentSpace(), x, y, z );
}

void printAlSpotLight( Xconst AlSpotLight *light, AlTM& )
{
	if( NULL == light ) return;

	printFile( "%sIntensity    = %g\n",
				indentSpace(), light->intensity() );
	printFile( "%sDecay        = %d\n",
				indentSpace(), light->decay() );
	printFile( "%sShadows      = %d\n",
				indentSpace(), light->shadows() );
	printFile( "%sDrop off     = %g\n",
				indentSpace(), light->dropOff() );
	printFile( "%sMax Bias     = %g\n",
				indentSpace(), light->maxBias() );
	printFile( "%sMin Bias     = %g\n",
				indentSpace(), light->minBias() );
	printFile( "%sSpread Angle = %g\n",
				indentSpace(), light->spreadAngle() );
	printFile( "%sOffset       = %d\n",
				indentSpace(), light->offset() );
	printFile( "%sMultFactor   = %d\n",
				indentSpace(), light->multFactor() );
	printFile( "%sShadowsize   = %d\n",
				indentSpace(), light->shadowSize() );
	printFile( "%sPenumbra     = %g\n",
				indentSpace(), light->penumbra() );

	double r,g,b;
	light->color( r, g, b);
	printFile( "%sColour       = %g %g %g\n",
				indentSpace(), r, g, b );

	double x,y,z;
	light->direction( x, y, z );
	printFile( "%sDirection    = %g %g %g\n",
				indentSpace(), x, y, z );
}

void printAlLinearLight( Xconst AlLinearLight *light, AlTM& )
{
	if( NULL == light ) return;

	printFile( "%sIntensity = %g\n",
				indentSpace(), light->intensity() );
	printFile( "%sDecay     = %d\n",
				indentSpace(), light->decay() );
	printFile( "%sShadows   = %d\n",
				indentSpace(), light->shadows() );
	double r,g,b;
	light->color( r, g, b);
	printFile( "%sColour    = %g %g %g\n", indentSpace(), r, g, b );

	double x,y,z;
	light->axis( x,y,z);
	printFile( "%sAxis      = %g %g %g\n", indentSpace(), x, y, z );
}

void printAlAreaLight( Xconst AlAreaLight *light, AlTM& )
{
	if( NULL == light ) return;

	printFile( "%sIntensity = %g\n",
				indentSpace(), light->intensity() );
	printFile( "%sDecay     = %d\n",
				indentSpace(), light->decay() );
	printFile( "%sShadows   = %d\n",
				indentSpace(), light->shadows() );

	double r,g,b;
	light->color( r, g, b);
	printFile( "%sColour    = %g %g %g\n",
				indentSpace(), r, g, b );

	double x,y,z;
	light->shortAxis( x,y,z);
	printFile( "%sShortAxis = %g, %g, %g\n",
				 indentSpace(), x, y, z);

	light->longAxis( x,y,z);
	printFile( "%sLong Axis = %g, %g, %g\n",
				 indentSpace(), x, y, z);
}

void printAlLightNode( Xconst AlLightNode *lightNode, AlTM& tm )
{
	if( NULL == lightNode ) return;

	switch( lightNode->type() )
	{
	  case kLightLookAtNodeType:
	  case kLightUpNodeType:
		printAlDagNode( lightNode, tm );
		break;

	default:
	{
		printAlDagNode( lightNode, tm );
		INDENT
		AlTM	newtm = tm;
		AlLight *light = lightNode->light(newtm);
		printAlLight( light, newtm );
		delete light;
		UNINDENT
		break;
	}
	}
}

void printAlCloud( Xconst AlCloud *cloud )
{
	if ( NULL == cloud ) return;
	
	printObjectName( cloud );
	INDENT
	printFile( "%sVisible = %s\n",indentSpace(),cloud->visible() ? "YES" : "NO" );
	printFile( "%sPicked = %s\n",indentSpace(),cloud->isPicked() ? "YES" : "NO" );
	printFile( "%sTransformations\n",indentSpace());
	INDENT
	double x,y,z;
	if ( cloud->translation(x,y,z) != sSuccess )
		return;
	printFile( "%sTranslation = %g %g %g\n",indentSpace(),x,y,z);
	if ( cloud->rotation(x,y,z) != sSuccess )
		return;
	printFile( "%sRotation = %g %g %g\n",indentSpace(),x,y,z);
	if ( cloud->scale(x,y,z) != sSuccess )
		return;
	printFile( "%sScale = %g %g %g\n",indentSpace(),x,y,z);
	UNINDENT

	double minX,minY,minZ,maxX,maxY,maxZ;
	if ( cloud->boundingBox( minX,minY,minZ,maxX,maxY,maxZ ) != sSuccess )
		return;
	printFile( "%sBounding Box Points %g,%g,%g to %g,%g,%g\n",
						indentSpace(),minX,minY,minZ,maxX,maxY,maxZ);
	int numOfPoints = cloud->numberOfPoints();
	printFile( "%sNumber of Points: %d\n",indentSpace(),numOfPoints );
	if ( numOfPoints != 0 )
	{
		printFile( "%sList of Points:\n",indentSpace());
		INDENT
		float *array = new float[numOfPoints * 3];
		if ( array == NULL )
			return;
		if ( cloud->points( numOfPoints * 3, array ) != sSuccess )
			return;

		float *arrayPtr=array;
		for ( int i = 0; i < numOfPoints ; i++ )
		{
			printFile( "%s( %f %f %f )\n",indentSpace(),arrayPtr[0],arrayPtr[1],arrayPtr[2]);
			arrayPtr+=3;
		}
		UNINDENT
	}
	UNINDENT
}

void printAlClusterNode( Xconst AlClusterNode *clusterNode, AlTM& tm )
{
	if( NULL == clusterNode ) return;

	printAlDagNode( clusterNode, tm );

	INDENT
	AlTM newtm = tm;
	AlCluster *cluster = clusterNode->cluster(newtm);
	printAlCluster( cluster, newtm );
	delete cluster;
	UNINDENT
}

void printAlCluster( Xconst AlCluster *cluster, AlTM& )
{
	if (NULL == cluster ) return;

	printObjectName( cluster );
	INDENT
	AlClusterRestrict restrict;
	cluster->clusterRestrict( restrict );
	printFile( "%sType         = %s\n", indentSpace(),
		(restrict == kMultiCluster ? "MULTI" : "EXCLUSIVE") );
	printFile( "%sIsEmpty?     = %s\n", indentSpace(),
										cluster->isEmpty() ? "YES" : "NO" );
	printFile( "%s# of members = %d\n", indentSpace(),
										cluster->numberOfMembers() );

	AlClusterMember *member = cluster->firstMember();
	if (member != NULL )
	{
		printFile( "%sMembers:\n", indentSpace() );
		INDENT

		do {
			AlCurveCV	*curveCV;
			AlSurfaceCV	*surfaceCV;
			AlObject	*object = member->object();

			printObjectName( object );
			if (curveCV = object->asCurveCVPtr())
			{
				printFile( "%s  %% effect = %g\n", indentSpace(),
											curveCV->percentEffect(cluster));
			}
			else if (surfaceCV = object->asSurfaceCVPtr())
			{
				printFile( "%s  %% effect = %g\n", indentSpace(),
											surfaceCV->percentEffect(cluster));
			}
			delete object;
		} while( sSuccess == member->nextClusterMemberD() );
		UNINDENT
		delete member;
	}
	UNINDENT
}

void printAlSet( Xconst AlSet *set )
{
	if( NULL == set ) {
		return;
	}

	// Print each set name and its type - exclusive or multiple...
	//
	if( set->isExclusive() ) {
		 printFile(
			   "Set named \"%s\" is EXCLUSIVE & has %d member(s):\n",
			   set->name(), set->numberOfMembers());
	} else {
		 printFile(
			   "Set named \"%s\" is MULTI & has %d member(s):\n",
			   set->name(), set->numberOfMembers());
	}

	// Print each set member
	//
	AlSetMember *setMember = set->firstMember();
	if( setMember != NULL )
	{
		INDENT
		do {
			AlObject *object = setMember->object();

			AL_ASSERT( object != NULL, "printAlSet: Internal logic error.");
			printObjectName( object );
			delete object;
		} while( sSuccess == setMember->nextSetMemberD());
		UNINDENT
		delete setMember;
	}
}

class cloudIterator: public AlIterator
{
public:
	virtual int func( AlObject* );
};

int cloudIterator::func( AlObject* obj )
{
	if ( obj == NULL )
		return 0;
	printAlCloud( obj->asCloudPtr() );
	return 0;
}

void printAllClouds()
{
	int rc;
	cloudIterator *ci = new cloudIterator;
	if ( ci == NULL )
		return;
	AlUniverse::applyIteratorToClouds( ci, rc );
	delete ci;
}

void printAllBlendCurves()
{
	AlBlendCurve *blendCurve = AlUniverse::firstBlendCurve();
	if ( blendCurve != NULL )
	{
		do {
			printAlBlendCurve( blendCurve );
		} while ( sSuccess == blendCurve->nextD( blendCurve ) );
		delete blendCurve;
	}
}

void printAllLocators()
{
	AlLocator *locator = AlUniverse::firstLocator();
	if ( locator != NULL )
	{
		while ( locator != NULL )
		{
			printAlLocator( locator );
			AlLocator *cur = locator;
			locator = AlUniverse::nextLocator(locator);
			delete cur;
		}
		delete locator;
	}
}

void printAllConstructionEntities()
{
	AlConstructionEntity *ce = AlUniverse::firstConstructionEntity();
	if ( ce != NULL )
	{
		while ( ce != NULL)
		{
			printAlConstructionEntity( ce );
			AlConstructionEntity *cur = ce;
			ce = AlUniverse::nextConstructionEntity(ce);
			delete cur;
		}
		delete ce;
	}
}

static void recurseLayerPrint(AlLayer* layer)
{
    printAlLayer( layer );

    AlLayer* child = layer->childLayer();
    if ( child )
        recurseLayerPrint( child );

    AlLayer* next = layer->nextLayer();
    if ( next )
        recurseLayerPrint( next );

    delete layer;
}

void printAllLayers()
{
	AlLayer *ly = AlUniverse::firstLayer();
	if ( ly )
		recurseLayerPrint(ly);
}

void printAllSets()
{
	// Print each set in the universe and its type (exclusive or multiple)
	// and its set members
	//
	AlSet *set = AlUniverse::firstSet();
	if( set != NULL )
	{
		do {
			printAlSet( set );
		} while( sSuccess == set->nextSetD());
		delete set;
	}
}

void printAllChannels()
{
	AlChannel	*channel = AlUniverse::firstChannel();
	if( channel != NULL )
	{
		do {
			printAlChannel(channel);
		} while( sSuccess == AlUniverse::nextChannelD(channel));
		delete channel;
	}
}

const char *extrapTypeString( AlActionExtrapType	type)
{
	switch(type)
	{
		case kEXTRAP_CONSTANT:	return "CONSTANT";
		case kEXTRAP_CYCLE:		return "CYCLE";
		case kEXTRAP_OSCILLATE:	return "OSCILLATE";
		case kEXTRAP_LINEAR:	return "LINEAR";
		case kEXTRAP_IDENTITY:	return "IDENTITY";

		default:
			return "??";
	}
}

const char *tripleComponentString( AlTripleComponent component)
{
	switch(component)
	{
		case kX_COMPONENT:	return "X";
		case kY_COMPONENT:	return "Y";
		case kZ_COMPONENT:	return "Z";

		default:
			return "??";
	}
}

void printAlChannel(Xconst AlChannel *channel)
{
	if (!channel)
		return;

	printChannelInfo( channel );
	switch (channel->channelType ()) {

		case kAnimChannel:
		{
			printFile( "  Animation channel:\n");
			for (int i = channel->numAppliedActions(); i >= 1; i--)
			{
				AlAction *action = channel->appliedAction(i);
				const char *actionName = action->name();

				if (action->type() == kMotionActionType)
				{
					printFile( "    Action %d: %s [", i, actionName);
					printFile( tripleComponentString(channel->appliedActionComponent(i)));
					printFile( "]\n");
				}
				else
				{
					printFile( "    Action %d: %s\n", i, actionName);
				}
				delete action;
			}
			break;
		}

		case kExprChannel:
		{
			const char *exprString = channel->expressionString ();
			printFile( "  Expression channel:\n");
			printFile( "    Expression = %s;\n", exprString );
			break;
		}
	}
}

void printAllActions()
{
AlAction *action = AlUniverse::firstAction();
AlAction *nextact = NULL;

for( ;action != NULL; action = nextact )
	{
	printAlAction(action);
	nextact = AlUniverse::nextAction( action );
	delete action;
	}
}

void printAlAction(Xconst AlAction *action)
{
	int		 	num;

	if (!action)
		return;

	const char *	className = determineObjectClassName(action->type());
	{
		const char *	name = action->name();
		const char *	comment= action->comment();

		printFile( "%s%s(%s)\n", indentSpace(), className, name);
		INDENT

		if ( comment == NULL )
			comment = "NULL";

		printFile( "%sComment: %s\n", indentSpace(), comment );
	}
	printFile( "%sPre-extrap type: ", indentSpace());
	printFile( extrapTypeString(action->extrapTypePRE()));
	printFile( "\n%sPost-extrap type: ", indentSpace());
	printFile( extrapTypeString(action->extrapTypePOST()));
	printFile( "\n");
	num = action->numChannelReferences();
	printFile( "%sNum channels using action: %d\n", indentSpace(), num);

	INDENT
	for (int i = num; i > 0; i--)
	{
		AlChannel *channel = action->channelReference(i);
		printFile( "(%d) ",i );
		printChannelInfo( channel );
		delete channel;
	}
	UNINDENT

	switch(action->type())
	{
		case kParamActionType:
			printAlParamAction(action->asParamActionPtr());
			break;
		case kMotionActionType:
			printAlMotionAction(action->asMotionActionPtr());
			break;
	}
	UNINDENT
}

void printAlParamAction( Xconst AlParamAction *action)
{
	printFile( "%sParam action keyframes:\n", indentSpace());
	INDENT

	AlKeyframe *keyframe = action->firstKeyframe();
	if( keyframe != NULL )
	{
		do {
			printAlKeyframe(keyframe);
		} while( sSuccess == keyframe->nextD());
		delete keyframe;
	}
	UNINDENT
}

void printAlMotionAction( Xconst AlMotionAction	*action )
{
	AlCurveNode	*curve = action->motionCurve();
	const char *curveName = curve->name();

	printFile( "%sMotion action curve:\n", indentSpace());
	INDENT
	printFile( "%s%s\n", indentSpace(), curveName);
	UNINDENT

	delete curve;
}

void printAlKeyframe( Xconst	AlKeyframe	*keyframe)
{
	double	s1, s2;

	if (!keyframe)
		return;

	const char *	className = determineObjectClassName(keyframe->type());

	printFile( "%s%s", indentSpace(), className);
	INDENT

	s1 = keyframe->location();
	s2 = keyframe->value();
	printFile( "(%.1f, %.1f) ", s1, s2);

	printFile( "in-tan: %.1f, out-tan: %.1f, locked? %s\n",
			keyframe->inTangent(), keyframe->outTangent(),
			(keyframe->isLocked() ? "TRUE" : "FALSE"));

	UNINDENT
}

void printAllShading()
{
	AlEnvironment	*env = AlUniverse::firstEnvironment();
	AlTM	tm = AlTM::identity();
	printAlEnvironment(env, tm);

	AlShader		*shader = AlUniverse::firstShader();
	if(shader != NULL)
	{
		do {
			printAlShader(shader, tm);
		} while( sSuccess == AlUniverse::nextShaderD(shader));
		delete shader;
	}
	delete env;
}

void printAlEnvironment( Xconst AlEnvironment *env, AlTM& tm)
{
	if (!env)
		return;

	printObjectName(env);

	INDENT

	// Print the fields on this environment
	//
	AlList*				list = env->fields();
	if ( list != NULL )
	{
		for( AlShadingFieldItem *fieldItem = (AlShadingFieldItem*)list->first();
			 fieldItem != NULL;
			 fieldItem = fieldItem->nextField() )
		{
			double fieldValue;

			// Get the name for the field and value
			env->parameter(fieldItem->field(), fieldValue);
			printFile( "%s%s = %g\n", indentSpace(),
				determineFieldName(fieldItem->field()), fieldValue);
		}
		delete list;
	}

	// Print the fields on this environment
	//
	list = env->mappedFields();
	if ( list != NULL )
	{
		printFile( "%sPOSSIBLE texture maps on this environment:\n",
					indentSpace());
		INDENT
		for( AlMappedFieldItem*	mappedItem = (AlMappedFieldItem*)list->first();
			 mappedItem != NULL;
			 mappedItem = mappedItem->nextField())
		{
			printFile( "%s%s\n", indentSpace(), mappedItem->field());
		}
		delete list;
		UNINDENT
	}

	// Print the animation on this environment (if any)
	//
	AlChannel *channel = env->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this environment:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while (sSuccess == env->nextChannelD(channel) );
		UNINDENT
		delete channel;
	}

	// Print the environment's textures
	//
	AlTexture	*texture = env->firstTexture();
	if (texture != NULL)
	{
		do {
			printAlTexture(texture, tm);
		} while( sSuccess == env->nextTextureD(texture));
		delete texture;
	}
	UNINDENT
}

void printAlShader( Xconst AlShader *shader, AlTM& tm)
{
	if (!shader)
		return;

	printObjectName(shader);

	INDENT

	// Print the fields on this shader
	//
	AlList*				list = shader->fields();
	if ( list != NULL )
	{
		for( AlShadingFieldItem* fieldItem = (AlShadingFieldItem*)list->first();
			 fieldItem != NULL;
			 fieldItem = fieldItem->nextField())
		{
			double fieldValue;

			// Get the name for the field and value
			shader->parameter(fieldItem->field(), fieldValue);
			printFile( "%s%s = %g\n", indentSpace(),
				determineFieldName(fieldItem->field()), fieldValue);
		}
		delete list;
	}

	// Print the fields on this shader
	//
	list = shader->mappedFields();
	if ( list != NULL )
	{
		printFile( "%sPOSSIBLE texture maps on this shader:\n",
			indentSpace());
		INDENT
		for( AlMappedFieldItem*	mappedItem = (AlMappedFieldItem*)list->first();
			 mappedItem != NULL;
			 mappedItem = mappedItem->nextField())
		{
			printFile( "%s%s\n", indentSpace(), mappedItem->field());
		}
		UNINDENT
		delete list;
	}

	// Print the animation on this shader (if any)
	//
	AlChannel *channel = shader->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this shader:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while ( sSuccess == shader->nextChannelD(channel));
		UNINDENT
		delete channel;
	}

	// Print the shader's textures
	//
	AlTexture	*texture = shader->firstTexture();
	if (texture != NULL)
	{
		do {
			printAlTexture(texture, tm);
		} while( sSuccess == shader->nextTextureD(texture));
		delete texture;
	}
	UNINDENT
}

void printAlTexture( Xconst AlTexture *texture, AlTM& tm)
{
	const char * temp;

	if (!texture)
		return;

	printObjectName(texture);

	INDENT

	temp = texture->textureType();
	if ( temp != NULL )
	{
		printFile( "%sTexture Type = %s\n", indentSpace(), temp );
	}

	temp = texture->fieldType();
	if ( temp != NULL )
	{
		printFile( "%sField Type = %s\n", indentSpace(), temp );
	}

	temp = texture->filename();
	if ( temp != NULL )
	{
		printFile( "%sFilename = %s\n", indentSpace(), temp );
	}

	AlTextureNode*	texDag = texture->textureNode();
	if ( NULL != texDag )
	{
		const char *dagName = texDag->name();
		printFile( "%sTexture AlDagNode = %s\n", indentSpace(), dagName );
		delete texDag;
	}

	// Print the fields on this texture
	//
	AlList*				list = texture->fields();
	if ( list != NULL )
	{
		for( AlShadingFieldItem* fieldItem = (AlShadingFieldItem*)list->first();
			 fieldItem != NULL;
			 fieldItem = fieldItem->nextField() )
		{
			double fieldValue;

			// Get the name for the field and value
			texture->parameter(fieldItem->field(), fieldValue);
			printFile( "%s%s = %g\n", indentSpace(),
				determineFieldName(fieldItem->field()), fieldValue);
		}
		delete list;
	}

	// Print the fields on this texture
	//
	list = texture->mappedFields();
	if( list != NULL )
	{
		printFile( "%sPOSSIBLE texture maps on this texture:\n", indentSpace());
		INDENT

		for( AlMappedFieldItem*	mappedItem = (AlMappedFieldItem*)list->first();
			 mappedItem != NULL;
			 mappedItem = mappedItem->nextField())
		{
			printFile( "%s%s\n", indentSpace(), mappedItem->field());
		}
		UNINDENT
		delete list;
	}

	// Print the animation on this texture (if any)
	//
	AlChannel *channel = texture->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this texture:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while ( sSuccess == texture->nextChannelD(channel));
		UNINDENT
		delete channel;
	}

	// Print the texture's sub-textures
	//
	AlTexture	*sub = texture->firstTexture();
	if( sub != NULL )
	{
		do {
			printAlTexture(sub, tm);
		} while( sSuccess == texture->nextTextureD(sub));
		delete sub;
	}
	UNINDENT
}

void printAlTextureNode( Xconst AlTextureNode* textureNode, AlTM& tm )
{
	if ( NULL == textureNode ) return;

	printAlDagNode( textureNode, tm );

	INDENT
	AlTM newtm = tm;
	AlTexture*	texture = textureNode->texture(newtm);
	if (texture != NULL)
	{
		const char *textureName = texture->name();
		printFile( "%sTexture = %s\n", indentSpace(), textureName );
	}
	delete texture;
	UNINDENT
}

void printAlPolysetVertex( const AlPolysetVertex* vertex, AlTM& tm )
{
	if ( NULL == vertex ) return;

	int					index;
	double				x, y, z;
	double				ux, uy, uz;
	double				nx, ny, nz;
	double				ax, ay, az;
	double              s, t;
	double              r, g, b, a;

	printObjectName( vertex );

	INDENT
	if ( vertex->worldPosition( x, y, z ) == sSuccess )
		printFile( "%s(%g %g %g) ", indentSpace(), x, y, z );

	if ( ( index = vertex->index() ) != -1 )
		printFile( "Index: %d\n", index );

	if ( vertex->unaffectedPosition( ux, uy, uz ) == sSuccess )
		printFile( "%sUnaffected position: (%g %g %g)\n",
			indentSpace(), ux, uy, uz);

	if ( vertex->affectedPosition( tm, ax, ay, az ) == sSuccess )
		printFile( "%sAffected position: (%g %g %g)\n",
			indentSpace(), ax, ay, az);

	if ( vertex->normal( nx, ny, nz ) == sSuccess )
		printFile( "%sNormal: (%g %g %g)\n", indentSpace(), nx, ny, nz);

	if ( vertex->st( s, t ) == sSuccess )
		printFile( "%sST: (%g %g)\n", indentSpace(), s, t );

	if ( vertex->color( r, g, b, a ) == sSuccess )
		printFile( "%sColor: (%g %g %g %g)\n", indentSpace(), r, g, b, a );

	// Print the clusters that this vertex is in (if any)
	//
	AlCluster *cluster = vertex->firstCluster();
	if (cluster != NULL)
	{
		printFile( "%sCLUSTERS this vertex is in:\n", indentSpace());
		INDENT
		do {
			AlClusterNode * cn = cluster->clusterNode();
			const char *clusterNodeName = cn->name();
			printFile( "%scluster: %s\n",
						indentSpace(), clusterNodeName);
			delete cn;
		} while ( sSuccess == vertex->nextClusterD(cluster) );
		UNINDENT
		delete cluster;
	}

	// Print the animation on this vertex (if any)
	//
	AlChannel *channel = vertex->firstChannel();
	if (channel != NULL)
	{
		printFile( "%sANIMATION on this vertex:\n", indentSpace());
		INDENT
		do {
			printChannelInfo( channel );
		} while ( sSuccess == vertex->nextChannelD(channel));
		UNINDENT
		delete channel;
	}
	UNINDENT
}

void printAlPolygon( const AlPolygon* polygon, AlTM& tm )
{
	if ( NULL == polygon ) return;

	int					index;
	int					numVertices;
	double				nx, ny, nz;

	printObjectName( polygon );

	INDENT
	if ( ( numVertices = polygon->numberOfVertices() ) != -1 )
		printFile( "%sNumber of Vertices = %d\n",
			indentSpace(), numVertices );

	if ( polygon->normal( nx, ny, nz ) == sSuccess )
		printFile( "%sNormal: (%g %g %g)\n",
			indentSpace(),nx, ny, nz);

	AlPolysetVertex*	vertex;
	for ( index = 0; index < numVertices; index++ )
	{
		vertex = polygon->vertex(index);
		printAlPolysetVertex( vertex, tm );
		delete vertex;
	}
	UNINDENT
}

void printAlPolyset( const AlPolyset* polyset, AlTM& tm )
{
	if ( NULL == polyset ) return;

	int					index;
	int					numVertices = polyset->numberOfVertices();
	int					numPolygons = polyset->numberOfPolygons();

	printObjectName( polyset );

	INDENT
	printFile( "%sNumber of Polygons = %d\n",
		indentSpace(), numPolygons );
	printFile( "%sNumber of Vertices = %d\n",
		indentSpace(), numVertices );

	// Print out the render info on this surface
	//
	AlRenderInfo		renderInfo;
	if( sSuccess == polyset->renderInfo( renderInfo ) )
	{
		printFile( "%sPolyset is %sDOUBLE SIDED.\n", indentSpace(),
				(renderInfo.doubleSided ? "" : "not "));
		printFile( "%sPolyset is %sopposite.\n", indentSpace(),
				(renderInfo.opposite ? "" : "not "));
		printFile( "%sPolyset will %scast shadows.\n", indentSpace(),
				(renderInfo.castsShadow ? "" : "not "));
	}

	AlShader*	shader = polyset->firstShader();
	if ( shader != NULL )
	{
		printFile( "%sShaders on this polyset:\n", indentSpace());
		INDENT
		do {
			const char *shaderName = shader->name();
			printFile( "%s%s\n", indentSpace(), shaderName);
		} while( sSuccess == polyset->nextShaderD( shader ) );
		UNINDENT
		delete shader;
	}

	AlPolysetVertex*	vertex;
	for ( index = 0; index < numVertices; index++ )
	{
		vertex = polyset->vertex(index);
		printAlPolysetVertex( vertex, tm );
		delete vertex;
	}

	AlPolygon*			polygon;
	for ( index = 0; index < numPolygons; index++ )
	{
		polygon = polyset->polygon(index);
		printAlPolygon(polygon, tm);
		delete polygon;
	}
	UNINDENT
}

void printAlPolysetNode( Xconst AlPolysetNode *polysetNode, AlTM& tm )
{
	if ( NULL == polysetNode ) return;

	printAlDagNode( polysetNode, tm );

	INDENT
	AlTM newtm = tm;
	AlPolyset *polyset = polysetNode->polyset(newtm);
	printAlPolyset( polyset, newtm );
	delete polyset;
	UNINDENT
}


void printAlMesh( const AlMesh* mesh )
{
	if ( NULL == mesh ) return;

	printObjectName( mesh );

	INDENT
	int numVertices  = mesh->numberOfVertices();
	int numTriangles = mesh->numberOfTriangles();
	printFile( "%sNumber of Triangles = %d\n", indentSpace(), numTriangles );
	printFile( "%sNumber of Vertices = %d\n", indentSpace(), numVertices );

	// Print out the render info on this surface
	//
	AlRenderInfo		renderInfo;
	if( sSuccess == mesh->renderInfo( renderInfo ) )
	{
		printFile( "%sMesh is %sDOUBLE SIDED.\n", indentSpace(),
				(renderInfo.doubleSided ? "" : "not "));
		printFile( "%sMesh is %sopposite.\n", indentSpace(),
				(renderInfo.opposite ? "" : "not "));
		printFile( "%sMesh will %scast shadows.\n", indentSpace(),
				(renderInfo.castsShadow ? "" : "not "));
	}
    
	AlShader*	shader = mesh->firstShader();
	if ( shader != NULL )
	{
		printFile( "%sShaders on this mesh:\n", indentSpace());
		INDENT
		do {
			const char *shaderName = shader->name();
			printFile( "%s%s\n", indentSpace(), shaderName);
		} while( sSuccess == mesh->nextShaderD( shader ) );
		UNINDENT
		delete shader;
	}

	// Draw flag
	printFile( "%sMesh draw flag: %d\n", indentSpace(), mesh->drawFlag() );

	// Flat Shaded
	printFile( "%sMesh is %sflat shaded\n", indentSpace(), mesh->isFlatShaded() ? "" : "not ");

	// Bounding Box
	double min[3], max[3];
	mesh->boundingBox( min, max);
	printFile( "%sBounding Box: Min(%g %g %g)  Max(%g %g %g)\n", indentSpace(), min[0], min[1], min[2], max[0], max[1], max[2] );

	// Range for uv coordinates
    float uvRange[4];
    mesh->uvRange( uvRange );
    printFile( "%sUV Range:     (%g %g %g %g)\n", indentSpace(),
               uvRange[0], uvRange[1],
               uvRange[2], uvRange[3] );
    
	// Vertices
	printFile( "%s%d Vertices:\n", indentSpace(), numVertices );
	int index;
	const float *vertices = mesh->vertices();
	if (vertices)
	{
		INDENT
		for (index = 0; index < numVertices; index++)
		{
			const float *vtx = vertices + 3*index;
			printFile( "%sVertex[%d]   (%g %g %g)\n", indentSpace(), index+1, vtx[0], vtx[1], vtx[2] );
		}
		UNINDENT
	}

	// Normals
	const float *normals = mesh->normals();
	if (normals)
	{
		printFile( "%s%d Normals:\n", indentSpace(), numVertices );
		INDENT
		for (index = 0; index < numVertices; index++)
		{
			const float *nrm = normals + 3*index;
			printFile( "%sNormal[%d]   (%g %g %g)\n", indentSpace(), index+1, nrm[0], nrm[1], nrm[2] );
		}
		UNINDENT
	}

	// UVs
	const float *uvs = mesh->uvs();
	if (uvs)
	{
		printFile( "%s%d UVs:\n", indentSpace(), numVertices );
		INDENT
		for (index = 0; index < numVertices; index++)
		{
			const float *uv = uvs + 2*index;
			printFile( "%sUV[%d]       (%g %g)\n", indentSpace(), index+1, uv[0], uv[1] );
		}
		UNINDENT
	}

	// Triangles
	printFile( "%s%d Triangles:\n", indentSpace(), numTriangles );
	const int *triangles = mesh->triangles();
	if (triangles)
	{
		INDENT
		for (index = 0; index < numTriangles; index++)
		{
			const int *tri = triangles + 3*index;
			int v1 = tri[0]+1;
			int v2 = tri[1]+1;
			int v3 = tri[2]+1;
			//if (v1 == v2 || v1 == v3 || v2 == v3)  // degenerate triangle
				//continue;
			// Triangle output by referencing indexes, similar to the face
			// command referencing vertex data in .obj file:
			//         f  v1/vt1/vn1 v2/vt2/vn2 v3/vt3/vn3
			// Each triplet of indexes specifies a geometric vertex, texture
			// vertex and vertex normal.
			printFile( "%sTriangle[%d]  %d/%d/%d  %d/%d/%d  %d/%d/%d\n", indentSpace(), index+1, v1, v1, v1, v2, v2, v2, v3, v3, v3 );
		}
		UNINDENT
	}

	UNINDENT
}

void printAlMeshNode( Xconst AlMeshNode *meshNode, AlTM& tm )
{
	if ( NULL == meshNode ) return;

	printAlDagNode( meshNode, tm );

	INDENT
	AlMesh *mesh = meshNode->mesh();
	printAlMesh( mesh );
	delete mesh;
	UNINDENT
}


void printAlBlendCurve( Xconst AlBlendCurve *blendCurve )
{
	if ( blendCurve == NULL ) return;
	
 	printObjectName( blendCurve );	

	INDENT
	
	AlCurveNode *curveNode;
	curveNode = blendCurve->curveNode();
	if ( curveNode != NULL )
	{
		printObjectName( curveNode );
		delete curveNode;
	}
	
	printFile( "%sDegree = %d\n",indentSpace(), blendCurve->degree() );
	printFile( "%sIs Picked = %s\n",indentSpace(),(blendCurve->isPicked()) ? "on" : "off"  );
	
	AlBlendCurve::AlKnotSpacingType kst;
	if ( blendCurve->knotSpacingType( kst ) == sSuccess )
	{
		const char *kst_string = NULL;
		switch ( kst )
		{
			case AlBlendCurve::kChord: kst_string="kChord";
								break;
			case AlBlendCurve::kUniform: kst_string="kUniform";
								break;
			default: kst_string="kUnknown";
								break;
		}
		printFile( "%sKnot Spacing: %s\n",indentSpace(),kst_string);						
	}
	
	int numPoints = blendCurve->numberOfPoints();	
	printFile( "%sNumber of Points = %d\n",indentSpace(), numPoints );
	
	double *points = new double [numPoints * 3 ];
	if ( points != NULL )
	{
		if ( blendCurve->points( numPoints, points ) == sSuccess )
		{
			INDENT
			int i;
			double *pt = points;
			for ( i = 0; i < numPoints; i++ )
			{
				printFile( "%s( %g %g %g ) \n",indentSpace(),pt[0],pt[1],pt[2]);
				pt += 3; 
			}
			UNINDENT
		}
		delete points;
	}
	
	AlBlendPoint *point = blendCurve->getPoint( 0 );
	if ( point != NULL )
	{
		do {
			printAlBlendPoint( point );
		} while ( sSuccess == point->nextD( point ) );
		delete point;
	}
	
	UNINDENT
}

void printAlBlendPoint( Xconst AlBlendPoint *blendPoint )
{
	if ( blendPoint == NULL ) return;

	printObjectName( blendPoint );
	INDENT
	
	INDENT
	double x,y,z;
	if ( blendPoint->point( x, y, z ) == sSuccess )
		printFile( "%sPoint: ( %g %g %g ) \n",indentSpace(),x,y,z);
	
	double t;
	if ( blendPoint->param( t ) == sSuccess )
		printFile( "%sParameterization: %g\n",indentSpace(),t);
		
	double tx,ty,tz;
	if ( blendPoint->tangent( tx, ty, tz ) == sSuccess )
		printFile( "%sTangent: ( %g %g %g )\n",indentSpace(),tx,ty,tz);
		
	AlBlendPoint::AlInterpolationDirectionType id;
	if ( blendPoint->interpolationDirectionType( id ) == sSuccess )
	{
		const char *id_string = NULL;
		switch ( id )
		{
			case AlBlendPoint::kLocation: 			id_string="kLocation"; break;
			case AlBlendPoint::kWorldSpace: 		id_string="kWorldSpace"; break;
			case AlBlendPoint::kGeometricInterp: 	id_string="kGeometricInterp"; break;
			default: 							id_string = "kUnknown"; break;
		}
		printFile( "%sInterpolation Direction: %s\n",indentSpace(),id_string);
	}
	
	AlBlendPoint::AlDirectionType dt;
	if ( blendPoint->directionType( dt ) == sSuccess )
	{
		const char *dt_string = NULL;
		switch ( dt )
		{
			case AlBlendPoint::kRay:		dt_string="kRay"; break;
			case AlBlendPoint::kParallel:	dt_string="kParallel"; break;
			default: 					dt_string="kUnknown"; break;
		}
		printFile( "%sDirection Type: %s\n",indentSpace(),dt_string);
	}

	AlBlendPoint::AlCurvatureType ct;
	if ( blendPoint->curvatureType( ct ) == sSuccess )
	{
		const char *ct_string = NULL;
		switch ( ct )
		{
			case AlBlendPoint::kGeometricCurv:		ct_string="kGeometricCurv"; break;
			case AlBlendPoint::kParametric:			ct_string="kParametric"; break;
			default: 							ct_string="kUnknown"; break;
		}
		printFile( "%sCurvature Type: %s\n",indentSpace(),ct_string);
	}

	AlBlendPoint::AlConstraintContinuityType cct;
	if ( blendPoint->constraintContinuityType( cct ) == sSuccess )
	{
		const char *cct_string = NULL;
		switch ( cct )
		{
			case AlBlendPoint::kG0:		cct_string="kG0"; break;
			case AlBlendPoint::kG1:		cct_string="kG1"; break;
			case AlBlendPoint::kG2:		cct_string="kG2"; break;
			case AlBlendPoint::kG3:		cct_string="kG3"; break;
			case AlBlendPoint::kG4:		cct_string="kG4"; break;
			default: 				cct_string="kUnknown"; break;
		}
		printFile( "%sConstraint Continuity Type: %s\n",indentSpace(),cct_string);
	}

	AlObject *obj = NULL;
	double p1, p2;
	if ( blendPoint->attachedTo( obj, p1, p2 ) == sSuccess )
	{
		char buf[100];	
		if ( obj->type() == kSurfaceNodeType || obj->type() == kCloudType )
			sprintf(buf, "u %f v %f",p1, p2 );
		else
			sprintf(buf, "t %f", p1 );
			
		printFile( "%sAttached to: %s %s %s\n",indentSpace(),obj->name(),
		determineObjectClassName( obj->type() ) ,buf );			
	}
	UNINDENT
	
	UNINDENT
}

void printAlLocator( AlLocator *locator )
{
	if ( locator == NULL ) return;

	printObjectName( locator );
	INDENT
	
	INDENT
	
	boolean invisible = locator->invisible();
	printFile( "%sInvisible: %s\n",indentSpace(), (invisible) ? "TRUE" : "FALSE" );
		
	boolean picked;
	if ( locator->isPicked( picked ) == sSuccess )
		printFile( "%sPicked: %s\n",indentSpace(), (picked) ? "TRUE" : "FALSE" );
		
	AlLayer *l = locator->layer();
	if ( l != NULL )
	{
		int layernum = l->number();
		printFile( "%sLayer number: %d\n",indentSpace(), layernum );
		delete l;
	}
		
	if ( locator->type() == kAnnotationLocatorType )
	{
		AlAnnotationLocator *al = locator->asAnnotationLocatorPtr();
		if ( al != NULL )
		{
			const char *string = al->string();
			if ( string != NULL )
				printFile( "%sstring: %s\n",indentSpace(),string);
				
			double x,y,z;
			if ( al->worldLeaderPosition(x,y,z) == sSuccess )
				printFile("%sWorld Leader Position: %g %g %g\n",indentSpace(),x,y,z);
				
			if ( al->localLeaderPosition(x,y,z) == sSuccess )
				printFile("%sLocal Leader Position: %g %g %g\n",indentSpace(),x,y,z);	
				
			boolean justify;
			if ( al->leftJustify( justify ) == sSuccess )
				printFile( "%sLeft justify: %s\n",indentSpace(), (justify) ? "TRUE" : "FALSE" );		
		}
	}
	else if ( locator->type() == kDistanceLocatorType )
	{
		AlDistanceLocator *dl = locator->asDistanceLocatorPtr();
		if ( dl != NULL )
		{
			INDENT
			printFile( "%sStart Point:\n",indentSpace());
			AlPoint *start = dl->startPoint();
			if ( start != NULL )
			{
				printAlConstructionEntity( start );
				delete start;
			}
			
			printFile( "%sEnd Point:\n",indentSpace());
			AlPoint *end = dl->endPoint();
			if ( end != NULL )
			{
				printAlConstructionEntity( end );
				delete end;
			}
			UNINDENT
			
			boolean trueDisplay = FALSE;
			if ( dl->trueDisplay( trueDisplay ) == sSuccess )
				printFile( "%sTrue display: %s\n",indentSpace(), (trueDisplay) ? "TRUE" : "FALSE" );		

			double offset;
			if ( dl->offset( offset ) == sSuccess )
				printFile("%sOffset: %g\n",indentSpace(),offset);

			double x,y,z;
			if ( dl->distance(x,y,z) == sSuccess )
				printFile("%sDistance: %g %g %g\n",indentSpace(),x,y,z);
		}
	}
	else if ( locator->type() == kAngleLocatorType )
	{
		AlAngleLocator *al = locator->asAngleLocatorPtr();
		if ( al != NULL )
		{
			INDENT
			printFile( "%sStart Point:\n",indentSpace());
			AlPoint *start = al->startPoint();
			if ( start != NULL )
			{
				printAlConstructionEntity( start );
				delete start;
			}

			printFile( "%sMiddle Point:\n",indentSpace());
			AlPoint *middle = al->middlePoint();
			if ( middle != NULL )
			{
				printAlConstructionEntity( middle );
				delete middle;
			}

			printFile( "%sEnd Point:\n",indentSpace());
			AlPoint *end = al->endPoint();
			if ( end != NULL )
			{
				printAlConstructionEntity( end );
				delete end;
			}
			UNINDENT
			
			boolean trueDisplay = FALSE;
			if ( al->trueDisplay( trueDisplay ) == sSuccess )
				printFile( "%sTrue display: %s\n",indentSpace(), (trueDisplay) ? "TRUE" : "FALSE" );		

			double offset;
			if ( al->offset( offset ) == sSuccess )
				printFile("%sOffset: %g\n",indentSpace(),offset);

			double a;
			if ( al->angle(a) == sSuccess )
				printFile("%sAngle: %g\n",indentSpace(),a);
		}
	}
	else if ( locator->type() == kRadialLocatorType )
	{
		AlRadialLocator *rl = locator->asRadialLocatorPtr();		
		if ( rl != NULL )
		{
			double radius;
			if ( rl->radius( radius ) == sSuccess )
				printFile("%sRadius: %g\n",indentSpace(),radius);
				
			double x,y,z;
			if ( rl->center( x,y,z ) == sSuccess )
				printFile("%sRadius: %g %g %g\n",indentSpace(),x,y,z);
			
			double param;
			if ( rl->parameter( param ) == sSuccess )
				printFile("%sParameter: %g\n",indentSpace(),param);

			double offset;
			if ( rl->offset( offset ) == sSuccess )
				printFile("%sOffset: %g\n",indentSpace(),offset);

			boolean justify;
			if ( rl->leftJustify( justify ) == sSuccess )
				printFile( "%sLeft justify: %s\n",indentSpace(), (justify) ? "TRUE" : "FALSE" );
				
			AlObject *obj = rl->attachedTo();
			if ( obj != NULL )
			{
					printFile( "%sAttached to: %s\n",indentSpace(),obj->name());
					delete obj;
			}	
		}
	}
	else if ( locator->type() == kDeviationLocatorType )
	{
		AlDeviationLocator *dl = locator->asDeviationLocatorPtr();
		if ( dl != NULL )
		{
			double deviation;
			if ( dl->deviation( deviation ) == sSuccess )
				printFile("%sDeviation: %g\n",indentSpace(),deviation);
				
			double x,y,z;
			if ( dl->deviationComponents( x,y,z ) == sSuccess )
				printFile("%sDeviation components: %g %g %g\n",indentSpace(),x,y,z);
			
			double param;
			if ( dl->parameter( param ) == sSuccess )
				printFile("%sParameter: %g\n",indentSpace(),param);

			double length;
			if ( dl->length( length ) == sSuccess )
				printFile("%sLength: %g\n",indentSpace(),length);

			if ( dl->offset(  x,y,z ) == sSuccess )
				printFile("%sOffset: %g %g %g\n",indentSpace(),x,y,z);

			boolean justify;
			if ( dl->leftJustify( justify ) == sSuccess )
				printFile( "%sLeft justify: %s\n",indentSpace(), (justify) ? "TRUE" : "FALSE" );
				
			AlObject *obj, *obj2;
			if ( dl->attachedTo( obj, obj2 ) == sSuccess )
			{
					printFile( "%sAttached to: %s %s\n",indentSpace(),obj->name(), obj2->name());
					delete obj;
					delete obj2;
			}	
		}
	}
	else if ( locator->type() == kMinmaxLocatorType )
	{
		AlMinmaxLocator *ml = locator->asMinmaxLocatorPtr();
		if ( ml != NULL )
		{
			boolean combdisplay;
			if ( ml->combDisplay( combdisplay ) == sSuccess )
				printFile( "%sComb display: %s\n",indentSpace(), (combdisplay) ? "TRUE" : "FALSE" );
		
			double combscale;
			if ( ml->combScale( combscale ) == sSuccess )
				printFile("%sComb scale: %g\n",indentSpace(),combscale);

			double combthreshold;
			if ( ml->combThreshold( combthreshold ) == sSuccess )
				printFile("%sComb threshold: %g\n",indentSpace(),combthreshold);

			double combdensity;
			if ( ml->combDensity( combdensity ) == sSuccess )
				printFile("%sComb density: %g\n",indentSpace(),combdensity);

			double mindist;
			if ( ml->minimumDistance( mindist ) == sSuccess )
				printFile("%sMinimum distance: %g\n",indentSpace(),mindist);

			double maxdist;
			if ( ml->maximumDistance( maxdist ) == sSuccess )
				printFile("%sMaximum distance: %g\n",indentSpace(),maxdist);

			boolean justify_min_dist;
			if ( ml->leftJustifyMinDistance( justify_min_dist ) == sSuccess )
				printFile( "%sLeft justify min distance: %s\n",indentSpace(), (justify_min_dist) ? "TRUE" : "FALSE" );

			boolean justify_max_dist;
			if ( ml->leftJustifyMaxDistance( justify_max_dist ) == sSuccess )
				printFile( "%sLeft justify max distance: %s\n",indentSpace(), (justify_max_dist) ? "TRUE" : "FALSE" );

			AlObject *obj, *obj2;
			if ( ml->attachedTo( obj, obj2 ) == sSuccess )
			{
					printFile( "%sAttached to: %s %s\n",indentSpace(),obj->name(), obj2->name());
					delete obj;
					delete obj2;
			}	
		}
	}
	
	UNINDENT
	
	UNINDENT
	
}

void printAlConstructionEntity( AlConstructionEntity *ce )
{
	if ( ce == NULL ) return;

	printObjectName( ce );
	INDENT
	
	INDENT

	boolean invisible = ce->invisible();
	printFile( "%sInvisible: %s\n",indentSpace(), (invisible) ? "TRUE" : "FALSE" );
		
	boolean picked = ce->isPicked();
	printFile( "%sPicked: %s\n",indentSpace(), (picked) ? "TRUE" : "FALSE" );
		
	AlLayer *l = ce->layer();
	if ( l != NULL )
	{
		int layernum = l->number();
		printFile( "%sLayer num: %d\n",indentSpace(), layernum );
		delete l;
	}

	if ( ce->type() == kConstructionVectorType )
	{
		AlConstructionVector *cev = ce->asConstructionVectorPtr();
		if ( cev != NULL )
		{
			double x,y,z;
			if ( cev->vector( x,y,z ) == sSuccess )
				printFile("%sVector: %g %g %g\n", indentSpace(), x,y,z);

			INDENT
			AlPoint *origin = cev->origin();
			if ( origin != NULL )
			{
				printFile( "%sOrigin:\n",indentSpace());
				printAlConstructionEntity( origin );
				delete origin;
			}

			AlPoint *end = cev->end();
			if ( end != NULL )
			{
				printFile( "%sEnd:\n",indentSpace());
				printAlConstructionEntity( end );
				delete end;
			}
			UNINDENT
		}
	}
	else if ( ce->type() == kConstructionPlaneType )
	{
		AlConstructionPlane *cp = ce->asConstructionPlanePtr();
		if ( cp != NULL )
		{		
			double x,y,z;
			if ( cp->origin( x,y,z ) == sSuccess )
				printFile("%sOrigin: %g %g %g\n", indentSpace(), x,y,z);
				
			if ( cp->scale( x,y ) == sSuccess )
				printFile("%sScale: %g %g\n", indentSpace(), x,y);

			if ( cp->rotate( x,y,z ) == sSuccess )
				printFile("%sRotate: %g %g %g\n", indentSpace(), x,y,z);

			if ( cp->translate( x,y,z ) == sSuccess )
				printFile("%sTranslate: %g %g %g\n", indentSpace(), x,y,z);
				
			double xaxis[3],yaxis[3],zaxis[3];
			if ( cp->axes( xaxis, yaxis, zaxis ) == sSuccess )
			{
				printFile("%sX axis: %g %g %g\n", indentSpace(),xaxis[0], xaxis[1], xaxis[2] );
				printFile("%sY axis: %g %g %g\n", indentSpace(),yaxis[0], yaxis[1], yaxis[2] );
				printFile("%sZ axis: %g %g %g\n", indentSpace(),zaxis[0], zaxis[1], zaxis[2] );			
			}
				
			double matrix[4][4];
			int    i;
			if ( cp->transformationMatrix( matrix ) == sSuccess )
			{
				printFile( "%sLocal Transform is: \n", indentSpace());
				for( i  = 0; i < 4; i ++ ) {
					printFile( "%s%g, %g, %g, %g\n", indentSpace(),
						matrix[i][0], matrix[i][1], matrix[i][2], matrix[i][3]);
				}
			}

			INDENT
			AlPoint *first = cp->first();
			if ( first != NULL )
			{
				printFile( "%sFirst:\n",indentSpace());
				printAlConstructionEntity( first );
				delete first;
			}

			AlPoint *second = cp->second();
			if ( second != NULL )
			{
				printFile( "%sSecond:\n",indentSpace());
				printAlConstructionEntity( second );
				delete second;
			}

			AlPoint *third = cp->third();
			if ( third != NULL )
			{
				printFile( "%sThird:\n",indentSpace());
				printAlConstructionEntity( third );
				delete third;
			}
			UNINDENT
		}
	}
	else if ( ce->type() == kSpacePointType )
	{
		AlSpacePoint *sp = ce->asSpacePointPtr();
		if ( sp != NULL )
		{
			double x,y,z;
			if ( sp->worldPosition( x,y,z ) == sSuccess )
				printFile("%sWorld position: %g %g %g\n", indentSpace(), x,y,z);
		}
	}
	else if ( ce->type() == kCurvePointType )
	{
		AlCurvePoint *cp = ce->asCurvePointPtr();
		if ( cp != NULL )
		{
			double param;
			if ( cp->parameter( param ) == sSuccess )
				printFile("%sParameter: %g\n", indentSpace(), param);

			double x,y,z;
			if ( cp->worldPosition( x,y,z ) == sSuccess )
				printFile("%sWorld position: %g %g %g\n", indentSpace(), x,y,z);

			if ( cp->tangent( x,y,z ) == sSuccess )
				printFile("%sTangent: %g %g %g\n", indentSpace(), x,y,z);

			if ( cp->normal1( x,y,z ) == sSuccess )
				printFile("%sNormal 1: %g %g %g\n", indentSpace(), x,y,z);
				
			if ( cp->normal2( x,y,z ) == sSuccess )
				printFile("%sNormal 2: %g %g %g\n", indentSpace(), x,y,z);
			
			double arclen;
			if ( cp->arcLength( arclen ) == sSuccess )
				printFile("%sArc length: %g\n", indentSpace(), arclen);

			double radius;
			if ( cp->radius( radius ) == sSuccess )
				printFile("%sRadius: %g\n", indentSpace(), radius );
				
			AlCurveNode *curveNode = cp->attachedTo();
			if ( curveNode != NULL )
			{
					printFile( "%sAttached to: %s\n",indentSpace(),curveNode->name());
					delete curveNode;
			}
		}
	}
	else if ( ce->type() == kSurfacePointType )
	{
		AlSurfacePoint *sp = ce->asSurfacePointPtr();
		if ( sp != NULL )
		{
			double param1, param2;
			if ( sp->parameter( param1, param2 ) == sSuccess )
				printFile("%sParameters: %g %g\n", indentSpace(), param1, param2);

			double x,y,z;
			if ( sp->worldPosition( x,y,z ) == sSuccess )
				printFile("%sWorld position: %g %g %g\n", indentSpace(), x,y,z);

			if ( sp->uTangent( x,y,z ) == sSuccess )
				printFile("%sU tangent: %g %g %g\n", indentSpace(), x,y,z);

			if ( sp->vTangent( x,y,z ) == sSuccess )
				printFile("%sV tangent: %g %g %g\n", indentSpace(), x,y,z);

			if ( sp->normal( x,y,z ) == sSuccess )
				printFile("%sNormal: %g %g %g\n", indentSpace(), x,y,z);
				
			double meanradius;
			if ( sp->meanRadius( meanradius ) == sSuccess )
				printFile("%sMean radius: %g\n", indentSpace(), meanradius );

			double gaussradius;
			if ( sp->gaussianRadius( gaussradius ) == sSuccess )
				printFile("%sGaussian radius: %g\n", indentSpace(), gaussradius );

			double principalminradius;
			if ( sp->principalMinimumRadius( principalminradius ) == sSuccess )
				printFile("%sPrincipal minimum radius: %g\n", indentSpace(), principalminradius );

			double principalmaxradius;
			if ( sp->principalMaximumRadius( principalmaxradius ) == sSuccess )
				printFile("%sPrincipal maximum radius: %g\n", indentSpace(), principalmaxradius );

			double uarclength;
			if ( sp->uArcLength( uarclength ) == sSuccess )
				printFile("%sU arc length: %g\n", indentSpace(), uarclength );

			double varclength;
			if ( sp->vArcLength( varclength ) == sSuccess )
				printFile("%sV arc length: %g\n", indentSpace(), varclength );

			AlSurfaceNode *surfaceNode = sp->attachedTo();
			if ( surfaceNode != NULL )
			{
					printFile( "%sAttached to: %s\n",indentSpace(),surfaceNode->name());
					delete surfaceNode;
			}
		}
	}
	else if ( ce->type() == kCurveOnSurfacePointType )
	{
		AlCurveOnSurfacePoint *cosp = ce->asCurveOnSurfacePointPtr();
		if ( cosp != NULL )
		{
			double param;
			if ( cosp->parameter( param ) == sSuccess )
				printFile("%sParameter: %g\n", indentSpace(), param);

			double x,y,z;
			if ( cosp->worldPosition( x,y,z ) == sSuccess )
				printFile("%sWorld position: %g %g %g\n", indentSpace(), x,y,z);

			if ( cosp->tangent( x,y,z ) == sSuccess )
				printFile("%sTangent: %g %g %g\n", indentSpace(), x,y,z);

			if ( cosp->normal1( x,y,z ) == sSuccess )
				printFile("%sNormal 1: %g %g %g\n", indentSpace(), x,y,z);
				
			if ( cosp->normal2( x,y,z ) == sSuccess )
				printFile("%sNormal 2: %g %g %g\n", indentSpace(), x,y,z);
			
			double arclen;
			if ( cosp->arcLength( arclen ) == sSuccess )
				printFile("%sArc length: %g\n", indentSpace(), arclen);

			AlCurveOnSurface *cos = cosp->attachedTo();
			if ( cos != NULL )
			{
					printFile( "%sAttached to: %s\n",indentSpace(),cos->name());
					delete cos;
			}
		}
	}
	
	UNINDENT
	
	UNINDENT

}

void printAlLayer( AlLayer *ly )
{
	if ( ly == NULL ) return;

	printObjectName( ly );
	INDENT
	
	INDENT

	int layernum = ly->number();
	printFile( "%sLayer number: %d\n",indentSpace(),layernum);
	
	boolean invisible = ly->invisible();
	printFile( "%sInvisible: %s\n",indentSpace(), (invisible) ? "TRUE" : "FALSE" );
	
	 AlLayer::AlLayerPickType pt;
	 if ( ly->pickability( pt ) == sSuccess )
			printFile( "%sPick type: %s\n",indentSpace(), 
				( pt == AlLayer::kLayerPickable ) ? "kLayerPickable" : 
					(
						( pt == AlLayer::kLayerReference ) ? "kLayerReference" :
							(
								( pt == AlLayer::kLayerInactive ) ? "kLayerInactive" :  "Unknown" 
							)
					)
				);
	
	boolean pb = ly->playback();
	printFile( "%sPlay Back: %s\n",indentSpace(), (pb) ? "TRUE" : "FALSE" );

	boolean di = ly->drawInstances();
	printFile( "%sDraw Instances: %s\n",indentSpace(), (di) ? "TRUE" : "FALSE" );
		
	boolean v = ly->visibleInLayerBar();
	printFile( "%sVisible in layer bar: %s\n",indentSpace(), (v) ? "TRUE" : "FALSE" );

	int c = ly->color();
	printFile( "%sColor: %d\n",indentSpace(), c);

	boolean picked = ly->isPicked();
	printFile( "%sPicked: %s\n",indentSpace(), (picked) ? "TRUE" : "FALSE" );
	
	// Symmetry
	boolean symOn = ly->isSymmetric();
	printFile( "%sSymmetric: %s\n",indentSpace(), (symOn) ? "TRUE" : "FALSE" );
	if ( symOn )
	{
		double x,y,z;
		if ( ly->symmetricOrigin( x, y, z ) == sSuccess )
			printFile( "%sSymmetric origin: %g %g %g\n",indentSpace(),x,y,z);
			
		if ( ly->symmetricNormal( x, y, z ) == sSuccess )
			printFile( "%sSymmetric normal: %g %g %g\n",indentSpace(),x,y,z);		
	}


	UNINDENT
	
	UNINDENT
}

void printBool( const char *string, boolean state )
{
	printFile( "%s%s: %s\n", indentSpace(), string, state ? "ON" : "OFF" );
}
void printInt( const char *string, int value )
{
	printFile( "%s%s: %d\n", indentSpace(), string, value );
}
void printDouble( const char *string, double value )
{
	printFile( "%s%s: %f\n", indentSpace(), string, value );
}

void printString( const char *string, const char *value )
{
	printFile( "%s%s: %s\n", indentSpace(), string, value );
}
void printRGB( const char *string,  double r, double g, double b)
{
	printFile( "%s%s: (%f, %f, %f)\n", indentSpace(), string, r,g,b );
}
const char *toString( AlRender::Source source )
{
	switch( source )
	{
		case AlRender::kAll:	return "All";
		case AlRender::kActive:	return "Active";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::Subdivision::Type type )
{
	switch( type )
	{
		case AlRender::Subdivision::kAdaptive:	return "Adaptive";
		case AlRender::Subdivision::kUniform:	return "Uniform";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::AnimationRange range )
{
	switch( range )
	{
		case AlRender::kGlobalRange:	return "Global";
		case AlRender::kMinMax:			return "MinMax";
		case AlRender::kTimeSlider:		return "TimeSlider";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::Scope scope )
{
	switch( scope )
	{
		case AlRender::kPerObject:	return "PerObject";
		case AlRender::kGlobal:		return "Global";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::Quality quality )
{
	switch( quality )
	{
		case AlRender::kLow:		return "Low";
		case AlRender::kMedium:		return "Medium";
		case AlRender::kHigh:		return "High";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::Misc::GeometrySource source )
{
	switch( source )
	{
		case AlRender::Misc::kModeler:		return "Modeler";
		case AlRender::Misc::kSDLFile:		return "SDLFile";
		default:				return "Unknown";
}
}

const char *toString( AlRender::Image::Format format )
{
	switch( format )
	{
		case AlRender::Image::kAlias:	return "Alias";
		case AlRender::Image::kSGI:		return "SGI";
		case AlRender::Image::kTIFF:	return "TIFF";
		case AlRender::Image::kTIFF16:	return "TIFF16";
		case AlRender::Image::kRLA:		return "RLA";
		case AlRender::Image::kFIDO:	return "FIDO";
		case AlRender::Image::kHARRY:	return "HARRY";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::Image::DepthFormat depthFormat )
{
	switch( depthFormat )
	{
		case AlRender::Image::kDepthAlias:		return "Alias";
		case AlRender::Image::kDepthComposer:	return "Composer";
		default:				return "Unknown";
	}
}

const char *toString( AlRender::Image::Fields fields )
{
	switch( fields )
	{
		case AlRender::Image::kOff:		return "Off";
		case AlRender::Image::kEven:	return "Even";
		case AlRender::Image::kOdd:		return "Odd";
		case AlRender::Image::kBoth:	return "Both";
		default:				return "Unknown";
	}
}

void printLimits( AlRender::Quality q )
{
	printFile( "%sSettings for quality %s:\n", indentSpace(), toString( q ) );
	INDENT

	printString( "qualityScope", toString( AlRender::qualityScope( q )));

	printFile( "%sSubdivision\n", indentSpace());
	INDENT
		printFile( "%stype: %s\n", indentSpace(), toString( AlRender::Subdivision::type(q)) );
		INDENT
			printFile( "%sUniform\n", indentSpace() );
			printInt( "uniformU", AlRender::Subdivision::uniformU(q));
			printInt( "uniformV", AlRender::Subdivision::uniformV(q));
		UNINDENT

		printFile( "%sAdaptive params\n", indentSpace() );
		INDENT
			printInt( "Min", AlRender::Subdivision::adaptiveMin(q));
			printInt( "Max", AlRender::Subdivision::adaptiveMax(q));
			printDouble( "Threshold", AlRender::Subdivision::adaptiveThreshold(q));
		UNINDENT
	UNINDENT

	printFile( "%sAntiAlias\n", indentSpace() );
	INDENT
		printInt( "min", AlRender::AntiAlias::min(q) );
		printInt( "max", AlRender::AntiAlias::max(q) );
		printDouble( "threshold", AlRender::AntiAlias::threshold(q));
		printBool( "jitter", AlRender::AntiAlias::jitter(q));
	UNINDENT

	printFile( "%sRaytracing Limits\n", indentSpace() );
	INDENT
		printBool( "createPreview",	AlRender::Limits::createPreview(q));
		printInt( "maxReflections",	AlRender::Limits::maxReflections(q));
		printInt( "maxRefractions",	AlRender::Limits::maxRefractions(q));
		printInt( "maxShadowLevels",	AlRender::Limits::maxShadowLevels(q));
		printInt( "subdivideRecurse",	AlRender::Limits::subdivideRecurse());
	UNINDENT

	UNINDENT
}

void printAllRenderGlobals()
{
	INDENT
	printFile( "%sMain settings\n", indentSpace() );
	INDENT
		printString( "render Source", toString( AlRender::render() ));
		printBool(	 "animation",	AlRender::animation() );
		printString( "animationRange", toString( AlRender::animationRange() ));
		printString( "hiddenLineParms", toString( AlRender::hiddenLineParms()));
		printString( "qualityType", toString( AlRender::qualityType()));
		printBool(   "modifyExtensions", AlRender::modifyExtensions() );
		printInt(    "extensionPadding", AlRender::extensionPadding() );
		printInt(    "startExtension", AlRender::startExtension() );
		printInt(    "byExtension", AlRender::byExtension() );
	UNINDENT

	printLimits( AlRender::kLow );
	printLimits( AlRender::kMedium );
	printLimits( AlRender::kHigh );

	printFile(	"%sComposite settings\n", indentSpace() );
	INDENT
		printBool(	"enabled",  AlRender::Composite::enabled() );
		printDouble("coverageThreshold", AlRender::Composite::coverageThreshold());
	UNINDENT

	printFile(	"%sBlur settings\n", indentSpace() );
	INDENT
		printBool(	"postFilter",	AlRender::Blur::postFilter());
		printInt(	"postCenter",	AlRender::Blur::postCenter());
		printInt(	"postAdjacent",	AlRender::Blur::postAdjacent());
		printInt(	"postDiagonal",	AlRender::Blur::postDiagonal());
		printBool(	"motionBlur",	AlRender::Blur::motionBlur());
		printDouble( "shutterAngle",AlRender::Blur::shutterAngle());
	UNINDENT

	printFile( "%sMisc settings\n", indentSpace() );
	INDENT
		printString("geometrySource", toString(AlRender::Misc::geometrySource()));
		printBool(	"textures", AlRender::Misc::textures());
		printBool(	"skeletons", AlRender::Misc::skeletons());
		printBool(	"noTransparencyInDepth", AlRender::Misc::noTransparencyInDepth());
		printDouble("transparency",AlRender::Misc::transparency());
		printBool(	"keepNonglowedImage", AlRender::Misc::keepNonglowedImage());
		printBool(	"spotLightDepthMaps", AlRender::Misc::spotLightDepthMaps());
		printBool(	"depthMapsInMemory", AlRender::Misc::depthMapsInMemory());
		printBool(	"attenuateTransparency", AlRender::Misc::attenuateTransparency());
	UNINDENT

	printFile(	"%sImage settings\n", indentSpace() );
	INDENT
		printString("format",		toString( AlRender::Image::format()));
		printString("depthFormat",	toString( AlRender::Image::depthFormat()));
		printString("fields",		toString( AlRender::Image::fields()));
		printBool(	"oddFirst",			AlRender::Image::oddFirst());
		printBool(	"ignoreFilmGate",	AlRender::Image::ignoreFilmGate());
		printDouble("gammaCorrection",	AlRender::Image::gammaCorrection());
		printBool(	"XYRatioLock",		AlRender::Image::XYRatioLock());
		printInt(	"resolutionX",		AlRender::Image::resolutionX());
		printInt(	"resolutionY",		AlRender::Image::resolutionY());
		printDouble("aspectRatio",		AlRender::Image::aspectRatio());
	UNINDENT

	AlList* list;
	if( sSuccess == AlRender::Image::readResolutions( list ))
	{
		printFile( "%sResolutions\n", indentSpace() );
		INDENT
			for( AlResolutionItem *item = (AlResolutionItem *) list->first();
				item; item = item->nextItem() )
			{
				printFile( "%s\"%s\" (%d,%d) 1:%f\n",  indentSpace(),
					item->name, item->x, item->y, item->aspect );
			}
		UNINDENT
		delete list;
	}

	printFile(	"%sHiddenLine settings\n", indentSpace() );
	INDENT
		double r,g,b;
		printBool(	"useFillColor",		AlRender::HiddenLine::useFillColor());
		AlRender::HiddenLine::fillColor(r,g,b); printRGB( "fillColor", r,g,b );
		AlRender::HiddenLine::lineColor(r,g,b); printRGB( "lineColor", r,g,b );
		printInt(	"patchLinesU",		AlRender::HiddenLine::patchLinesU());
		printInt(	"patchLinesV",		AlRender::HiddenLine::patchLinesV());
	UNINDENT

	printFile(	"%sParticles\n", indentSpace() );
	INDENT
		printBool(	"shown", AlRender::Particle::show() );
		printInt(	"simulationSubsteps", AlRender::Particle::simulationSubsteps());
		printInt(	"framesPerSecond", AlRender::Particle::framesPerSecond());
	UNINDENT

	UNINDENT
}

void printAl_AllProcess()
{
	// Print redirection should be set by the time this
	// function is called.
	
	printFile( "----------------------------------------- OBJECTS\n" );
	AlObject *first = AlUniverse::firstDagNode();
	AlTM	tm = AlTM::identity();
	printAllAlObjects( first, tm );
	delete first;

	printFile( "-------------------------------------------- SETS\n" );
	printAllSets();

	printFile( "---------------------------------------- CHANNELS\n" );
	printAllChannels();

	printFile( "----------------------------------------- ACTIONS\n" );
	printAllActions();

	printFile( "------------------------------------------ CLOUDS\n" );
	printAllClouds();

	printFile( "------------------------------------------ BLEND CURVES\n" );
	printAllBlendCurves();

	printFile( "------------------------------------------ LOCATORS\n" );
	printAllLocators();

	printFile( "------------------------------------------ CONSTRUCTION ENTITIES\n" );
	printAllConstructionEntities();

	printFile( "------------------------------------------ LAYERS\n" );
	printAllLayers();

	printFile( "----------------------------------------- SHADING\n" );
	printAllShading();

	printFile( "---------------------------------- RENDER GLOBALS\n" );
	printAllRenderGlobals();

	printFile( "-------------------------------------------------\n" );
}

void printAl_All( AlOutputType ot )
{
	setPrintFile( ot );

	printAl_AllProcess();
	
	endPrintFile();
}

void printAl_AllToFile( Xconst char *outputFileName, Xconst char *comment )
{
	if ( outputFileName == NULL )
	{
		setPrintFile( kStdout );
		printFile( "Warning: printAl_AllToFile() called with a NULL output filename.\n");
	}
	else
	{
		setPrintFile( outputFileName );
	}
	
	if ( comment != NULL )
		printFile("FILE COMMENT: <%s>\n",comment);
		
	printAl_AllProcess();
	
	endPrintFile();
}


const char*	determineObjectClassName( AlObjectType type )
{
	switch ( type )
	{
	  case kAmbientLightType:		return "AlAmbientLight";
	  case kAreaLightType:	 		return "AlAreaLight";
	  case kCameraEyeType:
	  case kCameraViewType:
	  case kCameraUpType:			return "AlCameraNode";
	  case kCameraType:				return "AlCamera";
	  case kPerspectiveCameraType:	return "AlPerspectiveCamera";
	  case kOrthographicCameraType:	return "AlOrthographicCamera";
	  case kCloudType:				return "AlCloud";
	  case kClusterType:			return "AlCluster";
	  case kClusterNodeType:		return "AlClusterNode";
	  case kClusterMemberType:		return "AlClusterMember";
	  case kCurveNodeType:			return "AlCurveNode";
	  case kCurveType:				return "AlCurve";
	  case kCurveCVType:			return "AlCurveCV";
	  case kCurveOnSurfaceType:		return "AlCurveOnSurface";
	  case kDagNodeType:			return "AlDagNode";
	  case kDirectionLightType:		return "AlDirectionLight";
	  case kFaceNodeType:			return "AlFaceNode";
	  case kFaceType:				return "AlFace";
	  case kGroupNodeType:			return "AlGroupNode";
	  case kLightLookAtNodeType:	return "AlLightLookAtNode";
	  case kLightNodeType:			return "AlLightNode";
	  case kLightType:				return "AlLight";
	  case kLightUpNodeType:		return "AlLightUpNode";
	  case kLinearLightType:		return "AlLinearLight";
	  case kNonAmbientLightType:	return "AlNonAmbientLight";
	  case kPointLightType:			return "AlPointLight";
	  case kSpotLightType:			return "AlSpotLight";
	  case kShellNodeType:			return "AlShellNode";
	  case kShellType:				return "AlShell";
	  case kSurfaceNodeType:		return "AlSurfaceNode";
	  case kSurfaceType:			return "AlSurface";
	  case kSurfaceCVType:			return "AlSurfaceCV";
	  case kSetType:				return "AlSet";
	  case kSetMemberType:			return "AlSetMember";
	  case kChannelType:			return "AlChannel";
	  case kActionType:				return "AlAction";
	  case kMotionActionType:		return "AlMotionAction";
	  case kParamActionType:		return "AlParamAction";
	  case kKeyframeType:			return "AlKeyframe";
	  case kStreamType:				return "AlStream";
	  case kShaderType:				return "AlShader";
	  case kTextureType:			return "AlTexture";
	  case kEnvironmentType:		return "AlEnvironment";
	  case kPolysetNodeType:		return "AlPolysetNode";
	  case kPolysetType:			return "AlPolyset";
	  case kPolygonType:			return "AlPolygon";
	  case kPolysetVertexType:		return "AlPolysetVertex";
      case kMeshType:               return "AlMesh";
      case kMeshNodeType:           return "AlMeshNode";
	  case kAttributeType:			return "AlAttribute";
	  case kArcAttributeType:		return "AlArcAttribute";
	  case kLineAttributeType:		return "AlLineAttribute";
	  case kCurveAttributeType:		return "AlCurveAttribute";
	  case kPlaneAttributeType:		return "AlPlaneAttribute";
	  case kConicAttributeType:		return "AlConicAttribute";
	  case kRevSurfAttributeType:	return "AlRevSurfAttribute";
	  case kJointType:			    return "AlJoint";
	  case kConstraintType:			return "AlConstraint";
	  case kPointConstraintType:	return "AlPointConstraint";
	  case kOrientationConstraintType:	return "AlOrientationConstraint";
	  case kAimConstraintType:		return "AlAimConstraint";
	  case kIKHandleType: 			return "AlIKHandle";
	  case kTextureNodeType:		return "AlTextureNode";
	  case kBlendCurveType:			return "AlBlendCurve";
	  case kBlendPointType:			return "AlBlendPoint";
	  case kLocatorType:			return "AlLocator";
	  case kAnnotationLocatorType:	return "AlAnnotationLocator";
	  case kDistanceLocatorType:	return "AlDistanceLocator";
	  case kAngleLocatorType:		return "AlAngleLocator";
	  case kRadialLocatorType:		return "AlRadialLocator";
	  case kDeviationLocatorType:	return "AlDeviationLocator";
	  case kMinmaxLocatorType:		return "AlMinMaxLocator";
	  case kConstructionEntityType:	return "AlConstructionEntity";
	  case kConstructionVectorType:	return "AlConstructionVector";
	  case kConstructionPlaneType:	return "AlConstructionPlane";
	  case kPointType:				return "AlPoint";
	  case kSpacePointType:			return "AlSpacePoint";
	  case kCurvePointType:			return "AlCurvePoint";
	  case kSurfacePointType:		return "AlSurfacePoint";
	  case kCurveOnSurfacePointType:return "AlCurveOnSurfacePoint";
	  case kLayerType:				return "AlLayer";

	  default:						return "unknownClassName";
	}
}

const char *determineFormTypeName( curveFormType form )
{
	switch( form )
	{
		case kOpen:		return "open";
		case kClosed:	return "closed";
		case kPeriodic:	return "periodic";
		default:		return "unknown";
	}
}

