#%
# \page _studio_::StudioScripting::ScriptingApi Interface to Scriptable Studio Functionality
# \p This script provides the interface to Studio functionality that is scriptable.

use AutoTest;
my $AUTO_TEST_LOG = $ENV{AUTO_TEST_LOG};
print "Running: basisOperations.pl\n";

# For timing
use Benchmark;
my $AUTO_TEST_TIMER1;
my $AUTO_TEST_TIMER2;

# THROW_EXCEPTION is global flag decide whether throw exception or not if socket server return status is 0
# By default we set to 1 to throw the exception so that QA can easy identify the error line number
# -- Michael Wei 2018.7.5
my $THROW_EXCEPTION=1;
sub throwExceptionIfFlagOn
{
	if ($THROW_EXCEPTION==1)
		{
			my $func_name = (caller(1))[3];
			die "Error: $func_name(@_) return 0, expect 1";
		}		
}

#
# UI supporting functions
#

my $btInfoPrint=1;
my $indentLevel=0;


sub setBtInfoOn
{
	my ($x) = @_;
	$btInfoPrint = $x;
}

my $FirstPass=1;
sub printBtMsg
{
	my ($arg)=@_;

	if ( $btInfoPrint== 1 )
	{
		my $identspace="\t"x$indentLevel;
		print "$identspace$arg\n";
		if ($AUTO_TEST_LOG)
		{	
			my $a="tmp.log";
			my $b="tmp2.log";

			system( "echo $identspace$arg >$a");
			if ($FirstPass == 1)
			{
				$FirstPass = 0;
				system( "cat $a > $AUTO_TEST_LOG" );
			}
			else 
			{
				system( "cat $AUTO_TEST_LOG $a > $b" );
				system( "mv $b $AUTO_TEST_LOG" );
			}
		}	
	}
}

sub INDENT
{
	$indentLevel++;
}

sub UNINDENT
{
	$indentLevel--;
	if ( $indentLevel < 0 )
	{
		$indentLevel=0;
	}
}

#%
# \fn debugPrint($msg);
# Sends a string to the Alias debug Window from Perl.
#
sub debugPrint
{
	my ($msg) = @_;
	AutoTest::debugPrint($msg);
}

#%
# \fn retrieveFile( $fileName )
# Retrieves the specified wire file.  On
# success, 1 is returned.  Otherwise 0 is
# returned.
#
sub retrieveFile
{	
	my ($fName) = @_;
	
	printBtMsg "retrieveFile: $fName";
	
	$status =  AutoTest::retrieveWireFile( $fName );
	if ( $status == 0 )
	{		
		printBtMsg "retrieveFile: AlUniverse_retrieve returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn storeFile( $fileName )
# Save entire scene to a wire file $fileName. On
# success, 1 is returned.  Otherwise 0 is
# returned.
#
sub storeFile
{
	my ( $fName ) = @_;
	printBtMsg "storeFile: \"$fName\"";
	
	$status = AutoTest::storeWireFile( $fName, undef );
	if ( $status == 0 )
	{
		printBtMsg "storeFile: AlUniverse_store returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn storeSDL( $fileName )
# Exports scene to SDL file $fileName. On
# success, 1 is returned.  Otherwise 0 is
# returned.
#
sub storeSDL
{
	my ( $fName  ) = @_;
	printBtMsg "storeSDL: \"$fName\"";

	# Export->SDL appears to use AlUniverse::writeSDLNoAnimation, so I
	# will do the same.
	$status = AutoTest::storeSdlFile( $fName, 0 );
	if ( $status == 0 )
	{
		printBtMsg "storeSDL: AlUniverse_writeSDLNoAnimation returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn redrawScreen(void)
# Redraw the Alias screen.
#
sub redrawScreen
{
	AutoTest::redrawScreen();
}

#%
# \fn flushScreen(void)
# Flush the screen -- use this routine with caution.
#
sub flushScreen
{
	AutoTest::flushScreen();
}

#%
# \fn systemCommand( $command )
# Provides system() functionality. On
# success, 1 is returned.  Otherwise 0 is
# returned.
#
sub systemCommand
{
	my ( @args ) = @_;
	printBtMsg "systemCommand: \@args";
	
	if ( system( @args ) != 0 )
	{
		printBtMsg "systemCommand: failed: system \@args";
		return 1;
	}
	
	throwExceptionIfFlagOn @_;
	return 0;
}

# Utility function for "walkDirectory".
sub walkDirectory2
{
	my ( $dirName, $recursionLevel ) = @_;
	if ( $recursionLevel < 0 )
	{
		throwExceptionIfFlagOn @_;
		return 0;
	}
	
	my ($file, @allfiles);

	opendir DIR, $dirName or return;
	@allfiles = readdir DIR;
	foreach $file (@allfiles)
	{
		if ( $file ne "." && $file ne ".." )
		{
			if ( -d "$dirName\/$file" )
			{
				printBtMsg "$dirName\/$file/ ";
				walkDirectory2( "$dirName\/$file", $recursionLevel -1 );
			} 
			else 
			{
				printBtMsg "$dirName\/$file ";
			}
		}
	}
	closedir DIR;
	
	throwExceptionIfFlagOn @_;
	return 0;
}


#%
# \fn walkDirectory( $directoryName $recursionLevel )
# Given a directory, this function walks the directory and its
# sub-directories down to the recursion level specified.  The names
# of files will be printed out.  This function serves as an example
# for others to use.
#
sub walkDirectory
{
	my ( $dirName, $recursionLevel ) = @_;
	logTraceCall ( "walkDirectory \"$dirName\" $recursionLevel" );
	printBtMsg "walkDirectory: \"$dirName\" $recursionLevel";

	return walkDirectory2( $dirName, $recursionLevel );
}


#^
# \fn runScripts( $dirName, $scriptNameMatches, $recursionLevel )
# Recursively steps through directories and sources scripts that match the
# regular expression $scriptNameMatches.
# Example:
#    runScripts $SCRIPTS_DIR {polyset.*\.pl} 4
#
sub runScripts
{
	my ( $dirName, $scriptNameMatches, $recursionLevel ) = @_;
	logTraceCall ( "runScripts \"$dirName\" \"$scriptNameMatches\" $recursionLevel" );
	printBtMsg "runScripts: \"$dirName\" \"$scriptNameMatches\" $recursionLevel";

	if ( $recursionLevel < 0 )
	{
		throwExceptionIfFlagOn @_;
		return 0;
	}

	my ($file, @allfiles);
	
	opendir DIR, $dirName or return;
	@allfiles = readdir DIR;
	foreach $file (@allfiles)
	{
		if ( $file ne "." && $file ne ".." && ( -d "$dirName\/$file" )  ) 
		{
			runScripts( "$dirName\/$file", "$scriptNameMatches", ( $recursionLevel -1 ) );
		}
		if ( $file =~ /^$scriptNameMatches/ ) 
		{
			require "$dirName\/$file";
		}
	}
	closedir DIR;
	throwExceptionIfFlagOn @_;
		return 0;
}


#%
# \fn pickObjectByName( $name )
# Pick object by name $name.
#
sub pickObjectByName
{
	my ( $name ) = @_;
	printBtMsg "pickObjectByName \"$name\"";

	my $result = AutoTest::pickObjectByName( $name );

	if ( $result == 0 )
	{		
		printBtMsg( "pickObjectByName failed: @_" );
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;

}



#%
# \fn loadShaderByName( $shaderName )
# Loads the specified shader from file.
#
sub loadShaderByName
{
	my ( $shaderName ) = @_;
	AutoTest::loadShaderByName( $shaderName );
}

#%
# \fn setTestName( $name )
# Sets the name of the test to be associated with the script in the logs
#
sub setTestName
{
	my ( $name ) = @_;
	printBtMsg "setTestName \"$name\"";

	return AutoTest::setTestName( $name );
}

# Retrieve the current all packets file
sub retrieveAllPacketsFile
{
        my $PROJECT = $ENV{PROJECT};
	if ( $PROJECT )
	{
	        $apfile = "$PROJECT/DataTransfer/allPackets/95_allpackets.wire";
		if ( $^O eq "MSWin32" )
		{
			$apfile = "$PROJECT\\DataTransfer\\allPackets\\95_allpackets.wire";
		}		
	}
	else
        {
	        $apfile = "/usr/repo/io/src/DataTransfer/allPackets/95_allpackets.wire";
		if ( $^O eq "MSWin32" )
		{
			$apfile = "D:\\mainline\\studio\\src\\DataTransfer\\allPackets\\95_allpackets.wire";
		}
	}
	return retrieveFile( $apfile );	
}

#%
# \fn zUpInUse()
# Returns 1 if the coordinate system if kZup.  0 is returned otherwise.
#
sub zUpInUse
{
	my $cs = AutoTest::zUpUniverse();
	if ( $cs )
	{
		return 1;
	}
	throwExceptionIfFlagOn @_;
	return 0;
}

# Wrapper for print
$verbose="true";
sub logMsg
{
	my ($msg) = @_;

	if ( $verbose eq "true" )
	{
		printBtMsg "$msg";
	}
}

#%
# \fn dumpSymbols( $functionName )
# Prints the Alias symbols associated with a function $functionName.
#
sub dumpSymbols 
{
	my ( $funcName ) = @_;

	printBtMsg( "dumpSymbols $funcName" );
	
	AutoTest::dumpSymbols( $funcName );
	return 1;
}

#%
# \fn setBrushColor( $r, $g, $b )
# Sets the current Brush color for the Paint tools.
#
sub setBrushColor 
{
	my ( $r, $g, $b ) = @_;

	printBtMsg( "setBrushColor $r, $g, $b" );
	
	AutoTest::setBrushColor( $r, $g, $b );
	return 1;
}

#%
# \fn exitStudio( )
# Causes Studio to exit, should be called only at the end of a script
#
sub exitStudio 
{
	AutoTest::exitStudio();
	return 1;
}

#%
# \fn saveFrameBuffer( $filename, $format )
# Saves the current contents of the Alias frame buffer to a file in the 
# AutoTest's current working directory.
#
sub saveFrameBuffer 
{
	my ( $filename, $format ) = @_;
	AutoTest::saveFrameBuffer( $filename, $format );
	return 1;
}

#%
# \fn imageDiff($image1, $image2, $histogram, $tolerance)
#		Does an image comparision between image1 and image2.  If pixels are found that differ by more
#		than the given tolerence then a histogram is saved at the specified location and errors are logged
#		to the AutoTest HTML records.  It is important to note that this funtion is intented to be run during 
#		Automated testing after an image has been exported.  The funtion uses Studio's current background color
#		as a baseline for the histogram and as such if this background color differs from the one in the image then 
#		the histogram results will be inaccurate.  
#

sub imageDiff 
{
	my ( $image1, $image2, $histogram, $tolerance ) = @_;
	AutoTest::imageDiff( $image1, $image2, $histogram, $tolerance);
	return 1;
}

#%
# \fn performanceCheckToggleShade( $numredraws, $estimatedtime, $tolerence )
#		Turns on toggle shade and then times how long it takes to complete a given number of redraws.  If the
#		total time is more than the estimated time by a factor greater than the given tolerence then errors
#		are recorded to the AutoTest HTML files.  Important:  This function WILL BE machine dependent.
#
sub performanceCheckToggleShade
{
	my ( $numredraws, $estimatedtime, $tolerence ) = @_;
	AutoTest::performanceCheckToggleShade( $numredraws, $estimatedtime, $tolerence );
	return 1;
}

#%
# \fn performanceCheckLoadTime( $wirefile, $estimatedtime, $tolerence )
#		Times the loading of a specified wire file.  If the
#		total time is more than the estimated time by a factor greater than the given tolerence then errors
#		are recorded to the AutoTest HTML files.  Important:  This function WILL BE machine dependent.
#
sub performanceCheckLoadTime
{
	my ( $wirefile, $estimatedtime, $tolerence ) = @_;
	AutoTest::performanceCheckLoadTime( $wirefile, $estimatedtime, $tolerence );
	return 1;
}

#%
# \fn dragDropFileOpen( $filename )
# Simulates opening a file via drag and drop in Windows
# 
#
sub dragDropFileOpen
{
	my ( $filename ) = @_;
	AutoTest::dragDropFileOpen( $filename );
	return 1;
}

#%
# \fn int verifyCurrentStateVsFile( $filename )
# Wire file verification
# 
#
sub verifyCurrentStateVsFile
{
	my ( $filename ) = @_;
	my $status = AutoTest::verifyCurrentStateVsFile( $filename );
	if ( $status == 0 )
	{		
		#autobot will check the result log to ignore some kinds of errors, so won't throw exception here
		#throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn void verifyModelCheckVsFile( $filename )
# Model Check file verification
# 
#
sub verifyModelCheckVsFile
{
	my ( $filename ) = @_;
	AutoTest::verifyModelCheckVsFile( $filename );
	return;
}

#%
# \fn void createModelCheckFile( $filename )
# Creates a Model Check file for verification
# 
#
sub createModelCheckFile
{
	my ( $filename ) = @_;
	AutoTest::createModelCheckFile( $filename );
	return;
}


#%
# \fn stringSymbol( $funcName, $symName )
# Gets an Alias symbol associated with a function.
# Function returns -1 if it fails.
#
sub stringSymbol
{
	my ( $funcName, $symName ) = @_;

	printBtMsg( "getStringSymbol $funcName $symName" );
	
	return AutoTest::stringSymbol( $funcName, $symName );
}

#%
# \fn setStringSymbol( $funcName, $symName, $val)
# Set an Alias symbol $symName associated with a 
# function $funcName to $val. On success 1 is
# returned.  O is returned otherwise.
# 
sub setStringSymbol
{
	my ( $funcName, $symName, $val) = @_;

	printBtMsg( "setStringSymbol $funcName $symName $val" );
	
	$result = AutoTest::setStringSymbol( $funcName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setStringSymbol failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn intSymbol( $funcName, $symName )
# Gets an Alias symbol associated with a function.
# Function returns -1 if it fails.
#
sub intSymbol
{
	my ( $funcName, $symName ) = @_;

	printBtMsg( "getIntSymbol $funcName $symName" );
	
	return AutoTest::intSymbol( $funcName, $symName );
}

#%
# \fn setIntSymbol( $funcName, $symName, $val)
# Set an Alias symbol $symName associated with a 
# function $funcName to $val. On success 1 is
# returned.  O is returned otherwise.
# 
sub setIntSymbol
{
	my ( $funcName, $symName, $val) = @_;

	printBtMsg( "setIntSymbol $funcName $symName $val" );
	
	$result = AutoTest::setIntSymbol( $funcName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setIntSymbol failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn floatSymbol( $funcName, $symName )
# Gets an Alias symbol $symName associated with a 
# function $funcName. Function returns -1 if it fails.
#
sub floatSymbol
{
	my ( $funcName, $symName ) = @_;

	printBtMsg( "getFloatSymbol $funcName $symName" );
	
	return AutoTest::floatSymbol( $funcName, $symName );
}

#%
# \fn setFloatSymbol( $funcName, $symName, $val )
# Set an Alias symbol $symName associated with a 
# function $funcName to value $val.  On success
# 1 is returned, otherwise 0 is returned.
#
sub setFloatSymbol
{
	my ( $funcName, $symName, $val) = @_;

	printBtMsg( "setFloatSymbol $funcName $symName $val" );
	
	$result = AutoTest::setFloatSymbol( $funcName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setFloatSymbol failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}


#Fix Sug #224047
#%
# \fn setShaderBoolAttribute( $objectName, $symName, $val)
#  Set a bool value( $val ) to a shader field of shader( $objectName )
#  On success 1 is returned, otherwise 0 is returned.
#
#
sub setShaderBoolAttribute
{
	my ( $objectName, $symName, $val) = @_;

	printBtMsg( "setShaderBoolAttribute $objectName $symName $val" );
	
	$result = AutoTest::setShaderBoolAttribute( $objectName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setShaderBoolAttribute failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn setShaderIntAttribute( $objectName, $symName, $val)
#  Set a int value( $val ) to a shader field of shader( $objectName )
#  On success 1 is returned, otherwise 0 is returned.
#
#
sub setShaderIntAttribute
{
	my ( $objectName, $symName, $val) = @_;

	printBtMsg( "setShaderIntAttribute $objectName $symName $val" );
	
	$result = AutoTest::setShaderIntAttribute( $objectName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setShaderIntAttribute failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}


#%
# \fn setShaderFloatAttribute( $objectName, $symName, $val)
#  Set a float value( $val ) to a shader field of shader( $objectName )
#  On success 1 is returned, otherwise 0 is returned.
#
#
sub setShaderFloatAttribute
{
	my ( $objectName, $symName, $val) = @_;

	printBtMsg( "setShaderFloatAttribute $objectName $symName $val" );
	
	$result = AutoTest::setShaderFloatAttribute( $objectName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setShaderFloatAttribute failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn setShaderStringAttribute( $objectName, $symName, $val)
#  Set a string value( $val ) to a shader field of shader( $objectName )
#  On success 1 is returned, otherwise 0 is returned.
#
#
sub setShaderStringAttribute
{
	my ( $objectName, $symName, $val) = @_;

	printBtMsg( "setShaderStringAttribute $objectName $symName $val" );
	
	$result = AutoTest::setShaderStringAttribute( $objectName, $symName, $val );
	if ( $result == 0 )
	{
		printBtMsg "setShaderStringAttribute failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn setShaderRGBAttribute ( $objectName, $symName,$r, $g, $b )
#  Set a color value ( $r, $g, $b ) to a shader field of shader( $objectName )
#  On success 1 is returned, otherwise 0 is returned. 
#
sub setShaderRGBAttribute 
{
	my ( $objectName, $symName, $r, $g, $b ) = @_;

	printBtMsg( "setShaderRGBAttribute $objectName $symName $r $g $b" );
	
	$result = AutoTest::setShaderRGBAttribute( $objectName, $symName, $r, $g, $b );
	if ( $result == 0 )
	{
		printBtMsg "setShaderRGBAttribute failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}


#%
# \fn assignShader( $shaderName )
# Assigns the shader to all currently selected objects.
#
sub assignShader
{
	my ( $shaderName ) = @_;
	AutoTest::assignShader( $shaderName );
}


#%
# \fn mapTexture( $objectName, $symName, $textureType, $textureName )
# Map a texture(Type is $textureType ) to a shader field( $symName ) of shader( $objectName )
#  On success 1 is returned, otherwise 0 is returned.
#
#
sub mapTexture
{
	my ( $objectName, $symName, $textureType, $textureName ) = @_;

	printBtMsg( "mapTexture  $objectName $symName $textureType $textureName " );
	
	$result = AutoTest::mapTexture( $objectName, $symName, $textureType, $textureName );
	if ( $result == 0 )
	{
		printBtMsg "mapTexture failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}


#%
# \fn setShaderName( $objectName, $newName )
# Set new name( $newName ) for a shader( $objectName )
# On success 1 is returned, otherwise 0 is returned.
#
#
sub setShaderName
{
	my ( $objectName, $newName ) = @_;

	printBtMsg( "setShaderName $objectName $newName " );
	
	$result = AutoTest::setShaderName( $objectName, $newName );
	if ( $result == 0 )
	{
		printBtMsg "setShaderName failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn setShaderModel( $objectName, $shadingModel)
# Set the shading model( $shadingModel ) for a shader ( $objectName ).
# On success 1 is returned, otherwise 0 is returned.
#
#
sub setShaderModel
{
	my ( $objectName, $shadingModel ) = @_;

	printBtMsg( "setShaderModel $objectName $shadingModel " );
	
	$result = AutoTest::setShaderModel( $objectName, $shadingModel );
	if ( $result == 0 )
	{
		printBtMsg "setShaderModel failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn setLightType( $objectName, $lightType )
# Set light type( $lightType ) for a light ( $objectName ).
# On success 1 is returned, otherwise 0 is returned.
#
#
sub setLightType
{
	my ( $objectName, $lightType ) = @_;

	printBtMsg( "setLightType $objectName $lightType " );
	
	$result = AutoTest::setLightType( $objectName, $lightType );
	if ( $result == 0 )
	{
		printBtMsg "setLightType failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
#\fn newShader( $objectName )
# create a shader and it will be displayed in Multi-lister window
sub newShader
{
	my ( $objectName ) = @_;
	printBtMsg( "newShader $objectName " );
	$result = AutoTest::newShader( $objectName );
	if ( $result == 0 )
	{
		printBtMsg "newShader failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};
#

#%
#\fn newLight( $objectName )
# create a light and it will be displayed in Multi-lister window
sub newLight
{
	my ( $objectName ) = @_;
	printBtMsg( "newLight $objectName" );
	$result = AutoTest::newLight( $objectName );
	if ( $result == 0 )
	{
		printBtMsg "newLight failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};
#

#%
#\fn newLayeredShader( $objectName )
# create a layered shader and it will be displayed in Multi-lister window
sub newLayeredShader
{
	my ( $objectName ) = @_;
	printBtMsg( "newLayeredShader $objectName" );	
	$result = AutoTest::newLayeredShader( $objectName );
	if ( $result == 0 )
	{
		printBtMsg "newLayeredShader failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};
#


#%
# \fn deleteShaderEntry( $objectName )
# delete a shader entry named $objectName exit in Multi-lister window
# On success 1 is returned, otherwise 0 is returned.
#
sub deleteShaderEntry
{
	my ( $objectName ) = @_;
	
	printBtMsg( "deleteShaderEntry $objectName" );
	
	$result = AutoTest::deleteShaderEntry( $objectName );
	if ( $result == 0 )
	{
		printBtMsg "deleteShaderEntry failed: @_";
		#ignore deleteShaderEntry return value
		#throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn addTextureRampHandle( $objectName, $position, $r, $g, $b )
# Add a ramp handle which represents a color point in ramp widget.
# On success 1 is returned, otherwise 0 is returned.
#
sub addTextureRampHandle
{
	my ( $objectName, $position, $r, $g, $b ) = @_;
	
	printBtMsg( "addTextureRampHandle $objectName $position $r $g $b" );
	
	$result = AutoTest::addTextureRampHandle( $objectName, $position, $r, $g, $b );
	if ( $result == 0 )
	{
		printBtMsg "addTextureRampHandle failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn deleteTextureRampHandle( $objectName, $key )
# Delete a ramp handle which represents a color point in ramp widget.
# On success 1 is returned, otherwise 0 is returned.
#
sub deleteTextureRampHandle
{
	my ( $objectName, $key ) = @_;
	
	printBtMsg( "deleteTextureRampHandle $objectName $key" );
	
	$result = AutoTest::deleteTextureRampHandle( $objectName, $key );
	if ( $result == 0 )
	{
		printBtMsg "deleteTextureRampHandle failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn setTextureRampHandleColor( $objectName, $key, $r, $g, $b )
# Set the color attribute for a color point which exists in ramp widget.
# On success 1 is returned, otherwise 0 is returned.
#
sub setTextureRampHandleColor
{
	my ( $objectName, $key, $r, $g, $b ) = @_;
	
	printBtMsg( "setTextureRampHandleColor $objectName $key $r $g $b" );
	
	$result = AutoTest::setTextureRampHandleColor( $objectName, $key, $r, $g, $b );
	if ( $result == 0 )
	{
		printBtMsg "setTextureRampHandleColor failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn setTextureRampHandlePosition( $objectName, $key, $position )
# Set the position attribute for a color point which exists in ramp widget.
# On success 1 is returned, otherwise 0 is returned.
#
sub setTextureRampHandlePosition
{
	my ( $objectName, $key, $position ) = @_;
	
	printBtMsg( "setTextureRampHandlePosition $objectName $key $position" );
	
	$result = AutoTest::setTextureRampHandlePosition( $objectName, $key, $position );
	if ( $result == 0 )
	{
		printBtMsg "setTextureRampHandlePosition failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};

#%
# \fn setPixFile( $objectName, $symName, $filePath )
# Browse a picture file for a shader field of a shader.
# On success 1 is returned, otherwise 0 is returned.
#
sub setPixFile
{
	my ( $objectName, $symName, $filePath ) = @_;
	
	printBtMsg( "setPixFile $objectName $symName $filePath" );
	
	$result = AutoTest::setPixFile( $objectName, $symName, $filePath );
	if ( $result == 0 )
	{
		printBtMsg "setPixFile failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
};


#
# End fix Sug #224047
#


# Initialize a tracing operation
my $traceFunction=undef;
my $traceWindow=undef;
my $traceButton=undef;
my $trace_x=undef;
my $trace_y=undef;
my $trace_z=undef;

#%
# \fn initTrace($winName, $tracingFunction, $t, $b, $x, $y, $z )
# Initialize a Studio tracing function such as sid brushes.
# On success 1 is returned. Otherwise 0 is returned.
#
sub initTrace
{
	my ($winName, $tracingFunction, $t, $b, $x, $y, $z ) = @_;
	
	if ( ! selectFunction( $tracingFunction ) )
	{
		printBtMsg "initTrace failed: @_";
		debugPrint("initTrace Failed to select @_");
		throwExceptionIfFlagOn @_;
		return 0;
	}

	$result = AutoTest::pressXYZ( $winName, $t, $b, $x, $y, $z, 0 );
	if ( $result == 0 )
	{
		printBtMsg "initTrace - press failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	
	$traceFunction = $tracingFunction;
	$traceWindow = $winName;
	$traceButton = $b;
	$trace_x = $x;
	$trace_y = $y;
	$trace_z = $z;
	return 1;
}

#%
# \fn appendTracePoint($t,$x,$y,$z)
# Add an additional trace point x,y,z.
# 1 is returned on success, 0 otherwise.
#
sub appendTracePoint
{	
	my ($t,$x,$y,$z) = @_;
	
	$result = AutoTest::moveXYZ( $traceWindow, $t, $traceButton, $x, $y, $z, 0 );
	if ( $result == 0 )
	{
		printBtMsg "appendTracePoint - append failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 

	$trace_x = $x;
	$trace_y = $y;
	$trace_z = $z;
	return 1;
}

#%
# \fn endTrace($t)
# Stop the tracing operation.
# 1 is returned on success, 0 otherwise.
#
sub endTrace
{
	my ($t) = @_;
	$result = AutoTest::releaseXYZ( $traceWindow, $t, $traceButton, 
                                    $trace_x, $trace_y, $trace_z, 1 );
	if ( $result == 0 )
	{
		printBtMsg "endTrace - release failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 	

	$traceFunction = undef;
	$traceWindow = undef;
	$trace_x = undef;
	$trace_y = undef;
	$trace_z = undef;
	
	return 1;
}

#%
# \fn clickWindowXYZ( $winName, $x, $y, $z )
# Emit a click at point x,y,z into the Alias 
# window $winName.
# 1 is returned on success, 0 otherwise.
#
sub clickWindowXYZ
{
	my ( $winName, $x, $y, $z ) = @_;

	printBtMsg( "clickWindowXYZ $winName $x $y $z" );
	
	$result = AutoTest::clickXYZ( $winName, $x, $y, $z );
	if ( $result == 0 )
	{
		printBtMsg "clickWindowXYZ failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn clickWindowButtonXYZ( $winName, $btn, $x, $y, $z )
# Emit a click at point x,y,z into the Alias 
# window $winName with a button modified $btn.
# 1 is returned on success, 0 otherwise.
#
sub clickWindowButtonXYZ
{
	my ( $winName, $btn, $x, $y, $z ) = @_;

	printBtMsg( "clickWindowButtonXYZ $winName $x $y $z" );
	
	$result = AutoTest::clickButtonXYZ( $winName, $btn, $x, $y, $z );
	if ( $result == 0 )
	{
		printBtMsg "clickWindowButtonXYZ failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn pickObjectUV( $objName, $u,  $v )
# Pick an object $objName at a specified $u, $v position.
#
sub pickObjectUV
{
	my ( $objName, $u,  $v ) = @_;

	printBtMsg( "pickObjectUV $objName $u  $v" );
	
	$result = AutoTest::pickObjectUV( $objName, $u, $v );
	if ( $result == 0 )
	{
		printBtMsg "pickObjectUV failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn fitEdwinWindow( $winName )
# perform a fit operation on an EdwinWindow $winName.
#
sub fitEdwinWindow
{
	my ( $winName, $expandTitles ) = @_;

	printBtMsg( "fitEdwinWindow $winName $expandTitles" );
	
	$result = AutoTest::fitEdwinWindow( $winName, $expandTitles );
	if ( $result == 0 )
	{
		printBtMsg "fitEdwinWindow failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn pickObjByName( $objName )
# Pick an object $objName.
# 1 is returned on success, 0 otherwise.
#
sub pickObjByName
{
	my ( $objName ) = @_;
	
	printBtMsg( "pickObjByName $objName" );
	
	$result = AutoTest::pickObjectByName( $objName );
	if ( $result == 0 )
	{		
		printBtMsg( "pickObjByName failed: @_" );
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn pickCurveCVByIndex( $objName, $uIndex )
# Pick a curve CV based on the U index of the CV.
# 1 is returned on success, 0 otherwise.
#
sub pickCurveCVByIndex
{
	my ( $objName, $uIndex ) = @_;
	
	printBtMsg( "pickCurveCVByIndex $objName $uIndex" );
	
	$result = AutoTest::pickCurveCVByIndex( $objName, $uIndex );
	if ( $result == 0 )
	{		
		printBtMsg( "pickCurveCVByIndex failed: @_" );
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn pickSurfaceCVByIndex( $objName, $uIndex, $vIndex )
# Pick a surface CV based on the UV index of the CV
# 1 is returned on success, 0 otherwise.
#
sub pickSurfaceCVByIndex
{
	my ( $objName, $uIndex, $vIndex ) = @_;
	
	printBtMsg( "pickSurfaceCVByIndex $objName $uIndex $vIndex" );
	
	$result = AutoTest::pickSurfaceCVByIndex( $objName, $uIndex, $vIndex );
	if ( $result == 0 )
	{		
		printBtMsg( "pickSurfaceCVByIndex failed: @_" );
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn pickedObjects(void)
# Return the list of picked objects as a string.
# 1 is returned on success, 0 otherwise.
#
sub pickedObjects
{
	printBtMsg( "pickedObjects " );
	
	$result = AutoTest::pickedObjects();
	if ( ! defined($result) )
	{		
		printBtMsg( "pickedObjects failed: @_" );
		throwExceptionIfFlagOn @_;
		return undef;
	} 
	return $result;
}

#%
# \fn renameDag( $objName,$newObjName )
# Rename dag $objName to $newObjName.
# 1 is returned on success, 0 otherwise.
#
sub renameDag
{
	my ($objName,$newObjName) = @_;

	printBtMsg( "renameDag $objName $newObjName" );
	
	$result = AutoTest::renameDag( $objName,$newObjName );
	if ( $result == 0 )
	{		
		printBtMsg( "pickSurfaceCVByIndex failed: @_" );
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn dagTreeNames( $objName )
# Return a string that contains the names of the
# dags below dag $objName.
# undef is returned if this function fails.
#
sub dagTreeNames
{
	my ($objName) = @_;

	printBtMsg( "dagTreeNames $objName" );
	
	$result = AutoTest::dagTreeNames($objName);
	if ( ! defined($result) )
	{		
		printBtMsg( "dagTreeNames failed: @_" );
		return undef;
	} 
	return $result;
}

#%
# \fn selectFunction( $funcName )
# Select an Alias function.  Equivalent to picking the button or
# menu item in Alias.  Find the name of the function by clicking
# on the menu or button with the middle mouse button.
# 1 is returned on success, 0 otherwise. 
# \note This function is used to detect the Escape key press
# which aborts scripting operations.
#
sub selectFunction
{
	my ( $funcName ) = @_;

	printBtMsg( "selectFunction $funcName" );

	$result = AutoTest::selectFunction( $funcName );
	if ( $result == 0 )
	{
		printBtMsg "selectFunction failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	if ( $result == -1 )
	{
		die "User aborted script, stopped";
	}

	return 1;
}

#%
# \fn outputFunctionSymbols( $funcName, $outputFileName, $verbose )
# Select an Alias function and print the full of symbols to a txt file.
# If verbose is false, only output symbol_name. 
# If verbose is true, output ui_function_name, symbol_name, symbol_type and symbol_value.
# A user can omit 3rd argument. By default, $verbose is 0.
# 1 is returned on success, 0 otherwise.
#
sub outputFunctionSymbols
{
	my ( $funcName, $outputFileName, $verbose ) = @_;
	
	printBtMsg( "outputFunctionSymbols $funcName $outputFileName $verbose" );
	
	# Make the internal default 0 for the 3rd argument
	if(! defined $verbose )
	{
	    $verbose = 0; 
	}
	
	$result = AutoTest::outputFunctionSymbols( $funcName, $outputFileName, $verbose );	
	if ( $result == 0 )
	{
	    printBtMsg( "outputFunctionSymbols failed: @_" );
	    throwExceptionIfFlagOn @_;
		return 0;
	}
	if ( $result == -1 )
	{
	    die "User aborted script, stopped";	
	}
	
	return 1;
}

#%
# \fn setTripleSymbol( $funcName, $symName, $val1, $val2, $val3 )
# Set an Alias symbol $symName associated with a function $funcName 
# to value $val1 $val2 $val3. On success1 is returned, otherwise 0 
# is returned.
#
sub setTripleSymbol
{
       my ( $funcName, $symName, $val1, $val2, $val3 ) = @_;

       printBtMsg( "setTripleSymbol $funcName $symName $val1 $val2 $val3" );
       
       $result = AutoTest::setTripleSymbol( $funcName, $symName, $val1, $val2, $val3 );
       if ( $result == 0 )
       {
            printBtMsg "setTripleSymbol failed: @_";
            throwExceptionIfFlagOn @_;
		return 0;
       } 
       if ( $result == -1 )
       {
            die "User aborted script, stopped";	       
       }
       return 1;
}

#%
# \fn resetFunctionSymbols( $funcName )
# Select an Alias function and reset its full of symbols.
# 1 is returned on success, 0 otherwise.
#
sub resetFunctionSymbols
{
	my ( $funcName ) = @_;
	
	printBtMsg( "resetFunctionSymbols $funcName" );
	
	$result = AutoTest::resetFunctionSymbols( $funcName );	
	if ( $result == 0 )
	{
	    printBtMsg( "resetFunctionSymbols failed: @_" );
	    throwExceptionIfFlagOn @_;
		return 0;
	}
	if ( $result == -1 )
	{
	    die "User aborted script, stopped";	
	}
	
	return 1;
}

#%
# \fn loadSymbleTableFrom( $funcName )
# This function copies UI symble list of the given uifunction to
# symble table, and update the c variables.
#
sub loadSymbleTableFrom
{
	my ( $funcName ) = @_;

	printBtMsg( "loadSymbleTableFrom $funcName" );

	$result = AutoTest::loadSymbleTableFrom( $funcName );
	if ( $result == 0 )
	{
		printBtMsg "loadSymbleTableFrom failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	if ( $result == -1 )
	{
		die "User aborted script, stopped";
	}

	return 1;
}

#%
# \fn clickGoButton( $label )
# Caches the result for the next Go button response.
# 1 is returned on success, 0 otherwise. 
#
sub clickGoButton
{
	my ( $label ) = @_;

	printBtMsg( "clickGoButton $label" );
	
	$result = AutoTest::clickGoButton( $label );
	if ( $result == 0 )
	{
		printBtMsg "clickGoButton failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn clickBuildButton( $label )
# Caches the result for the next Build button response.
# 1 is returned on success, 0 otherwise.
#
sub clickBuildButton
{
	my ( $label ) = @_;

	printBtMsg( "clickBuildButton $label" );
	
	$result = AutoTest::clickBuildButton( $label );
	if ( $result == 0 )
	{
		printBtMsg "clickBuildButton failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn confirmAnswerPrompt( $label )
# Caches the next result for the next Confirm box.
# 1 is returned on success, 0 otherwise.
#
sub confirmAnswerPrompt
{
	my ( $label ) = @_;

	printBtMsg( "confirmAnswerPrompt $label" );
	
	$result = AutoTest::confirmAnswer( $label );
	if ( $result == 0 )
	{
		printBtMsg "confirmAnswerPrompt failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

#%
# \fn fileBrowserPrompt( $fileName )
# Caches the next file name for the file browser
# 1 is returned on success, 0 otherwise.
#
sub fileBrowserPrompt
{
	my ( $fileName ) = @_;

	printBtMsg( "fileBrowserPrompt $fileName" );
	
	my $prevFileName = AutoTest::fileBrowserPrompt( $fileName );

	return 1;
}

#%
# \fn clearBrowserPrompt(void)
# Clear the file browser.  Unless you call this function,
# Alias will use the cached answer.
# 1 is returned on success, 0 otherwise.
#
sub clearBrowserPrompt
{
	printBtMsg( "clearBrowserPrompt " );
	
	my $prevFileName = AutoTest::clearBrowserPrompt();

	return 1;
}

#%
# \fn sendString( $str )
# Send a string $str to the prompt line.
# 1 is returned on success, 0 otherwise.
#
sub sendString
{
	my ( $str ) = @_;

	printBtMsg( "sendString $str" );

	# Tab below allows sending a string to
	# the prompt even if we are in single
	# hot key mode.
	AutoTest::sendString( "$str" );
	return 1;
}

#%
# \fn selectFromViewPanel( )
# Notifies the plug-in we are selecting an item available on the view panel
# 1 is returned on success, 0 otherwise.
#
sub selectFromViewPanel
{
	my ( $from, $to ) = @_;
	
	printBtMsg( "selectFromViewPanel $from $to" );

	AutoTest::selectFromViewPanel( "$from", "$to" );
	return 1;
}

#%
# \fn startPickChooser( )
# Notifies the plug-in we are starting a new list of pick chooser entries to be forced
# 1 is returned on success, 0 otherwise.
#
sub startPickChooser
{
	printBtMsg( "startPickChooser" );

	# Tab below allows sending a string to
	# the prompt even if we are in single
	# hot key mode.
	AutoTest::startPickChooser( );
	return 1;
}

#%
# \fn startPickChooser( )
# Notifies the plug-in we are done sending pick chooser entries to be forced
# 1 is returned on success, 0 otherwise.
#
sub endPickChooser
{
	my ( $winname, $btn, $x, $y, $z ) = @_;
	printBtMsg( "endPickChooser $winname $btn $x $y $z" );

	# Tab below allows sending a string to
	# the prompt even if we are in single
	# hot key mode.
	AutoTest::endPickChooser( $winname, $btn, $x, $y, $z );
	return 1;
}

#%
# \fn addPickChooserEntry( $name )
# Adds an entry to the pick chooser force list
# 1 is returned on success, 0 otherwise.
#
sub addPickChooserEntry
{
	my ( $name ) = @_;
	printBtMsg( "endPickChooser $name" );

	# Tab below allows sending a string to
	# the prompt even if we are in single
	# hot key mode.
	AutoTest::addPickChooserEntry( $name );
	return 1;
}

#%
# \fn dataTransOptionDrv( $dt_method, $dt_retrieve_method, $dt_store_method )
# close datatranfer option window.
#
sub dataTransOptionDrv
{
	my ( $dt_method, $dt_retrieve_method, $dt_store_method ) = @_;

	printBtMsg( "dataTransOptionDrv $dt_method, $dt_retrieve_method, $dt_store_method" );

	# Tab below allows sending a string to
	# the prompt even if we are in single
	# hot key mode.
	AutoTest::dataTransOptionDrv( $dt_method, $dt_retrieve_method, $dt_store_method );
	return 1;
}

#
# Snapping support
#

#%
# \fn snappingToGrid(void)
# Return a non-zero value if we are snapping to grid.
# Zero is returned if this snap mode is off.
#
sub snappingToGrid
{
	return AutoTest::snappingToGrid();
}

#%
# \fn snappingToCurve(void)
# Returns a non-zero value if we are snapping to curves.
# Zero is returned if this snap mode is off.
#
sub snappingToCurve
{
	return AutoTest::snappingToCurve();
}

#%
# \fn snappingToCV(void)
# Returns a non-zero value if we are snapping to CVs
# Zero is returned if this snap mode is off.
#
sub snappingToCV
{
	return AutoTest::snappingToCV();
}

#%
# \fn snapToGrid($on)
# Set grid snap mode to $on.
#
sub snapToGrid
{
	my ($on) = @_;
	
	if ( $on )
	{
		if ( ! snappingToGrid() )
		{
			AutoTest::toggleSnapToGrid();
		}
	}
	else
	{
		if ( snappingToGrid() )
		{
			AutoTest::toggleSnapToGrid();
		}
	}
}

#%
# \fn snapToCurve( $on )
# Set curve snap mode to $on.  Note that it is not sufficient to call
# this function with a true parameter.  A curve to snap to needs
# to be picked.
#
sub snapToCurve
{
	my ($on) = @_;

	if ( $on )
	{
		if ( ! snappingToCurve() )
		{
			AutoTest::toggleSnapToCurve();
		}
	}
	else
	{
		if ( snappingToCurve() )
		{
			AutoTest::toggleSnapToCurve();
		}
	}
}

#%
# \fn snapToCV( $on )
# Set CV snap mode to $on.
#
sub snapToCV
{
	my ($on) = @_;

	if ( $on )
	{
		if ( ! snappingToCV() )
		{
			AutoTest::toggleSnapToCV();
		}
	}
	else
	{
		if ( snappingToCV() )
		{
			AutoTest::toggleSnapToCV();
		}
	}
}

#%
# \fn atSaveSTEP( $fileName )
# Save the current model as a step file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveSTEP
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveSTEP: \"$fName\"" );

	$status = AutoTest::atSaveSTEP( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveSTEP: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveJT( $fileName )
# Save the current model as a JT file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveJT
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveJT: \"$fName\"" );

	$status = AutoTest::atSaveJT( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveJT: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveCATIAV5( $fileName )
# Save the current model as a CATIAV5 file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveCATIAV5
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveCATIAV5: \"$fName\"" );

	$status = AutoTest::atSaveCATIAV5( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveCATIAV5: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveInventor( $fileName )
# Save the current model as an inventor file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveInventor
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveInventor: \"$fName\"" );

	$status = AutoTest::atSaveInventor( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveInventor: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveGranite( $fileName )
# Save the current model as a pro/E file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveGranite
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveGranite: \"$fName\"" );

	$status = AutoTest::atSaveGranite( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveGranite: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveCAI( $fileName )
# Save the current model as a cai file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveCAI
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveCAI: \"$fName\"" );

	$status = AutoTest::atSaveCAI( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveCAI: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveDBVIEW( $fileName )
# Save the current model as a dbview file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveDBVIEW
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveDBVIEW: \"$fName\"" );

	$status = AutoTest::atSaveDBVIEW( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveDBVIEW: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveUGII( $fileName )
# Save the current model as a ug file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveUGII
{
	my ( $fName ) = @_;
	printBtMsg "atSaveUGII: \"$fName\"";

	$status = AutoTest::atSaveUGII( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveUGII: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveIges( $fileName )
# Save the current model as a iges file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveIges
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveIges: \"$fName\"" );

	$status = AutoTest::atSaveIges( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveIges: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveDxf( $fileName )
# Save the current model as a dxf file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveDxf
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveDxf: \"$fName\"" );

	$status = AutoTest::atSaveDxf( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveDxf: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn atSaveVdafs( $fileName )
# Save the current model as a vdafs file named $fileName.
# 1 is returned on success, 0 otherwise.
# 
sub atSaveVdafs
{
	my ( $fName ) = @_;
	printBtMsg( "atSaveVdafs: \"$fName\"" );

	$status = AutoTest::atSaveVdafs( $fName );
	if ( $status == 0 )
	{
		printBtMsg "atSaveVdafs: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setDynXsecNumber
{
	my ( $step_size ) = @_;
	printBtMsg( "setDynXsecNumber $step_size" );

	$status = AutoTest::setDynXsecNumber( $step_size );
	if ( $status == 0 )
	{
		printBtMsg "setDynXsecNumber: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setDynXsecStepsize
{
	my ( $step_size ) = @_;
	printBtMsg( "setDynXsecStepsize $step_size" );

	$status = AutoTest::setDynXsecStepsize( $step_size );
	if ( $status == 0 )
	{
		printBtMsg "setDynXsecStepsize: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setSurfCurvActive
{
	my ( $flag ) = @_;
	printBtMsg( "setSurfCurvActive $flag" );

	$status = AutoTest::setSurfCurvActive( $flag );
	if ( $status == 0 )
	{
		printBtMsg "setSurfCurvActive: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setSurfCurvAutorange
{
	my ( $flag ) = @_;
	printBtMsg( "setSurfCurvAutorange $flag" );

	$status = AutoTest::setSurfCurvAutorange( $flag );
	if ( $status == 0 )
	{
		printBtMsg "setSurfCurvAutorange: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setSurfHighlightActive
{
	my ( $flag ) = @_;
	printBtMsg( "setSurfHighlightActive $flag" );

	$status = AutoTest::setSurfHighlightActive( $flag );
	if ( $status == 0 )
	{
		printBtMsg "seSurftHighlightActive: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setSurfHighlightRangeset
{
	my ( $flag ) = @_;
	printBtMsg( "setSurfHighlightRangeset $flag" );

	$status = AutoTest::setSurfHighlightRangeset( $flag );
	if ( $status == 0 )
	{
		printBtMsg "setSurfHighlightRangeset: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setIntParmType
{
	my ( $type ) = @_;
	printBtMsg( "setIntParmType $type" );

	$status = AutoTest::setIntParmType( $type );
	if ( $status == 0 )
	{
		printBtMsg "setIntParmType: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setAoMirrorPlane
{
	my ( $plane ) = @_;
	printBtMsg( "setAoMirrorPlane $plane" );

	$status = AutoTest::setAoMirrorPlane( $plane );
	if ( $status == 0 )
	{
		printBtMsg "setAoMirrorPlane: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setMoTransx
{
	my ( $x ) = @_;
	printBtMsg( "setMoTransx $x" );

	$status = AutoTest::setMoTransx( $x );
	if ( $status == 0 )
	{
		printBtMsg "setMoTransx: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}
sub setMoTransy
{
	my ( $y ) = @_;
	printBtMsg( "setMoTrans $y" );

	$status = AutoTest::setMoTransy( $y );
	if ( $status == 0 )
	{
		printBtMsg "setMoTransy: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}
sub setMoTransz
{
	my ( $z ) = @_;
	printBtMsg( "setMoTransz $z" );

	$status = AutoTest::setMoTransz( $z );
	if ( $status == 0 )
	{
		printBtMsg "setMoTransz: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setMoScalex
{
	my ( $x ) = @_;
	printBtMsg( "setMoScalex $x" );

	$status = AutoTest::setMoScalex( $x );
	if ( $status == 0 )
	{
		printBtMsg "setMoScalex: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}
sub setMoScaley
{
	my ( $y ) = @_;
	printBtMsg( "setMoScaley $y" );

	$status = AutoTest::setMoScaley( $y );
	if ( $status == 0 )
	{
		printBtMsg "setMoScalyey: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setMoScalez
{
	my ( $z ) = @_;
	printBtMsg( "setMoScalez $z" );

	$status = AutoTest::setMoScalez( $z );
	if ( $status == 0 )
	{
		printBtMsg "setMoScalez: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setMoCpynumber
{
	my ( $num ) = @_;
	printBtMsg( "setMoCpynumber $num" );

	$status = AutoTest::setMoCpynumber( $num );
	if ( $status == 0 )
	{
		printBtMsg "setMoCpynumber: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setMoGeomType
{
	my ( $type ) = @_;
	printBtMsg( "setMoGeomType $type" );

	$status = AutoTest::setMoGeomType( $type );
	if ( $status == 0 )
	{
		printBtMsg "setMoGeomType: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setGroupCompensateAndUnder
{
	my ( $compensate, $under ) = @_;
	printBtMsg( "setGroupCompensateAndUnder $compensate, $under" );

	$status = AutoTest::setGroupCompensateAndUnder( $compensate, $under );
	if ( $status == 0 )
	{
		printBtMsg "setGroupCompensateAndUnder: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub sendReturn
{
	my $status = AutoTest::sendReturn();
	if ( $status == 0 )
	{
		printBtMsg "sendReturn: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub setCelldebug
{
	my ( $number ) = @_;
	printBtMsg( "setCelldebug $number" );

	$status = AutoTest::setCelldebug( $number);
	if ( $status == 0 )
	{
		printBtMsg "setCelldebug: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn sizeAliasWorkingArea( $x, $y )
# Experimental routine to set the current size
# of the Studio window.
# 1 is returned on success, 0 otherwise.
#
sub sizeAliasWorkingArea 
{
	my ( $x, $y ) = @_;
	printBtMsg( "sizeAliasWorkingArea $x $y" );

	$status = AutoTest::sizeAliasWorkingArea( $x, $y );
	if ( $status == 0 )
	{
		printBtMsg "sizeAliasWorkingArea: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

#%
# \fn maxAliasWorkingArea( )
# Make Alias window maximized.
# 1 is returned on success, 0 otherwise.
#
sub maxAliasWorkingArea 
{
	$status = AutoTest::maxAliasWorkingArea( );
	if ( $status == 0 )
	{
		printBtMsg "maxAliasWorkingArea: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub startTiming
{
	$AUTO_TEST_TIMER1 =  new Benchmark;
}

sub finishTiming
{
	$AUTO_TEST_TIMER2 =  new Benchmark;
	my $timeDiff = timediff( $AUTO_TEST_TIMER2, $AUTO_TEST_TIMER1 );	
	print "# Operation took: ", timestr($timeDiff), "\n";
}

sub uisymbol
{
	my ( $str, $value ) = @_;
	printBtMsg( "uisymbol $str $value" );
	
	$status = AutoTest::uisymbol( "(ui-symbol \"$str\" \"$value\" )" );
	if ( $status == 0 )
	{
		printBtMsg "uisymbol: returned $status";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return 1;
}

sub saveSpfNext
{
	AutoTest::saveSpfNext()
}

sub saveSpfTextNext
{
	AutoTest::saveSpfNext()
}

#%
# \fn productName(void)
# Return the name string of the current product.
#
sub productName
{
	return AutoTest::productName();
}

#%
# \fn sourcePerlScript( $scriptName )
# Utility function to source a perl script.
#
sub sourcePerlScript
{
	my ( $scriptName ) = @_;
	printBtMsg("sourcePerlScript $scriptName" );
	open (FILE, $scriptName ) or return;
	my @lines = <FILE>;
	close FILE;
	eval "@lines";
}

#%
# \fn lastWireFileRetrieved(void)
# Return the name of the last wire file
# retrieved.
#
sub lastWireFileRetrieved
{
	return AutoTest::lastWireFileRetrieved();
}

#%
# \fn lastWireFileSaved(void)
# Return the name of the last wire file
# saved.
#
sub lastWireFileSaved
{
	return AutoTest::lastWireFileSaved();
}

#%
# \fn viewFrame( $frameNum )
# Set the currently view frame number.
# 1 is returned on success, 0 otherwise.
#
sub viewFrame
{
	my ($frameNum) = @_;
	if ( ! defined($frameNum) )
	{
		printBtMsg "viewFrame: needs a frame number";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	my $result = AutoTest::viewFrame( $frameNum );
	if ( $result == 0 )	# sSuccess
	{
		return 1;
	}
	throwExceptionIfFlagOn @_;
		return 0;
}

#
# Internal use function.  Called by
# other check routines.
#
sub assertCheck
{
	my ($assertType,$objName,$objData) = @_;
	if ( !defined($assertType) || !defined($objName) || !defined($objData) )
	{
		printBtMsg "assertCheck: needs a type, object name and object data";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	my $result = AutoTest::assertCheck($assertType,$objName,$objData);
	return $result;
}

#%
# \fn assertPointCheck( $objName, $x, $y, $z )
# Return 1 if point $objName is at point xyz.
# Else return 0.
#
sub assertPointCheck
{
	my ($objName,$x,$y,$z) = @_;
	if ( !defined($objName) || !defined($x) || !defined($y) || !defined($z) )
	{
		printBtMsg "assertPointCheck: needs a object name and xyz position";
		throwExceptionIfFlagOn @_;
		return 0;
	}
	return assertCheck( "point", $objName,"$x $y $z" );
}

sub timeId
{
	return time();
}

#%
# \fn initializeDefaultSetup()
# This function loads in a default setup that
# should be used for writing framework scripts.
#
sub initializeDefaultSetup
{
	# Remove the modelling windows
	confirmAnswerPrompt( "YES" );
	setIntSymbol( "DelWindows", "mo_delwind", 812 );
	selectFunction( "DelWindows" );

	# The default setup is contained in a wire
	# file.
	my $initFile = $FILES_DIR . "default_autotest_view.wire";
	if ( -f "$initFile" )
	{
		my $result = retrieveFile( $initFile );
		if ( $result )
		{
			return 1;
		}
	}
	printBtMsg "initializeDefaultSetup: failed\n";
	throwExceptionIfFlagOn @_;
		return 0;
}

#%
# \fn layerPickObjects( $layerName )
#  Pick objects on the layer named $layerName
#  On success 1 is returned, otherwise 0 is returned.
#
#
sub layerPickObjects
{
	my ( $layerName) = @_;

	printBtMsg( "layerPickObjects $layerName " );
	
	$result = AutoTest::layerPickObjects( $layerName );
	if ( $result == 0 )
	{
		printBtMsg "layerPickObjects failed: @_";
		throwExceptionIfFlagOn @_;
		return 0;
	} 
	return 1;
}

# Return value for source
1;
