/*global define, describe, it, expect, beforeEach, afterEach */
/*jslint white: true */

define (["lodash",		"src/utils"],
function(lodash, 		utils) {
	'use strict';

	return function() {
		describe("utils.mixin", function () {
			var array = ['A', 'B'],
				source = { a : function(array) { return array[0]; }, b : 'B', c : {} };

			beforeEach(function () {
			});

			afterEach(function() {
			});

			describe("with object as a source", function () {

				it("should mixin all objects methods into destination", function () {
					var destination = {};
					utils.mixin(destination, source);
					// functions are there...
					expect(destination.a(array)).toEqual('A');
					// ...others are not
					expect(destination.b).not.toBeDefined();
					expect(destination.c).not.toBeDefined();
				});

				it("should mixin methods from all sources", function () {
					var instance;
					function Ctor (array) { this.array = array; }
					utils.mixin(Ctor, source, 
						{ 
							d: function() { return this.array[0]; },
							a: function(array) { return array[1]; }
						}
					);

					instance = new Ctor(array);
					expect(instance.d()).toEqual('A');
					// subsequent sources override previous sources
					expect(instance.a(array)).toEqual('B');
				});


				it("should mixin methods into prototype object when destination is a constructor", function () {
					var instance;
					function Ctor (array) { this.array = array; }
					utils.mixin(Ctor, source);
					utils.mixin(Ctor, { d: function() { return this.array[0]; }});

					// functions are in prototype...
					expect(Ctor.prototype.a).toBeDefined();
					expect(Ctor.prototype.d).toBeDefined();
					// ...others are not
					expect(Ctor.prototype.b).not.toBeDefined();
					expect(Ctor.prototype.c).not.toBeDefined();

					instance = new Ctor(array);
					expect(instance.a(array)).toEqual('A');
					expect(instance.d()).toEqual('A');
				});
			});

			describe("with constructor as a source", function () {

				it("should support inheritence", function () {

					function Base() {
						this.pBase = "base private";
					}
					utils.mixin(Base, { getBaseState : function () { return this.pBase; }});

					function Derived() {
						this.Base();
						this.pDerived = "derived private";
					}
					utils.mixin(Derived, Base, { getDerivedState : function () { return this.pDerived; }});

					var instance = new Derived();

					expect(instance.getDerivedState).toBeDefined();
					expect(instance.getBaseState).toBeDefined();
					expect(instance.getBaseState()).toEqual("base private");
					expect(instance.getDerivedState()).toEqual("derived private");
				});

				it("should preserve semantics of constructor and instanceof operators", function () {

					var constructorGetter = { getConstructor : function() { return this.constructor; }},
						baseInstance, derivedInstance;

					function Base() {}
					utils.mixin(Base, constructorGetter);

					function Derived() {}
					utils.mixin(Derived, Base);

					baseInstance = new Base();
					derivedInstance = new Derived();

					expect(baseInstance.getConstructor()).toBe(Base);
					expect(derivedInstance.getConstructor()).toBe(Derived);
					expect(baseInstance instanceof Base).toBe(true);
					expect(derivedInstance instanceof Derived).toBe(true);
					expect(baseInstance instanceof Derived).toBe(false);
					expect(derivedInstance instanceof Derived).toBe(true);
					expect(baseInstance instanceof Function).toBe(false);
					expect(derivedInstance instanceof Function).toBe(false);
				});

				it("should guard against enumerable constructor field", function () {
					function Base() {}
					// non-enumerable constructor is already set
					expect(Base.prototype.constructor).toBe(Base);
					expect(lodash(Base.prototype).keys().some("constructor")).toBe(false);
					// but we wipe it accidentally and it becomes enumerable
					Base.prototype = { constructor : Base };
					// verify that test is setup correctly
					expect(lodash(Base.prototype).keys().some("constructor")).toBe(true);

					// verify that we throw type error
					function Derived() {}
					expect(lodash.partial(utils.mixin, Derived, Base)).toThrow();
				});
			});
		});
	};
});
