/*global define, require */
/*jslint white: true */

/*
	math utils:

	This file implements math related utilities.
*/

define(function () {
	'use strict';

	var radiansPerDegree = Math.PI / 180,
		degreesPerRadian = 180 / Math.PI;


	function deg(inRadians) {
		return degreesPerRadian * inRadians;
	}

	function rad(inDegrees) {
		return radiansPerDegree * inDegrees;
	}

	function lerp(a, b, t) {
		return a + t * (b - a);
	}

	function equalWithinEpsilon(inX1F, inX2F, inEpsilonF) {
		inEpsilonF = inEpsilonF || 1e-10;

		return Math.abs(inX1F - inX2F) < inEpsilonF;
	}

	function equalToZeroWithinEpsilon(inValueF, inEpsilonF) {
		return equalWithinEpsilon(inValueF, 0.0, inEpsilonF);
	}

	function cubeRoot(inValueF) {
		return Math.pow(inValueF, 1 / 3);
	}

	function solveLinearRoot(a, b) {
		var root = 0;
		if (equalToZeroWithinEpsilon(a)) {
			root = 0.0;				/* irrational */
		} else {
			root = -b / a;			/* one root */
		}
		return root;
	}

	function solveOneQuadraticRoot(a, b, c) {
		var root, atimes2 = 2.0 * a, ctimes2 = 2.0 * c, e = b * b - atimes2 * ctimes2, roots = [0, 0];

		if (equalToZeroWithinEpsilon(a)) {
			root = solveLinearRoot(b, c, root);
		} else {
			if (e < 0.0) {
				throw new Error("No rational roots.");
			} else {
				if (equalToZeroWithinEpsilon(e)) {			/* one root           */
					root = -b / atimes2;
				} else {									/* two roots          */
					e = Math.sqrt(e);
					if (b > 0.0) {							/* make answer stable */
						roots[0] = -ctimes2 / (e + b);
						roots[1] = -(e + b) / atimes2;
					} else {								/* b <= 0             */
						roots[0] = ctimes2 / (e - b);
						roots[1] = (e - b) / atimes2;
					}
					if (roots[0] >= 0.0 && roots[0] <= 1.0) {
						root = roots[0];
					} else if (roots[1] >= 0.0 && roots[1] <= 1.0) {
						root = roots[1];
					}
				}
			}
		}
		return root;
	}

	function solveOneCubicRoot(a, b, c, d) {
		var root = 0, a1, a2, a3, q, q_cube, r, r_sqr, a1_third, phi, tmp, roots = [0, 0, 0];

		if (equalToZeroWithinEpsilon(a)) {
			root = solveOneQuadraticRoot(b, c, d);
		} else {
			a1 = b / a;
			a2 = c / a;
			a3 = d / a;
			a1_third = a1 / 3.0;
			q = (a1 * a1 - 3.0 * a2) / 9.0;
			q_cube = q * q * q;
			r = (2.0 * a1 * a1 * a1 - 9.0 * a1 * a2 + 27.0 * a3) / 54.0;
			r_sqr = r * r;

			if (q_cube >= r_sqr) {						/* two or three roots */
				if (equalToZeroWithinEpsilon(q)) {		/* degenerate case of */
					root = -a1_third;					/* one triple root    */
				} else {
					phi = Math.acos(r / Math.sqrt(q_cube));
					tmp = -2.0 * Math.sqrt(q);
					roots[0] = tmp * Math.cos(phi / 3.0) - a1_third;
					roots[1] = tmp * Math.cos((phi + Math.PI * 2) / 3.0) - a1_third;
					roots[2] = tmp * Math.cos((phi + 4.0 * Math.PI) / 3.0) - a1_third;

					if (roots[0] > 0.0 && roots[0] < 1.0) {
						root = roots[0];
					} else if (roots[1] >= 0.0 && roots[1] <= 1.0) {
						root = roots[1];
					} else if (roots[2] >= 0.0 && roots[2] <= 1.0) {
						root = roots[2];
					}
				}
			} else if (r >= 0.0) {
				tmp = cubeRoot(Math.sqrt(r_sqr - q_cube) + r);
				root = -tmp - q / tmp - a1_third;
			} else {
				tmp = cubeRoot(Math.sqrt(r_sqr - q_cube) - r);
				root = tmp + q / tmp - a1_third;
			}
		}
		return root;
	}

	function loop(inT, inLoopType) {
		if (inLoopType !== undefined) {
			if (inT < 0) {
				inT = Math.abs(inT);
				inT = Math.floor(inT) + (1 - (inT % 1));
			}
			if (inLoopType === "cycle") {
				inT = inT % 1;
			} else if (inLoopType === "pingpong") {
				inT = inT % 2;
				if (inT > 1) {
					inT = 2 - inT;
				}
			}
		} else {
			inT = Math.max(0, inT);
			inT = Math.min(1, inT);
		}
		return inT;
	}

	function intersectLines(pt1, pt2, pt3, pt4) {
		var result = [], den, cross12, cross34;

		den = (pt1[0] - pt2[0]) * (pt3[1] - pt4[1]) - (pt1[1] - pt2[1]) * (pt3[0] - pt4[0]);

		if (den) {
			cross12 = (pt1[0] * pt2[1] - pt1[1] * pt2[0]);
			cross34 = (pt3[0] * pt4[1] - pt3[1] * pt4[0]);
			result[0] = (cross12 * (pt3[0] - pt4[0]) - cross34 * (pt1[0] - pt2[0])) / den;
			result[1] = (cross12 * (pt3[1] - pt4[1]) - cross34 * (pt1[1] - pt2[1])) / den;
			return result;
		}

		// Line are parallel, return unidentified.
		return undefined;
	}

	function log10(n) {
		return Math.log(n) / Math.log(10);
	}

	function normalizedToDecibel(inNormalized) {
		// Normalized zero is equivalent to decibel -infinity, so we limit to a very small value which approaches zero.
		if (inNormalized < 1e-15) {
			inNormalized = 1e-15; // (-300dB)
		}

		// Normalized one is equivalent to decibel zero.
		return 20.0 * log10(inNormalized);
	}

	function decibelToNormalized(inDecibel) {
		// The smaller the negative decibel value, the closer we'll get to normalized zero.
		// However, this routine will never actually return true normalized zero.
		return Math.pow(10.0, inDecibel / 20.0);
	}

	return {
		oneThird: 1.0 / 3.0,
		lerp: lerp,
		loop: loop,
		equalWithinEpsilon: equalWithinEpsilon,
		equalToZeroWithinEpsilon: equalToZeroWithinEpsilon,
		cubeRoot: cubeRoot,
		solveLinearRoot: solveLinearRoot,
		solveOneQuadraticRoot: solveOneQuadraticRoot,
		solveOneCubicRoot: solveOneCubicRoot,
		intersectLines: intersectLines,
		deg: deg,
		rad: rad,
		log10: log10,
		normalizedToDecibel: normalizedToDecibel,
		decibelToNormalized: decibelToNormalized
	};
});
