/*global define, require */
/*jslint white: true */

/*
	curve utils:

	This file implements curve related utilities.
*/

define ([	"src/math/Vec2",	"src/math/mathUtils",	"src/utils"],
function(	V2,					mathUtils,				utils) {
	'use strict';

	function evaluate1DCurve(c0, c1, c2, c3, t) {
		var oneMinusT = 1.0 - t, tSquared = t * t, tCubed = t * t * t;
		return c3 * tCubed + oneMinusT * (c2 * 3 * tSquared + oneMinusT * (c1 * 3 * t + oneMinusT * c0));
	}

	function evaluate1DCurveDeriv(c0, c1, c2, c3, t) {
		return 3 * (-(c0) + (c1) + t * (2 * ((c0) - 2 * (c1) + (c2)) + t * (-(c0) + 3 * ((c1) - (c2)) + (c3))));
	}

	function evaluate1DCurveDeriv2(c0, c1, c2, c3, t) {
		return 2 * (3 * ((c0) - 2 * (c1) + (c2)) + 3 * t * (-(c0) + 3 * ((c1) - (c2)) + (c3)));
	}

	function evaluate2DCurve(inCurvePts, t) {
		var pts = [];
		pts[0] = evaluate1DCurve(inCurvePts[0][0], inCurvePts[1][0], inCurvePts[2][0], inCurvePts[3][0], t);
		pts[1] = evaluate1DCurve(inCurvePts[0][1], inCurvePts[1][1], inCurvePts[2][1], inCurvePts[3][1], t);
		return pts;
	}

	function evaluate2DCurveDeriv(inCurvePts, t) {
		var pts = [];
		pts[0] = evaluate1DCurveDeriv(inCurvePts[0][0], inCurvePts[1][0], inCurvePts[2][0], inCurvePts[3][0], t);
		pts[1] = evaluate1DCurveDeriv(inCurvePts[0][1], inCurvePts[1][1], inCurvePts[2][1], inCurvePts[3][1], t);
		return pts;
	}

	function evaluate2DCurveDeriv2(inCurvePts, t) {
		var pts = [];
		pts[0] = evaluate1DCurveDeriv2(inCurvePts[0][0], inCurvePts[1][0], inCurvePts[2][0], inCurvePts[3][0], t);
		pts[1] = evaluate1DCurveDeriv2(inCurvePts[0][1], inCurvePts[1][1], inCurvePts[2][1], inCurvePts[3][1], t);
		return pts;
	}

	function getB0(inT) {
		var one_minus_t = 1.0 - inT;
		return one_minus_t * one_minus_t * one_minus_t;
	}

	function getB1(inT) {
		var one_minus_t = 1.0 - inT;
		return 3 * inT * one_minus_t * one_minus_t;
	}

	function getB2(inT) {
		var one_minus_t = 1.0 - inT;
		return 3 * inT * inT * one_minus_t;
	}

	function getB3(inT) {
		return inT * inT * inT;
	}

	function subdivide2DCurve(inCurvePts, t, outCurve1Pts, outCurve2Pts) {
		var p0 = [], p1 = [], p2 = [], p3 = [], p4 = [], p5 = [], p6 = [], p7 = [], i;
		for (i = 0; i < 2; i += 1) {
			p0[i] = inCurvePts[0][i];
			p1[i] = mathUtils.lerp(inCurvePts[0][i], inCurvePts[1][i], t);
			p2[i] = mathUtils.lerp(inCurvePts[1][i], inCurvePts[2][i], t);
			p3[i] = mathUtils.lerp(inCurvePts[2][i], inCurvePts[3][i], t);
			p4[i] = mathUtils.lerp(p1[i], p2[i], t);
			p5[i] = mathUtils.lerp(p2[i], p3[i], t);
			p6[i] = mathUtils.lerp(p4[i], p5[i], t);
			p7[i] = inCurvePts[3][i];
		}

		outCurve1Pts[0] = p0;
		outCurve1Pts[1] = p1;
		outCurve1Pts[2] = p4;
		outCurve1Pts[3] = p6;
		outCurve2Pts[0] = p6;
		outCurve2Pts[1] = p5;
		outCurve2Pts[2] = p3;
		outCurve2Pts[3] = p7;
	}

	function copyCurveInPlace(inCurvePts, outCurvePts) {
		var i;
		for (i = 0; i < 4; i += 1) {
			outCurvePts[i] = [];
			outCurvePts[i][0] = inCurvePts[i][0];
			outCurvePts[i][1] = inCurvePts[i][1];
		}
	}
	function trim2DCurve(inCurvePts, inMinT, inMaxT, outCurvePts) {
		var curve1Pts = [], curve2Pts = [], splitT, tDiv;

		if (inMinT === 0 && inMaxT === 1) {
			copyCurveInPlace(inCurvePts, outCurvePts);
		} else {
			subdivide2DCurve(inCurvePts, inMinT, curve1Pts, curve2Pts);

			if (inMaxT === 1) {
				copyCurveInPlace(curve2Pts, outCurvePts);
			} else {
				tDiv = 1.0 - inMinT;
				if (tDiv) {
					splitT = (inMaxT - inMinT) / tDiv;
				} else {
					splitT = 1.0;
				}
				subdivide2DCurve(curve2Pts, splitT, outCurvePts, curve1Pts);
			}
		}
	}

	function computeTangentOrthogonalLength(inPt0, inPt1, inPt3) {
		var vecs = [[0, 0], [0, 0]], hypLen, adjLen, oppLen;

		V2.subtract(inPt1, inPt0, vecs[0]);
		V2.subtract(inPt3, inPt0, vecs[1]);

		V2.normalize(vecs[1], vecs[1]);

		adjLen = V2.dot(vecs[1], vecs[0]);
		hypLen = V2.magnitude(vecs[0]);
		oppLen = Math.sqrt(hypLen * hypLen - adjLen * adjLen);

		return oppLen;
	}

	function computeChordLength(inCurve) {
		return V2.distance(inCurve[0], inCurve[1]) + V2.distance(inCurve[1], inCurve[2]) + V2.distance(inCurve[2], inCurve[3]);
	}

	function isLinear(inCurve, inMinT, inMaxT, inEpsilon) {
		var subCurvePts = [], oppLen1, oppLen2;

		trim2DCurve(inCurve, inMinT, inMaxT, subCurvePts);

		inEpsilon = inEpsilon || 1e-2;

		oppLen1 = computeTangentOrthogonalLength(subCurvePts[0], subCurvePts[1], subCurvePts[3]);
		oppLen2 = computeTangentOrthogonalLength(subCurvePts[3], subCurvePts[2], subCurvePts[0]);

		return mathUtils.equalToZeroWithinEpsilon(oppLen1 + oppLen2, inEpsilon);
	}

	function forceBezierIncreasingXValues(ioBezPts) {
		if ((ioBezPts[3][0] < ioBezPts[0][0])) {
			throw new Error("Invalid increasing curve fit.");
		}
		if (ioBezPts[1][0] < ioBezPts[0][0]) {
			ioBezPts[1] = utils.clone(true, [], ioBezPts[0]);
		}
		if (ioBezPts[3][0] < ioBezPts[2][0]) {
			ioBezPts[2] = utils.clone(true, [], ioBezPts[3]);
		}
		if (ioBezPts[2][0] < ioBezPts[1][0]) {
			var xMid = (ioBezPts[3][0] + ioBezPts[0][0]) * 0.5, iPt;

			// Fix the tangents...
			iPt = mathUtils.intersectLines(ioBezPts[0], ioBezPts[1], [xMid, 0], [xMid, 1]);
			if (iPt !== undefined) {
				ioBezPts[1][0] = xMid;
				ioBezPts[1][1] = iPt[1];
			}
			iPt = mathUtils.intersectLines(ioBezPts[3], ioBezPts[2], [xMid, 0], [xMid, 1]);
			if (iPt !== undefined) {
				ioBezPts[2][0] = xMid;
				ioBezPts[2][1] = iPt[1];
			}
		}
	}

	// This allows you to smooth a list of points as they come in (from the mouse)
	// and do so in a way that doesn't change history.  If we create c0,c1,c2,c3,
	// c4 will equal c3, c5 will keep the same tangent as c2-c3, c6 will point from
	// c7 to c5.  All tangents are 1/3 the lengh from the end points (c0/c3, c4/c7).
	function IncrementalLineToCurveCreator(inForceAlwaysIncreasingXValues) {
		var previousCurve, previousPoint, that = {};

		inForceAlwaysIncreasingXValues = inForceAlwaysIncreasingXValues || 0;

		that.nextPoint = function (inPoint) {
			var curve, d;
			if (previousPoint !== undefined) {
				curve = [];
				curve[0] = V2.clone(previousPoint);
				curve[3] = V2.clone(inPoint);
				curve[1] = V2.clone(curve[3]);
				V2.subtract(curve[1], curve[0], curve[1]);
				d = V2.magnitude(curve[1]);
				if (previousCurve === undefined) {
					curve[2] = V2.clone(curve[0]);
					V2.subtract(curve[2], curve[3], curve[2]);
					if (V2.magnitudeSquared(curve[1]) > 0) {
						V2.normalize(curve[1], curve[1], mathUtils.oneThird * d);
					}
					V2.add(curve[1], curve[0], curve[1]);
				} else {
					curve[1] = V2.clone(previousCurve[3]);
					V2.subtract(curve[1], previousCurve[2], curve[1]);
					if (V2.magnitudeSquared(curve[1]) > 0) {
						V2.normalize(curve[1], curve[1], mathUtils.oneThird * d);
					}
					V2.add(curve[1], curve[0], curve[1]);
					curve[2] = V2.clone(curve[1]);
					V2.subtract(curve[2], curve[3], curve[2]);
				}
				if (V2.magnitudeSquared(curve[2]) > 0) {
					V2.normalize(curve[2], curve[2], mathUtils.oneThird * d);
				}
				V2.add(curve[2], curve[3], curve[2]);
				if (inForceAlwaysIncreasingXValues) {
					forceBezierIncreasingXValues(curve);
				}
			}
			previousPoint = inPoint;
			previousCurve = curve;
			return previousCurve;
		};

		that.getPreviousCurve = function () {
			return previousCurve;
		};

		return that;
	}

	return {
		ellipseHandleScale : 4 / 3 * (Math.sqrt(2) - 1),
		evaluate1DCurve : evaluate1DCurve,
		evaluate2DCurve : evaluate2DCurve,
		evaluate2DCurveDeriv : evaluate2DCurveDeriv,
		evaluate2DCurveDeriv2 : evaluate2DCurveDeriv2,
		getB0 : getB0,
		getB1 : getB1,
		getB2 : getB2,
		getB3 : getB3,
		isLinear : isLinear,
		computeChordLength : computeChordLength,
		trim2DCurve : trim2DCurve,
		subdivide2DCurve : subdivide2DCurve,
		forceBezierIncreasingXValues : forceBezierIncreasingXValues,
		IncrementalLineToCurveCreator : IncrementalLineToCurveCreator,
		// Compute the t-value along the 2d bezier 'curve' whose x-coordinate is x.
		evaluateTfromX : function (inCurvePts, inX) {
			var a, b, c, d, t;

			if (inX > inCurvePts[3][0] || mathUtils.equalWithinEpsilon(inX, inCurvePts[3][0])) {
				t = 1.0;
			} else if (inX < inCurvePts[0][0] || mathUtils.equalWithinEpsilon(inX, inCurvePts[0][0])) {
				t = 0.0;
			} else {
				d = inCurvePts[0][0] - inX;
				c = 3 * (inCurvePts[1][0] - inCurvePts[0][0]);
				b = 3 * (inCurvePts[0][0] - 2 * inCurvePts[1][0] + inCurvePts[2][0]);
				a = 3 * (inCurvePts[1][0] - inCurvePts[2][0]) + inCurvePts[3][0] - inCurvePts[0][0];

				t = mathUtils.solveOneCubicRoot(a, b, c, d);
			}
			return t;
		},
		// Evaluate Y from X.
		evaluateYfromX : function (inCurvePts, inX) {
			var t, xy;

			t = this.evaluateTfromX(inCurvePts, inX);
			xy = this.evaluate2DCurve(inCurvePts, t);

			return xy[1];
		}
	};
});
