/*global define, require */
/*jslint white: true */

/*
	curve utils:

	This file implements curve related utilities.
*/

define([	"src/math/Vec2",	"src/math/mathUtils",	"src/math/curveUtils"],
function(	V2,					mathUtils,				curveUtils) {
	'use strict';

	function generateBezier(inPts, inStartIndex, inEndIndex, inUPrime, inTanStart, inTanEnd, inForceIncreasingXValues) {
		var i, a = [], nPts, c = [[], []], x = [], det_c0_c1, det_c0_x, det_x_c1,
			alpha_l, alpha_r, basis, tmp = [], tmp1 = [], deriv_epsilon = 1e-4, distance, outCurve;

		outCurve = [];

		nPts = inEndIndex - inStartIndex + 1;

		/* Compute the A's	*/
		for (i = 0; i < nPts; i += 1) {
			basis = curveUtils.getB1(inUPrime[i]);
			V2.scale(basis, inTanStart, tmp);
			basis = curveUtils.getB2(inUPrime[i]);
			V2.scale(basis, inTanEnd, tmp1);
			a[i] = [tmp, tmp1];
		}

		 /* Create the C and X matrices	*/
		c[0][0] = 0.0;
		c[0][1] = 0.0;
		c[1][0] = 0.0;
		c[1][1] = 0.0;
		x[0]    = 0.0;
		x[1]    = 0.0;

		for (i = 0; i < nPts; i += 1) {

			c[0][0] += V2.dot(a[i][0], a[i][0]);
			c[0][1] += V2.dot(a[i][0], a[i][1]);
			c[1][0] = c[0][1];
			c[1][1] += V2.dot(a[i][1], a[i][1]);

			basis = curveUtils.getB0(inUPrime[i]);
			V2.scale(basis, inPts[inStartIndex], tmp1);
			tmp = tmp1;

			basis = curveUtils.getB1(inUPrime[i]);
			V2.scale(basis, inPts[inStartIndex], tmp1);
			V2.add(tmp1, tmp, tmp);

			basis = curveUtils.getB2(inUPrime[i]);
			V2.scale(basis, inPts[inEndIndex], tmp1);
			V2.add(tmp1, tmp, tmp);

			basis = curveUtils.getB3(inUPrime[i]);
			V2.scale(basis, inPts[inEndIndex], tmp1);
			V2.add(tmp1, tmp, tmp);

			V2.subtract(inPts[inStartIndex], tmp, tmp);
			x[0] += V2.dot(a[i][0], tmp);
			x[1] += V2.dot(a[i][1], tmp);
		}

		/* Compute the determinants of C and X	*/
		det_c0_c1 = c[0][0] * c[1][1] - c[1][0] * c[0][1];
		det_c0_x  = c[0][0] * x[1]    - c[0][1] * x[0];
		det_x_c1  = x[0]    * c[1][1] - x[1]    * c[0][1];

		/* Finally, derive alpha values	*/
		if (mathUtils.equalToZeroWithinEpsilon(det_c0_c1, deriv_epsilon)) {
			det_c0_c1 = (c[0][0] * c[1][1]) * 10 * deriv_epsilon;
		}
		if (mathUtils.equalToZeroWithinEpsilon(det_c0_c1, deriv_epsilon)) {	// If its STILL == 0 (duplicate ctpt)    sss 9/95
			det_c0_c1 = 10 * deriv_epsilon;
		}
		alpha_l = det_x_c1 / det_c0_c1;
		alpha_r = det_c0_x / det_c0_c1;

		outCurve[0] = V2.clone(inPts[inStartIndex]);
		outCurve[3] = V2.clone(inPts[inEndIndex]);
		outCurve[1] = V2.clone(inTanStart);
		outCurve[2] = V2.clone(inTanEnd);
		 /*  If alpha negative, use the Wu/Barsky heuristic (see text) */
		if (alpha_l < 0.0 || alpha_r < 0.0) {
			distance = V2.distance(inPts[inEndIndex], inPts[inStartIndex]) * mathUtils.oneThird;
			alpha_l = distance;
			alpha_r = distance;
		}

		/*  First and last control points of the Bezier curve are */
		/*  positioned exactly at the first and last data points */
		/*  Control points 1 and 2 are positioned an alpha distance out */
		/*  on the tangent vectors, left and right, respectively */
		V2.scale(alpha_l, outCurve[1], outCurve[1]);
		V2.scale(alpha_r, outCurve[2], outCurve[2]);
		V2.add(outCurve[1], outCurve[0], outCurve[1]);
		V2.add(outCurve[2], outCurve[3], outCurve[2]);

		if (inForceIncreasingXValues) {
			curveUtils.forceBezierIncreasingXValues(outCurve);
		}

		return outCurve;
	}

	/*
	 *  chordLengthParameterize :
	 *	Assign parameter values to digitized points 
	 *	using relative distances between points.
	 */
	function chordLengthParameterize(inPts, inStartIndex, inEndIndex) {
		var i, scale, distance, first_pt, second_pt, outValues;

		outValues = [];
		outValues[0] = 0.0;
		for (i = inStartIndex + 1; i <= inEndIndex; i += 1) {
			first_pt = inPts[i];
			second_pt = inPts[i - 1];

			distance = V2.distance(first_pt, second_pt) * mathUtils.oneThird;

			outValues[i - inStartIndex] = outValues[i - inStartIndex - 1] + distance;
		}

		if (outValues[inEndIndex - inStartIndex] !== 0) {
			scale = 1.0 / outValues[inEndIndex - inStartIndex];
		} else {
			scale = 0.0;
		}

		for (i = inStartIndex + 1; i <= inEndIndex; i += 1) {
			outValues[i - inStartIndex] = outValues[i - inStartIndex] * scale;
		}

		return outValues;
	}

	// Returns array of max error and split point.
	function computeMaxError(inPts, inStartIndex, inEndIndex, inCurve, inUValues) {
		var	i, max_dist, dist, pt, vec = [], split_point;

		split_point = (inEndIndex + inStartIndex + 1) / 2;
		max_dist = 0.0;

		for (i = inStartIndex + 1; i < inEndIndex; i += 1) {
			pt = curveUtils.evaluate2DCurve(inCurve, inUValues[i - inStartIndex]);

			V2.subtract(pt, inPts[i], vec);
			dist = V2.dot(vec, vec);

			if (dist >= max_dist) {
				max_dist = dist;
				split_point = i;
			}
		}

		return [max_dist, split_point];
	}

	function newtonRaphsonRootFind(inCurve, inData, inU) {
		var	numerator, denominator, q_u, q1_u, q2_u, u_prime; /* u evaluated at Q, Q', & Q''	*/

		// Compute Q(u), Q'(u) and Q''(u)
		q_u = curveUtils.evaluate2DCurve(inCurve, inU);
		q1_u = curveUtils.evaluate2DCurveDeriv(inCurve, inU);
		q2_u = curveUtils.evaluate2DCurveDeriv2(inCurve, inU);

		/* Compute f(u)/f'(u) */
		numerator = (q_u[0] - inData[0]) * (q1_u[0]) + (q_u[1] - inData[1]) * (q1_u[1]);
		denominator = (q1_u[0]) * (q1_u[0]) + (q1_u[1]) * (q1_u[1]) + (q_u[0] - inData[0]) * (q2_u[0]) + (q_u[1] - inData[1]) * (q2_u[1]);

		/* u = u - f(u)/f'(u) */
		if (denominator === 0) {	// added sss 9/95 to deal w/ coincident vertices
			u_prime = inU;
		} else {
			u_prime = inU - (numerator / denominator);
		}

		return u_prime;
	}

	function reparameterize(inPts, inStartIndex, inEndIndex, inUValues, inCurve, outUValues) {
		var nPts = inEndIndex - inStartIndex + 1, i;

		for (i = inStartIndex; i <= inEndIndex; i += 1) {
			outUValues[i - inStartIndex] = newtonRaphsonRootFind(inCurve, inPts[i], inUValues[i - inStartIndex]);
		}
	}

	function computeCenterTangent(inPts, inCenterIndex) {
		var v1 = [], v2 = [], tanCenter = V2.create();
		V2.subtract(inPts[inCenterIndex - 1], inPts[inCenterIndex], v1);
		V2.subtract(inPts[inCenterIndex], inPts[inCenterIndex + 1], v2);

		tanCenter[0] = (v1[0] + v2[0]) * 0.5;
		tanCenter[1] = (v1[1] + v2[1]) * 0.5;

		V2.normalize(tanCenter, tanCenter);
		return tanCenter;
	}

	function fitCubic(inPts, inStartIndex, inEndIndex, inTanStart, inTanEnd, inRecurse, inErrorTolerance, inMaxIterations, inForceIncreasingXValues, outCurves) {
		var	bez_pts = [], u, uPrime, temp, maxErrorSplit, nPts, iterationError, tanCenter = [], i, distance, curve, outMaxError = 0, iPt, xMid;

		iterationError = inErrorTolerance * inErrorTolerance;
		nPts = inEndIndex - inStartIndex + 1;

		/*  Use heuristic if region only has two points in it:
			Set tangents to 1/3 distance between endpoints. */
		if (nPts === 2) {
			distance = V2.distance(inPts[inEndIndex], inPts[inStartIndex]) * mathUtils.oneThird;

			bez_pts[0] = V2.clone(inPts[inStartIndex]);
			bez_pts[3] = V2.clone(inPts[inEndIndex]);

			bez_pts[1] = V2.clone(inTanStart);
			bez_pts[2] = V2.clone(inTanEnd);

			V2.normalize(bez_pts[1], bez_pts[1]);
			V2.normalize(bez_pts[2], bez_pts[2]);

			V2.scale(distance, bez_pts[1], bez_pts[1]);
			V2.scale(distance, bez_pts[2], bez_pts[2]);

			V2.add(bez_pts[1], bez_pts[0], bez_pts[1]);
			V2.add(bez_pts[2], bez_pts[3], bez_pts[2]);

			if (inForceIncreasingXValues) {
				curveUtils.forceBezierIncreasingXValues(bez_pts);
			}

			outCurves.push(bez_pts);
		} else {
			//  Parameterize points, and attempt to fit curve
			u = chordLengthParameterize(inPts, inStartIndex, inEndIndex);

			curve = generateBezier(inPts, inStartIndex, inEndIndex, u, inTanStart, inTanEnd, inForceIncreasingXValues);

			//  Find max deviation of points to fitted curve
			maxErrorSplit = computeMaxError(inPts, inStartIndex, inEndIndex, curve, u);

			outMaxError = maxErrorSplit[0];

			if (outMaxError < inErrorTolerance) {
				outCurves.push(curve);
			} else {
				//  If error not too large, try some reparameterization and iteration
				if (outMaxError < iterationError || !inRecurse) {
					for (i = 0; i < inMaxIterations; i += 1) {
						reparameterize(inPts, inStartIndex, inEndIndex, u, curve, uPrime);
						curve = generateBezier(inPts, inStartIndex, inEndIndex, uPrime, inTanStart, inTanEnd, inForceIncreasingXValues);
						maxErrorSplit = computeMaxError(inPts, inStartIndex, inEndIndex, curve, uPrime);
						outMaxError = maxErrorSplit[0];
						if (outMaxError < inErrorTolerance) {
							outCurves.push(curve);
							break;
						}
						temp = u;
						u = uPrime;
						uPrime = temp;
					}
				}

				if (inRecurse && outMaxError >= inErrorTolerance) {
					// Fitting failed -- split at max error point and fit recursively
					tanCenter = computeCenterTangent(inPts, maxErrorSplit[1]);

					outMaxError = fitCubic(inPts, inStartIndex, maxErrorSplit[1], inTanStart, tanCenter, true, inErrorTolerance, inMaxIterations, inForceIncreasingXValues, outCurves);

					V2.negate(tanCenter, tanCenter);

					outMaxError = fitCubic(inPts, maxErrorSplit[1], inEndIndex, tanCenter, inTanEnd, true, inErrorTolerance, inMaxIterations, inForceIncreasingXValues, outCurves);
				} else {
					// return the single curve, even if its error is greater than desired
					outCurves.push(curve);
				}
			}
		}
		return outMaxError;
	}

	return {
		// Fit a curve to an array of points.
		fitCurvesToPoints : function (inPts, inErrorTolerance, inMaxIterations, inForceIncreasingXValues) {
			var	tanStart = V2.create(), tanEnd = V2.create(), firstIndex = 0,
				maxError = 0, numPts = inPts.length, curves, outCurves = [], i, j, x;

			if (inPts === undefined || numPts <= 1) {
				throw new Error("Invalid points to fit to curve.");
			}

			V2.subtract(inPts[1], inPts[0], tanStart);
			V2.normalize(tanStart, tanStart);

			V2.subtract(inPts[numPts - 2], inPts[numPts - 1], tanEnd);
			V2.normalize(tanEnd, tanEnd);

			maxError = fitCubic(inPts, 0, numPts - 1, tanStart, tanEnd, true,
								inErrorTolerance, inMaxIterations, inForceIncreasingXValues, outCurves);
			return outCurves;
		}
	};
});
