/*global define, require */
/*jslint white: true */

/*
	This class provides the 2d vector implementation.
*/

define	(["lodash"],
function (lodash) {
	'use strict';

	var	
	// vec2 objects will inherit array methods using prototype chain injection.
	// see rationale: http://perfectionkills.com/how-ecmascript-5-still-does-not-allow-to-subclass-an-array/to type injection works well.
		vec2_prototype = [],
		proto = "__proto__";
	
	function vec2 (object) {
		object = object || [];
		object[proto] = vec2_prototype;
		return  object;
	}

	function initWithEntries (v0, v1, result) {
		result = result || [];
		result[0] = v0;
		result[1] = v1;
		return result;
	}
	vec2.initWithEntries = initWithEntries;

	function initWithArray (arr, result) {
		result = result || [];
		result[0] = arr[0];
		result[1] = arr[1];
		return result;
	}
	vec2.initWithArray = initWithArray;

	/**
	 * Create vector from entries.
	 * @deprecated ues initWithEntries instead
	 */
	function create  (x0, y0, result) {
		result = result || [];
		return initWithEntries(x0 || 0, y0 || 0);
	}
	vec2.create = create;

	function clone  (vec) {
		return vec.slice(0, 2);
	}
	vec2.clone = clone;

	function negate  (vec, result) {
		result = result || [];
		result[0] = -vec[0];
		result[1] = -vec[1];

		return result;
	}
	vec2.negate = negate;

	// a x + y = a x plus y
	function axpy (a, x, y, result) {
		result = result || [];
		result[0] = a * x[0] + y[0];
		result[1] = a * x[1] + y[1];
		return result;
	}
	vec2.axpy = axpy;

	// x / y = x dividedby y
	function xdy (x, y, result) {
		result = result || [];
		result[0] = x[0] / y[0];
		result[1] = x[1] / y[1];
		return result;
	}
	vec2.xdy = xdy;

	// x * y = x multipliedby y
	function xmy (x, y, result) {
		result = result || [];
		result[0] = x[0] * y[0];
		result[1] = x[1] * y[1];
		return result;
	}
	vec2.xmy = xmy;

	function add (vecA, vecB, result) {
		result = result || [];
		result[0] = vecA[0] + vecB[0];
		result[1] = vecA[1] + vecB[1];
		return result;
	}
	vec2.add = add;

	function scale  (scalar, vec, result) {
		result = result || [];
		result[0] = scalar * vec[0];
		result[1] = scalar * vec[1];
		return result;
	}
	vec2.scale = scale;

	function subtract (vecA, vecB, result) {
		result = result || [];
		result[0] = vecA[0] - vecB[0];
		result[1] = vecA[1] - vecB[1];
		return result;
	}
	vec2.subtract = subtract;

	function dot  (vecA, vecB) {
		return vecA[0] * vecB[0] + vecA[1] * vecB[1];
	}
	vec2.dot = dot;

	function magnitudeSquared  (vec) {
		var x = vec[0], y = vec[1];
		return x * x + y * y;
	}
	vec2.magnitudeSquared = magnitudeSquared;

	function magnitude  (vec) {
		var x = vec[0], y = vec[1];
		return Math.sqrt(x * x + y * y);
	}
	vec2.magnitude = magnitude;

	function normalize  (vec, result, len0) {
		var len = len0 || 1,
			mag = magnitude(vec),
			ilen = 1;
		
		if (mag) {
			ilen = len / mag;
		}

		result = result || [];
		result[0] = vec[0] * ilen;
		result[1] = vec[1] * ilen;
		return result;
	}
	vec2.normalize = normalize;

	function distanceSquared  (vecA, vecB) {
		return magnitudeSquared(subtract(vecB, vecA));
	}
	vec2.distanceSquared = distanceSquared;

	function distance  (vecA, vecB) {
		return Math.sqrt(distanceSquared(vecA, vecB));
	}
	vec2.distance = distance;

	function equals  (v0, v1) {
		return v0[0] === v1[0] && v0[1] === v1[1];
	}
	vec2.equals = equals;

	function equalsApproximately  (v0, v1, eps0) {
		// default to float eps -- 24 bits of precision: 2 ^ -23
		var eps = eps0 || Math.pow(2, -23);

		return	Math.abs(v0[0] - v1[0]) < eps 
		&&		Math.abs(v0[1] - v1[1]) < eps;
	}
	vec2.equalsApproximately = equalsApproximately;

	function transformLinear (mat3, vec, result) {
		var x = vec[0], y = vec[1];

		result = result || [];
		result[0] = x * mat3[0] + y * mat3[3];
		result[1] = x * mat3[1] + y * mat3[4];

		return result;
	}
	vec2.transformLinear = transformLinear;

	function transformAffine (mat3, vec, result) {
		var x = vec[0], y = vec[1];

		result = result || [];
		result[0] = x * mat3[0] + y * mat3[3] + mat3[6];
		result[1] = x * mat3[1] + y * mat3[4] + mat3[7];

		return result;
	}
	vec2.transformAffine = transformAffine;

	function transformProjective (mat3, vec, result) {
		var x = vec[0], y = vec[1],
			invw = 1 / (x * mat3[2] + y * mat3[5] + mat3[8]);

		result = result || [];
		result[0] = invw * (x * mat3[0] + y * mat3[3] + mat3[6]);
		result[1] = invw * (x * mat3[1] + y * mat3[4] + mat3[7]);

		return result;
	}
	vec2.transformProjective = transformProjective;

	/*
	 * The following vector methods will be available as methods on vec2 objects (instances of vec2 type).
	 * They are primarily syntactic sugar which delegates to the above free functions.
	 * Use them as convenient but when efficiency really matters use free functions instead.
	 */

	function delegateTo (invoke) {
		return function (name) {
			var method = vec2[name];
			if (! method) { throw new Error("vec2: method not found."); }
			if (vec2_prototype[name]) { throw new Error("vec2: method already exists."); }
			vec2_prototype[name] = invoke(method);
		};
	}

	vec2_prototype.init = function (v0, v1) {
		return initWithEntries(v0, v1, this);
	};

	vec2_prototype.negate = function () {
		return negate(this, vec2());
	};

	vec2_prototype.clone = function () {
		return vec2(this.slice(0, 2));
	};

	lodash.forEach(["xdy", "xmy", "add", "subtract"], 
		delegateTo(function (method) {
			return function (arg) {
				return method(this, arg, vec2());
			};
		}));

	vec2_prototype.axpy = function (a, y) {
		return axpy(a, this, y, vec2());
	};

	vec2_prototype.normalize = function (len0) {
		return normalize(this, vec2(), len0);
	};

	lodash.forEach(["scale", "transformLinear", "transformAffine", "transformProjective"],
		delegateTo(function (method) {
			return function (arg) {
				return method(arg, this, vec2());
			};
		}));


	lodash.forEach(["dot", "magnitude", "magnitudeSquared", "distanceSquared", "distance", "equals", "equalsApproximately"],
		delegateTo(function (method) {
			return function (arg) {
				return method(this, arg);
			};
		}));

	return vec2;
});
