/*global define, require */
/*jslint white: true */

/*
	curve utils:

	This file implements the 2d spatial time graph object.
*/

define ([	"src/math/curveUtils",	"src/math/curveDistanceEval",	"src/math/TimeGraph",	"src/math/mathUtils",
			"src/utils"],
function(	curveUtils,				curveDistanceEval,				TimeGraph,				mathUtils,
			utils) {
	'use strict';

	return function SpatialTimeGraph(inCurves) {
		var that = this;

		that.distancetoTGraph = undefined;
		that.virtualTGraph = undefined;	// optionally defined.  Remaps t (in:0-1, out:0-1), used to control velocity.
		that.spatialCurves = [];

		that.appendCurves = function (inCurve) {
			var	i;
			for (i = 0; i < inCurve.length; i += 1) {
				this.spatialCurves.push(utils.clone(true, [], inCurve[i]));
			}
			this.distancetoTGraph = curveDistanceEval.calculateDistanceTimeGraph(inCurve, this.distancetoTGraph);
		};

		that.appendCurve = function (inCurve) {
			this.appendCurves([inCurve]);
		};

		that.appendLine = function (inLine) {
			this.appendCurve([[inLine[0], inLine[0], inLine[1], inLine[1]]]);
		};

		that.getTotalDistance = function () {
			var		d = 0;
			if (this.distancetoTGraph !== undefined) {
				d = this.distancetoTGraph.xMax;
			}
			return d;
		};

		that.evaluateXYFromDistance = function (inD) {
			var	tSpatial, result, spatialCurveIndex;

			if (this.distancetoTGraph !== undefined) {
				tSpatial = this.distancetoTGraph.evaluateYFromX(inD);
				spatialCurveIndex = Math.floor(tSpatial);
				tSpatial = tSpatial - spatialCurveIndex;
				if (spatialCurveIndex >= this.spatialCurves.length) {
					spatialCurveIndex = this.spatialCurves.length - 1;
					tSpatial = 1.0;
				}
				result = curveUtils.evaluate2DCurve(this.spatialCurves[spatialCurveIndex], tSpatial);
			}
			return result;
		};

		that.evaluate = that.evaluateXYFromDistance;

		// optionally defined.  Maps 0-1 to 0-1, used to control velocity.
		that.setVirtualTGraph = function (inTimeGraph) {
			this.distancetoTGraph.setVirtualTGraph(inTimeGraph);
		};

		that.getVirtualTGraph = function () {
			return this.distancetoTGraph.getVirtualTGraph();
		};

		that.getT = function (inT, inLoopType) {
			return this.distancetoTGraph.getT(inT, inLoopType);
		};

		that.evaluateXYFromT = function (inT, inLoopType) {
			var	t = this.getT(inT, inLoopType), result, x;

			if (this.distancetoTGraph !== undefined) {
				x = mathUtils.lerp(this.distancetoTGraph.getXMin(), this.distancetoTGraph.getXMax(), t);
				result = this.evaluateXYFromDistance(x);
			}
			return result;
		};

		that.clone = function () {
			var newTimeGraph = utils.clone(true, {}, this);
			return newTimeGraph;
		};

		if (inCurves !== undefined) {
			that.appendCurves(inCurves);
		}

		return that;
	};
});
