
 /*global define, console, nmlImpl, require */
/*jslint white: true */

/*
	This class provides the an event handler to zoom and scroll a view.

	Graphs:
		"Mouse/Path/Distance"
		"Mouse/Path"
		"Mouse/Position"	(virtual, for accessing the value and based on Mouse/Path/Distance and Mouse/Path)
*/


define(["src/math/curveUtils", "src/events/keyCodes", "src/utils", "src/build/Builder"],
function (curveUtils, keyCodes, utils, Builder) {
	'use strict';

	return function EventGraphs(inId, inStage) {
		var that = this, getTimeFunc = nmlImpl.getCurrentTime, strEndsWith, getVirtualGraphValue, builder = new Builder();

		that.id = inId;

		that.resetGraphs = function () {
			this.graphs = {};
			this.incrementalGraphs = {};
			this.lastPathUpdateTime = undefined;
			this.lastPathUpdateDistance = 0;
		};

		that.resetGraphs();

		strEndsWith = function (str, suffix) {
			return str.indexOf(suffix, str.length - suffix.length) !== -1;
		};

		getVirtualGraphValue = function (inId, inT, inX0, inEvalObj) {
			var inIdParts, result;

			inIdParts = inId.split("Position");
			if (inIdParts.length === 2) {
				result = getVirtualGraphValue(inIdParts[0] + "Path/Distance", inT, inX0, inEvalObj);
				if (result !== undefined) {
					return getVirtualGraphValue(inIdParts[0] + "Path", inT, result, inEvalObj);
				}
			} else {
				return inEvalObj.getGraphValue(inId, inT, inX0);
			}
		};

		that.getEvaulator = function (behaviorInstanceId0, inTimeSpan0, inEventGraphFilePath0) {
			var result = {};

			// Handles only non-virtual graphs
			result.evaluatesAtGlobalTime = function () {
				return !behaviorInstanceId0 || !inEventGraphFilePath0;
			};

			// Handles only non-virtual graphs
			result.getTimeSpan = function (inTimeStack) {
				return this.evaluatesAtGlobalTime() ? inTimeStack.getRootTimeSpan() :
														(inTimeSpan0 ? inTimeSpan0 : inTimeStack.getLocalTimeSpan());
			};

			// Handles only non-virtual graphs
			result.getGraphValue = function (inGraphId, inTime, inX0) {
				return nmlImpl.getEventGraphValue(inId, behaviorInstanceId0, inEventGraphFilePath0, inGraphId, inTime, inX0);
			};

			// Handles only non-virtual graphs
			result.getGraphValueExtent = function (inGraphId, inXMin, inXMax) {
				return nmlImpl.getEventValueExtent(inId, behaviorInstanceId0, inEventGraphFilePath0, inGraphId, inXMin, inXMax);
			};

			// Handles virtual and non-virtual graphs
			result.getValue = function (inGraphId, inTime) {
				return getVirtualGraphValue(inGraphId, inTime, undefined, this);
			};

			return result;
		};

		that.getNonPathDistanceGraphs = function () {
			var g, pathKey = "Path", pathDistanceKey = "Path/Distance", distGraphKey, result = {};
			for (g in this.graphs) {
				if (this.graphs.hasOwnProperty(g)) {
					if (strEndsWith(g, pathKey)) {
						distGraphKey = g + "/Distance";
						result[g] = { pathGraph: true, graph: this.graphs[g], distGraphKey: distGraphKey, distGraph: this.graphs[distGraphKey] };
					} else if (!strEndsWith(g, pathDistanceKey)) {
						result[g] = { pathGraph: false, graph: this.graphs[g] };
					}
				}
			}

			return result;
		};

		that.deleteOld = function (inOldTime) {
			var nonPathDistGraphs = this.getNonPathDistanceGraphs(), g, graphData, dist;
			for (g in nonPathDistGraphs) {
				if (nonPathDistGraphs.hasOwnProperty(g)) {
					graphData = nonPathDistGraphs[g];

					if (graphData.pathGraph) {
						if (graphData.distGraph) {
							graphData.distGraph.deleteOld(inOldTime);
							dist = this.getGraphValue(graphData.distGraphKey, inOldTime);
							if (dist !== undefined) {
								graphData.graph.deleteOld(dist);
							}
						}
					} else {
						graphData.graph.deleteOld(inOldTime);
					}
				}
			}
		};

		that.update = function (inTime, inOldTime0) {
			nmlImpl.updateEventGraph(that.id, inTime, inOldTime0);
		};

		that.getGraphLastTime = function (inId, inT) {
			var g = this.graphs[inId], result = inT;
			if (g !== undefined) {
				result = g.getXMax();
				if (result === undefined) {
					result = inT;
				}
			}
			return result;
		};

		that.getGraphLastValue = function (inId, inT) {
			var g = this.graphs[inId], result, lastTime = this.getGraphLastTime(inId, inT);
			if (g !== undefined) {
				result = g.evaluate(lastTime);
			}
			return result;
		};

		that.getGraphKeys = function (inKeyPrefix0) {
			// Returns an array of graphs
			var g, grphs = [], matchB;
			for (g in this.graphs) {
				if (this.graphs.hasOwnProperty(g)) {
					matchB = (inKeyPrefix0 === undefined) || g.indexOf(inKeyPrefix0) === 0;
					if (matchB) {
						grphs.push(g);
					}
				}
			}
			return grphs;
		};

		that.getGraph = function (inId) {
			return this.graphs[inId];
		};

		that.hasNonVirtualGraph = function (inId) {
			return this.graphs.hasOwnProperty(inId);
		};

		that.hasGraph = function (inId) {
			var inIdParts, result;

			inIdParts = inId.split("Position");
			if (inIdParts.length === 2) {
				result = this.hasNonVirtualGraph(inIdParts[0] + "Path") && this.hasNonVirtualGraph(inIdParts[0] + "Path/Distance");
			} else {
				result = this.hasNonVirtualGraph(inId);
			}
			return result;
		};

		that.getValueExtent = function (inId, inMinX, inMaxX) {
			var result, graph;
			graph = this.getGraph(inId);

			if (graph !== undefined) {
				result = graph.getValueExtent(inMinX, inMaxX);
			}

			return result;
		};

		that.getGraphValue = function (inId, inT, inX0) {
			var result, graph, graphXMax;
			graph = this.getGraph(inId);

			if (graph !== undefined) {
				inT = inT || getTimeFunc();
				inX0 = inX0 || inT;
				if (typeof graph.getXMax === 'function') {
					graphXMax = graph.getXMax();
					if (inX0 > graphXMax) {
						inX0 = graphXMax; // Get the latest value available.
					}
				}
				result = graph.evaluate(inX0);
			}

			return result;
		};

		that.getValue = function (inGraphId, inTime, inX0) {
			return getVirtualGraphValue(inGraphId, inTime, inX0, this);
		};

		that.getMaxValue = function (inId) {
			var result, graph = this.getGraph(inId);

			if (graph !== undefined) {
				result = graph.getYMax();
			}
			return result;
		};

		that.getMinValue = function (inId) {
			var result, graph = this.getGraph(inId);

			if (graph !== undefined) {
				result = graph.getYMin();
			}
			return result;
		};

		that.createTimeGraph = function (inGraphId) {
			if (this.graphs[inGraphId] === undefined) {
				this.graphs[inGraphId] = builder.newTimeGraph(undefined);
			}
		};

		that.sustainGraphValue = function (inGraphId, inTime) {
			this.createTimeGraph(inGraphId);
			if (this.hasGraph(inGraphId)) {
				var graph = this.getGraph(inGraphId);
				graph.sustainValue(inTime);
			}
		};

		function appendCurveToGraph(inGraph, inCurve) {
			inGraph.appendCurve(inCurve);
		}

		function appendLineToGraph(inGraph, inLine) {
			inGraph.appendLine(inLine);
		}

		that.setNewHoldValue = function (inId, inTime, inNewValue) {
			var lastValue, lastTime, holdTime, g, segment, oneNanoSecond = 0.000001;
			holdTime = this.getGraphLastTime(inId, undefined);
			if (holdTime !== undefined) {
				holdTime = Math.max(holdTime, inTime - oneNanoSecond);
			}
			this.sustainGraphValue(inId, holdTime);
			lastValue = this.getGraphLastValue(inId, inTime) || inNewValue;
			if (lastValue !== undefined) {
				lastTime = this.getGraphLastTime(inId, inTime);
				g = this.getGraph(inId);
				segment = [[lastTime, lastValue], [inTime, inNewValue]];
				appendLineToGraph(g, segment);
			}
		};

		that.incrementNewHoldValue = function (inId, inTime) {
			var inNewValue = 1, g;
			g = this.getGraph(inId);
			if (g) {
				inNewValue = g.getYMax() + 1;
			}
			this.setNewHoldValue(inId, inTime, inNewValue);
		};

		that.createSpatialTimeGraph = function (inGraphId) {
			if (this.graphs[inGraphId] === undefined) {
				this.graphs[inGraphId] = builder.newSpatialTimeGraph(undefined);
			}
		};

		that.createIncrementalGraph = function (inGraphId, inPos, inSpatialGraphB) {
			var lastCurve;
			if (this.incrementalGraphs[inGraphId] === undefined) {
				this.incrementalGraphs[inGraphId] = new curveUtils.IncrementalLineToCurveCreator(!inSpatialGraphB);
			}
			lastCurve = this.incrementalGraphs[inGraphId].nextPoint(inPos);

			if (lastCurve !== undefined) {
				if (inSpatialGraphB) {
					this.createSpatialTimeGraph(inGraphId);
				} else {
					this.createTimeGraph(inGraphId);
				}
			}
		};

		that.updateIncrementalGraph = function (inGraphId, inPos, inSpatialGraphB) {
			var lastCurve;
			this.createIncrementalGraph(inGraphId, inPos, inSpatialGraphB);
			lastCurve = this.incrementalGraphs[inGraphId].getPreviousCurve();

			if (lastCurve !== undefined) {
				appendCurveToGraph(this.getGraph(inGraphId), lastCurve);
			}
		};

		that.removeGraph = function (inGraphId) {
			// This doesn't work with virtual graphs and currently doesn't need to.  -jacquave
			if (this.graphs[inGraphId] !== undefined) {
				delete this.graphs[inGraphId];
			}
		};

		that.updateSpatial2D = function (inGraphIdPrefix, inTime, inPos) {
			var d = 0, distSegment, distGraphId, pathGraphId, pathCurve;

			distGraphId = inGraphIdPrefix + "/Path/Distance";
			pathGraphId = inGraphIdPrefix + "/Path";

			this.createTimeGraph(distGraphId);

			this.createIncrementalGraph(pathGraphId, inPos, true);
			pathCurve = this.incrementalGraphs[pathGraphId].getPreviousCurve();

			if (pathCurve !== undefined) {
				this.lastPathUpdateDistance = this.getGraph(pathGraphId).getTotalDistance();
				appendCurveToGraph(this.getGraph(pathGraphId), pathCurve);
				d = this.getGraph(pathGraphId).getTotalDistance();
			}

			// TODO: remove lastPathUpdateTime?
			this.lastPathUpdateTime = this.getGraphLastTime(distGraphId, inTime);
			distSegment = [[this.lastPathUpdateTime, this.lastPathUpdateDistance], [inTime, d]];
			appendLineToGraph(this.getGraph(distGraphId), distSegment);

			this.lastPathUpdateTime = inTime;
		};

		// for non-spatial
		that.update1D = function (inGraphId, inTime, inValue, inSustainOldValueB0) {
			var valueSegment, lastTime, lastValue;

			this.createTimeGraph(inGraphId);

			lastTime = this.getGraphLastTime(inGraphId, inTime);
			lastValue = this.getGraphLastValue(inGraphId, inTime);
            if (lastValue === undefined) {
                lastValue = inValue;
            }

			if (inSustainOldValueB0) {
				valueSegment = [[lastTime, lastValue], [inTime, lastValue]];
				appendLineToGraph(this.getGraph(inGraphId), valueSegment);
				valueSegment = [[inTime, lastValue], [inTime, inValue]];
				appendLineToGraph(this.getGraph(inGraphId), valueSegment);
			} else {
				valueSegment = [[lastTime, lastValue], [inTime, inValue]];
				appendLineToGraph(this.getGraph(inGraphId), valueSegment);
			}
		};
		
		that.publishGraphs = function (inGraphDataArray) {
			var i, g;
			for (i = 0; i < inGraphDataArray.length; i += 1) {
				g = inGraphDataArray[i];
				this[g.updateFuncName](g.graphKey, g.time, g.value, g.sustainOldValue);
			}
		};
		
		that.unpublishGraphs = function (inGraphDataKeysArray) {
			var i;
			for (i = 0; i < inGraphDataKeysArray.length; i += 1) {
				this.removeGraph(inGraphDataKeysArray[i]);
			}
		};

		that.updateMouseWheelGraph = function (inGraphId, inTime, inDeltaValue) {
			var lastValue;
			that.sustainGraphValue(inGraphId, inTime);
			lastValue = that.getGraphLastValue(inGraphId, inTime) || 0;
			that.setNewHoldValue(inGraphId, inTime, lastValue + inDeltaValue);
		};

		that.isKeyPressed = function (inKeyCodes, inT) {
			var result = false, i;

			for (i = 0; inKeyCodes && !result && i < inKeyCodes.length; i += 1) {
				result = (this.getValue(keyCodes.getKeyGraphId(inKeyCodes[i]), inT) || 0) > 0;
			}

			return result;
		};

		function mouseEventFunc(inEventGraphs) {
			var that = {};

			that.handleEvent = function (event) {
				var pos = event.position.slice(0), t = getTimeFunc();
				if (event.type === "out") {
					inEventGraphs.setNewHoldValue("Mouse/Over", t, 0);
				} else {
					if (event.type === "down") {
						if (event.button === "left") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Left", t, 1);
						} else if (event.button === "right") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Right", t, 1);
						} else if (event.button === "middle") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Middle", t, 1);
						}
					} else if (event.type === "up") {
						if (event.button === "left") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Left", t, 0);
						} else if (event.button === "right") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Right", t, 0);
						} else if (event.button === "middle") {
							inEventGraphs.setNewHoldValue("Mouse/Down/Middle", t, 0);
						}
					} else if (event.type === "over") {
						inEventGraphs.sustainGraphValue("Mouse/Path/Distance", t);
						inEventGraphs.setNewHoldValue("Mouse/Over", t, 1);
					} else if (event.type === "wheel" && event.wheelDelta !== undefined) {
						inEventGraphs.updateMouseWheelGraph("Mouse/Wheel/X", t, event.wheelDelta[0]);
						inEventGraphs.updateMouseWheelGraph("Mouse/Wheel/Y", t, event.wheelDelta[1]);
					}

					inEventGraphs.updateSpatial2D("Mouse", t, pos);
				}
				return false;
			};

			return that;
		}

		function updateModKey(inEventKeyValue, inKeyCode, inT, ioEventGraphs) {
			if (inEventKeyValue) {
				ioEventGraphs.incrementNewHoldValue(keyCodes.getKeyGraphId(inKeyCode), inT);
			} else {
				ioEventGraphs.setNewHoldValue(keyCodes.getKeyGraphId(inKeyCode), inT, 0);
			}
		}

		function updateModKeys(inEvent, inT, ioEventGraphs) {
			updateModKey(inEvent.shiftKey, keyCodes.shiftKey, inT, ioEventGraphs);
			updateModKey(inEvent.ctrlKey, keyCodes.cmdKey, inT, ioEventGraphs);
			updateModKey(inEvent.altKey, keyCodes.altKey, inT, ioEventGraphs);
		}

		function keyEventFunc(inEventGraphs) {
			var that = {};

			that.handleEvent = function (event) {
				var handledB = false, t = getTimeFunc(), graphId, keyGraphs, keyPrefix, g;

				if (event.type === "keydown" || event.type === "keyup") {
					graphId = keyCodes.getKeyGraphId(event.keyCode);
					if (event.type === "keydown") {
						inEventGraphs.incrementNewHoldValue(graphId, t);
					} else {
						inEventGraphs.setNewHoldValue(graphId, t, 0);
					}
				} else if (event.type === "gain focus") {
					keyPrefix = keyCodes.getKeyGraphPrefix();
					// Clear all keyboard based event graphs to zero.
					keyGraphs = inEventGraphs.getGraphKeys(keyPrefix);

					for (g in keyGraphs) {
						if (keyGraphs.hasOwnProperty(g)) {
							inEventGraphs.setNewHoldValue(keyGraphs[g], t, 0);
						}
					}
					updateModKeys(event, t, inEventGraphs);
				} else if (event.type === "update mod keys") {
					updateModKeys(event, t, inEventGraphs);
				}
				// Perhaps we should update mod-keys when we gain focus?  Probably not needed...
				return handledB;
			};

			return that;
		}

		that.timer = function (inTime) {
			this.sustainGraphValue("Mouse/Path/Distance", inTime);
		};

		that.clone = function () {
			var newEventGraph = utils.clone(true, {}, this);
			return newEventGraph;
		};

		inStage.installKeyEventHandler(keyEventFunc(this).handleEvent);
		inStage.installMouseEventHandler(mouseEventFunc(this).handleEvent);

		return that;
	};
});
