/*global define, parseJSON */
/*jslint white: true */

/*
	serialize:

	This class provides ability to load artwork via JSON.
*/

define  ([  "src/build/Handle",	"src/build/Container",	"src/build/Image",	"src/build/Graphics",
			"src/build/Path",	"src/build/Color",		"src/build/Puppet",	"src/build/Layer",
			"src/utils",		"src/math/Mat3",		"src/math/Vec2",	"src/math/mathUtils"],
function(   Handle,             Container,              Image,              Graphics,
			Path,               Color,                  Puppet,				Layer,
			utils,				mat3,					vec2,				mathUtils) {
	'use strict';

	var loadChildrenFunc;

	function addLookupItem(inData, ioArray) {
		var item, n = inData.getName();
		if (ioArray.lookup === undefined) {
			ioArray.lookup = {};
		}
		item = ioArray.lookup[n];
		if (item !== undefined) {
			throw new Error("Invalid JSON duplicate item name: " + n);
		} else {
			ioArray.push(inData);
			if (n !== undefined) {
				ioArray.lookup[n] = ioArray.length - 1;
			}
		}
	}

	function getLookupItem(inName, inArray) {
		var idx, result;
		if (inArray.lookup !== undefined) {
			idx = inArray.lookup[inName];
			result = inArray[idx];
		}

		if (result === undefined) {
			throw new Error("Invalid JSON item lookup.");
		}
		return result;
	}

	function loadPaths(inData, outPaths) {
		var i, j, o, n, cmd, cmdName, f;
		if (inData.paths) {
			for (i = 0; i < inData.paths.length; i += 1) {
				o = new Path();
				n = inData.paths[i].name;
				if (n !== undefined) {
					o.setName(n);
				}

				if (inData.paths[i].cmds) {
					for (j = 0; j < inData.paths[i].cmds.length; j += 1) {
						cmd = inData.paths[i].cmds[j];
						cmdName = cmd[0];
						f = o[cmdName];
						if (typeof f === 'function') {
							if (cmdName === "bezierTo") {
								f.call(o, cmd[1][0], cmd[1][1], cmd[1][2]);
							} else {
								f.call(o, cmd[1]);
							}
						} else {
							throw new Error("Invalid JSON path command.");
						}
					}
				}
				addLookupItem(o, outPaths);
			}
		}
	}

	function loadNode(inData, outNode) {
		var n, m, a, mtxCmds, i, amt, abt;
		n = inData.name;
		if (n !== undefined) {
			outNode.setName(n);
		}
		m = inData.matrix;
		if (m !== undefined) {
			outNode.setMatrix(mat3.initWithArrayOfArrays(m));
		}
		mtxCmds = inData.matrixCmds;
		if (mtxCmds !== undefined && mtxCmds.length > 0) {
			m = outNode.getMatrix();
			for (i = 0; i < mtxCmds.length; i += 1) {
				if (mtxCmds[i][0] === "translate") {
					amt = mtxCmds[i][1].amount;
					if (amt !== undefined) {
						m = m.pretranslate(amt);
					}
				} else if (mtxCmds[i][0] === "scale") {
					amt = mtxCmds[i][1].amount;
					abt = mtxCmds[i][1].about;
					if (amt !== undefined) {
						if (abt === undefined) {
							abt = [0, 0];
						}
						m = m.pretranslate(vec2.negate(abt)).prescale(amt).pretranslate(abt);
					}
				} else if (mtxCmds[i][0] === "rotate") {
					amt = mtxCmds[i][1].amount;
					abt = mtxCmds[i][1].about;
					if (amt !== undefined) {
						if (abt === undefined) {
							abt = [0, 0];
						}
						m = m.pretranslate(vec2.negate(abt)).prerotate(mathUtils.rad(amt)).pretranslate(abt);
					}
				}
			}
			outNode.setMatrix(m);
		}
		a = inData.alpha;
		if (a !== undefined) {
			outNode.setAlpha(a);
		}
		if (inData.enabled !== undefined && inData.enabled === false) {
			outNode.setVisibleEnabled(false);
		}
	}

	function getPath(inPathNames, inPaths) {
		var p, pTemp, i;
		if (inPathNames.constructor === Array) {
			for (i = 0; i < inPathNames.length; i += 1) {
				pTemp = getLookupItem(inPathNames[i], inPaths);
				if (p === undefined) {
					p = pTemp.clone();
				} else {
					p.append(pTemp);
				}
			}
		} else {
			p = getLookupItem(inPathNames, inPaths);
		}
		return p;
	}

	function loadGraphics(inData, inPaths, outGraphics) {
		var i, j, o, cmd, cmdName, f, p, c;
		if (inData.graphics) {
			for (i = 0; i < inData.graphics.length; i += 1) {
				o = new Graphics();
				loadNode(inData.graphics[i], o);

				if (inData.graphics[i].cmds) {
					for (j = 0; j < inData.graphics[i].cmds.length; j += 1) {
						cmd = inData.graphics[i].cmds[j];
						cmdName = cmd[0];
						f = o[cmdName];
						if (typeof f === 'function') {
							// Unpack the params.
							c = cmd[1].color;
							c = new Color(c[0], c[1], c[2], c[3]);
							if (cmdName === 'fillRect') {
								f.call(o, c, cmd[1].rect);
							} else if (cmdName === 'fillPath') {
								p = getPath(cmd[1].path, inPaths);
								f.call(o, c, p);
							} else if (cmdName === 'strokeRect') {
								f.call(o, c, cmd[1].rect, cmd[1].width, cmd[1].endCaps, cmd[1].joins, cmd[1].miterLimit);
							} else if (cmdName === 'strokePath') {
								p = getPath(cmd[1].path, inPaths);
								f.call(o, c, p, cmd[1].width, cmd[1].endCaps, cmd[1].joins, cmd[1].miterLimit);
							}
						} else {
							throw new Error("Invalid JSON graphic command.");
						}
					}
				}
				addLookupItem(o, outGraphics);
			}
		}
	}

	function loadImages(inData, outImages) {
		var i, o;
		if (inData.images) {
			for (i = 0; i < inData.images.length; i += 1) {
				o = new Image(inData.images[i].uri);
				loadNode(inData.images[i], o);

				addLookupItem(o, outImages);
			}
		}
	}

	function loadChild(inChild, inGraphics, inImages, ioSymbols, ioContainers, outContainer) {
		var o;

		function addNode (data, node, parent) {
			loadNode(data, node);
			if (parent) { parent.addChild(node); }
		}

		if (inChild.cloneGraphic) {
			o = getLookupItem(inChild.cloneGraphic, inGraphics);
			o = o.clone();
			addNode(inChild, o, outContainer);
		} else if (inChild.cloneImage) {
			o = getLookupItem(inChild.cloneImage, inImages);
			o = o.clone();
			addNode(inChild, o, outContainer);
		} else if (inChild.cloneSymbol) {
			o = getLookupItem(inChild.cloneSymbol, ioSymbols);
			o = o.clone(true);
			addNode(inChild, o, outContainer);
		} else {
			o = new Container();
			addNode(inChild, o, outContainer);
			// setup children
			loadChildrenFunc(inChild.children, inGraphics, inImages, ioSymbols, ioContainers, o);
		}
		return o;
	}

	function loadChildren(inChildren, inGraphics, inImages, ioSymbols, ioContainers, outContainer) {
		var i;
		if (inChildren) {
			for (i = 0; i < inChildren.length; i += 1) {
				loadChild(inChildren[i], inGraphics, inImages, ioSymbols, ioContainers, outContainer);
			}
		}
	}

	function loadContainer(inData, inGraphics, inImages, inSymbols, ioContainers) {
		var o = loadChild(inData, inGraphics, inImages, inSymbols, ioContainers);
		if (ioContainers !== undefined) {
			addLookupItem(o, ioContainers);
		}
		return o;
	}

	loadChildrenFunc = loadChildren;

	function loadContainers(inData, inGraphics, inImages, inSymbols, outContainers) {
		var i, o, containers = [];
		if (inData.containers) {
			for (i = 0; i < inData.containers.length; i += 1) {
				o = loadContainer(inData.containers[i], inGraphics, inImages, inSymbols, containers);
				outContainers.push(o);
			}
		}
	}

	function makeHandleTree(data) {
		var i, n,
			tree = new Handle(data);

		if (data.children) {
			n = data.children.length;
			for (i = 0; i < n; i += 1) {
				tree.addChild(makeHandleTree(data.children[i]));
			}
		}

		return tree;
	}

	function makePuppetTree (data, graphics, images) {

		function makePuppet (data) {
			var args = {
					name : data.name,
					behaviors : data.behaviors,
					layers : data.layers.map(function (tLayer) {
						var source = null,
							matrix = mat3.initWithArrayOfArrays(tLayer.matrix);
						if (tLayer.tPuppet) {
							source = makePuppet(tLayer.tPuppet);
						} else if (tLayer.tSkin) {
							source = loadContainer(tLayer.tSkin, graphics, images);
						} else {
							utils.assert(false, "unknown layer type.");
						}

						var layer = new Layer(source, {
							name : tLayer.name, 
							bindingId : tLayer.bindingID, 
							tag : tLayer.tag,
							tagBackstageId : tLayer.tagBackstageId, 
							matLayer_Source : matrix, 
							motionMode : tLayer.motionMode,
							warpHash : tLayer.warpHash,
							bUserVisible : tLayer.bUserVisible
						});

						layer.setVisible(tLayer.bVisible);
						layer.setWarpWithParent(tLayer.bWarpWithParent);

						return layer;
					}),
					handle : makeHandleTree(data.handle),
					warpType : data.warpDomain,
					warpHash: data.warpHash,
					warpMeshExpansion: data.warpMeshExpansion
				
				},

				puppet = new Puppet(args);


			return puppet;
		}

		if (data.puppet) {
			return makePuppet(data.puppet);
		}

		return null;
	}

	function loadSymbols(inData, inGraphics, inImages, ioSymbols) {
		var i, o;
		if (inData.symbols) {
			for (i = 0; i < inData.symbols.length; i += 1) {
				o = loadContainer(inData.symbols[i], inGraphics, inImages, ioSymbols);
				addLookupItem(o, ioSymbols);
			}
		}
	}

	function loadSuccess(inData, inSuccessFunc) {
		var paths = [], graphics = [], images = [], symbols = [], containers = [],
			puppetTree;

		loadPaths(inData, paths);
		loadGraphics(inData, paths, graphics);
		loadImages(inData, images);
		loadSymbols(inData, graphics, images, symbols);
		loadContainers(inData, graphics, images, symbols, containers);
		puppetTree = makePuppetTree(inData, graphics, images);
		inSuccessFunc(containers, puppetTree);
	}

	function loadFailure(/*inData*/) {
		throw new Error("Unable to parse JSON file.");
	}

	function load(inURI, inSuccess, inFailure) {
		parseJSON(inURI, function (inData) { loadSuccess(inData, inSuccess); }, loadFailure);
	}

	return {
		load: load,
		loadData: loadSuccess
	};
});