/*global define */
/*jslint white: true */

/*
	This namespace encapsulates functions for transforming art tree into Planar Straight Line Graphs (PSLGs).
*/

define(["Container", "CommandPlayer", "Matrix2d"],
	function (Container, CommandPlayer, Matrix2d) {
		'use strict';
		function allocPSLG() {
			var pen = [0, 0],
				closer_node = false,
				nodes = [],
				segments = [];

			function addNode(point) {
				if (nodes.length > 0) {
					var last = nodes[nodes.length - 1];
					if (point[0] === last[0] || point[1] === last[1]) {
						return;
					}
				}
				nodes.push(point.slice(0));
				// one-based indexing
				closer_node = nodes.length;
			}

			function addSegmentTo(point) {
				nodes.push(point.slice(0));
				// one-based indexing
				segments.push([nodes.length - 1, nodes.length]);
			}

			function addClosingSegment() {
				// one-based indexing
				segments.push([nodes.length, closer_node]);
			}

			function objectToLua(name, obj) {
				var result = name + "{",
					needs_delimit = false,
					key;
				for (key in obj) {
					if (obj.hasOwnProperty(key)) {
						if (needs_delimit) {
							result += ", ";
						}
						result += key + " = " + obj[key];
						needs_delimit = true;
					}
				}

				result += "}";
				return result;
			}

			function arrayToLua(name, arr) {
				var result = name + "{" + arr.join(", ") + "}";
				return result;
			}

			function movePen(p) {
				pen = p.slice(0);
			}

			return {
				initWithPath : function (path) {
					var player = new CommandPlayer(path);
					player.play(this);
					return this;
				},

				combineWith : function (another) {
					var ns     = another.getNodes(),
						ss     = another.getSegments(),
						offset = nodes.length;
					ns.forEach(function (n) { nodes.push(n); });
					ss.forEach(function (s) {
						segments.push(s.map(function (i) {
							return i + offset;
						}));
					});
				},

				getNodes : function () {
					return nodes;
				},

				getSegments : function () {
					return segments;
				},

				toString : function () {
					var i, j, n,
						result = objectToLua("pslg", {nodes : nodes.length, segs : segments.length}) + "\n";
					for (i = 0, n = nodes.length; i < n; i += 1) {
						result += arrayToLua("v", nodes[i]) + "\n";
					}

					for (j = 0, n = segments.length; j < n; j += 1) {
						result += arrayToLua("s", segments[j]) + "\n";
					}

					return result;
				},

				applyTransform : function (xform) {
					var i, n;
					for (i = 0, n = nodes.length; i < n; i += 1) {
						nodes[i] = xform.apply(nodes[i]);
					}
				},

				moveTo : function (inParams) {
					movePen(inParams.pts[0]);
					return this;
				},

				lineTo : function (inParams) {
					addNode(pen);
					addSegmentTo(inParams.pts[0]);
					movePen(inParams.pts[0]);
					return this;
				},

				bezierTo : function (inParams) {
					var i;
					addNode(pen);
					for (i = 0; i < 3; i += 1) {
						addSegmentTo(inParams.pts[i]);
					}
					movePen(inParams.pts[2]);
					return this;
				},

				close : function () {
					addClosingSegment();
					movePen(nodes[closer_node - 1]);
					closer_node = false;
					return this;
				}
			};
		}

		function allocWalker() {
			function State() {
				this.nodeStack     = [];
				this.xformStack    = [];
				this.current       = null;
				this.currentToRoot = null;
				this.containers    = [];
				this.primitives    = [];
			}

			function splitContents(that) {
				if (that.current.constructor === Container) {
					var children = that.current.getChildren();
					children.forEach(function (n) {
						if (n.constructor === Container) {
							that.containers.push(n);
						} else {
							that.primitives.push(n);
						}
					});
				}
			}

			var methods = {
				init : function (root) {
					if (root.constructor === Container) {
						this.current = root;
// Decide if it's more comment to transform to this root (1) or its parent (2).
						this.currentToRoot = new Matrix2d();
						this.currentToRoot.setIdentity(); // (1)
//                        this.currentToRoot = root.getMatrix(); // (2)
						splitContents(this);
					}
					return this;
				},

				done : function () {
					return (this.current === null);
				},

				next : function () {
					if (this.containers.length > 0) {
						var nodeStack    = this.nodeStack,
							xformStack   = this.xformStack,
							parentToRoot = this.currentToRoot;

						this.containers.forEach(function (n) {
							nodeStack.push(n);
							var currentToRoot = parentToRoot.clone().appendTransform(n.getMatrix());
							xformStack.push(currentToRoot);
						});
					}

					this.containers = [];
					this.primitives = [];
					if (this.nodeStack.length > 0) {
						this.current = this.nodeStack.pop();
						this.currentToRoot = this.xformStack.pop();

						splitContents(this);
					} else {
						this.current = null;
					}

					return this;
				},

				getCurrent : function () {
					return this.current;
				},

				getCurrentToRoot : function () {
					return this.currentToRoot;
				},

				getContainers : function () {
					return this.containers;
				},

				getPrimitives : function () {
					return this.primitives;
				}
			};

			State.prototype = methods;
			return new State();
		}

		return {
			allocPSLG : allocPSLG,
			allocWalker : allocWalker
		};
	});
