/*
	SdkLayer:

	This object is the abstract facade that 3rd parties communicate with to control a layer, as returned
	from a behavior layer param.
	
*/

define(["src/utils"],
function(utils) {
	"use strict";


	function SdkLayer(inLayer) {
		this.privateLayer = inLayer;	// privateLayer might be a Layer or a TrackItem
		// to make this truly private from 3rd parties, we'd have to use a closure here for every entry point,
		//	which could be slow and cumbersome. And having the option to cheat could be helpful for 3rd parties,
		//	as long as they know they're cheating and need to request new APIs.
		
		this.sharedBehaviorFrameData = {};	// cleared out after each frame (@@@ clear out in engine)
		// TODO: this.sharedBehaviorSceneData = {};	// persists across frames
	}
	
	utils.mixin(SdkLayer, {
		getSource :		function ()			{ return this.privateLayer.getSource(); },
		getName :		function ()			{ return this.privateLayer.getName(); },
		getId :			function ()			{ return this.privateLayer.getStageId(); }, // only stable within a single stage execution
		getPuppet :		function ()			{ return this.privateLayer.getPuppet(); },	// same as getSource if source is a puppet; TODO: wrap to return SdkPuppet
		getParentPuppet : function ()		{ return this.privateLayer.parentPuppet; },	// returns null when this layer is TrackItem
		getParentLayer : function ()		{ var p = this.getParentPuppet();		// returns null when this layer is TrackItem
											 	if (p) {
													return p.getParentLayer().getSdkLayer();
												} else {
													return null;
												}
											},

		// for inter-behavior communication; cleared at end of each frame automatically
		//	TODO: move implemention & storage into privateLayer once TrackItems are sourced by Layers (so that stageLayer will have this data)
		setSharedFrameData : function (key, value) { this.sharedBehaviorFrameData[key] = value; },
		getSharedFrameData : function (key)	{ return this.sharedBehaviorFrameData[key]; },
		
		// this one should _only_ be exposed on the privateLayer
		clearSharedFrameData : function () { this.sharedBehaviorFrameData = {}; },
		
		setOpacity :	function (opacity)	{ return this.privateLayer.setOpacity(opacity); },
		getOpacity :	function ()			{ return this.privateLayer.getOpacity(); },
		setVisible :	function (bEnabled)	{ return this.privateLayer.setVisible(bEnabled); },
		getVisible :	function ()			{ return this.privateLayer.getVisible(); },

		// may want to implement getBounds on Layerish and call it from here instead of calling getSource here?
		getBounds : 	function ()			{ return this.privateLayer.getSource().getBounds(); },

		// calls the given function once for each layer, with a single sdkLayer as the argument
		// starts with the "this" layer and includes all children of sources; return false from fn to stop iterating
		//	based on more general treeContainer.js version, but we can't use the entire treeContainer mixin because (1) we don't want a writable
		//	tree, and (2) we don't want a parallel tree one level off of layers (vs. items)
		forEachLayerBreadthFirst : function (fn) {
			var layer,
				queue = [ this ];
			
			function pushChild (lay2) {
				queue.push(lay2);
			}

			while (queue.length > 0) {
				layer = queue.shift();
				if (fn(layer) === false) {
					break;
				}
				
				if (!layer.forEachDirectChildLayer(pushChild)) {
					break;
				}
					
			}

			return this;
		},
		
		// if the function returns false, the iteration stops and the return value is also false; iterates in front-to-back order
		forEachDirectChildLayer : function (fn) {
			var sourcePuppet, children, i;
			sourcePuppet = this.getPuppet(); // returns false if layer source isn't a puppet

			if (sourcePuppet) {
				children = sourcePuppet.getLayers();
				for (i = children.length-1; i >= 0 ; --i) {
					var layer = children[i];
					if (layer.getUserVisible()) {
						if (fn(layer.getSdkLayer()) === false) {
							return false;
						}
					}
				}
			}
			
			return true;
		}

	});
	
	return SdkLayer;
});