/*
	Puppet:

	This object represents the puppet on the stage.
*/

define(["src/utils", "src/build/treeLeaf", "src/build/treeContainer", "src/build/DisplayContainerStorage", "src/build/Container",
		"src/math/Vec2", "src/math/Mat3", "lodash", "src/build/StageId", "src/build/HandleTreeArray"],
function (utils, treeLeaf, treeContainer, DisplayContainerStorage, Container,
		vec2, mat3, lodash, StageId, HandleTreeArray) {
	"use strict";

	var kOriginLeaf = "__Origin__Leaf__";

	function layerWarpsIndependently (layer) {
		return !layer.getWarpWithParent();
	}

	function layerWarpsWithParent (layer) {
		return layer.getWarpWithParent();
	}

	/**
	 * Find puppet handle that the sublayer attaches to.
	 * Use bind() to derive partial function for memoization.
	 * @param this Puppet with handle to attach to.
	 * @param layer Sublayer attached to puppet.
	 */
	function findAttachToHandle (layer) {
		/*jshint validthis: true*/
		var id = layer.getTagBackstageId(),
			root = this.getHandleTreeRoot(),
			tag = null;

		// TODO: right now identifiers are only unique within a puppet so 
		// layers must attach to handles within the super-puppet.
		root.preOrderEach(function (hi) {
			if (hi.getTagBackstageId() === id) {
				tag = hi;
				return false;
			}
		});

		if (tag === null) {
			console.logToUser("Layer '" + layer.getName() + "' did not attach to '" + id + "' in puppet '" + this.getName());
			tag = root;
		}

		return tag;
	}

	function forEachSelectedPuppetLayer (puppet, isSelected, fn) {
		// the fastest (according to lodash) iteration pattern
		var index = -1,
			collection = puppet.getLayers(),
			length = collection ? collection.length : 0,
			subLayer;

		while (++index < length) {
			subLayer = collection[index];

			if ( subLayer.getPuppet() && isSelected(subLayer) ) {
				if (fn(subLayer, index, collection) === false)	break;
			}
		}

		return collection;
	}
/*
	function forEachWarperLayer (puppet, fn) {
		// the fastest (according to lodash) iteration pattern
		var index = -1,
			collection = puppet.getLayers(),
			length = collection ? collection.length : 0,
			subLayer;

		while (++index < length) {
			subLayer = collection[index];

			if ( subLayer.getWarpWithParent() && subLayer.getPuppet() ) {
				if (fn(subLayer, index, collection) === false)	break;
			}
		}

		return collection;
	}
*/
	function getPuppetData (obj) { 
		return obj.pPuppet; 
	}

	function getTreeData (obj) { 
		return obj.pTree;
	}

	function Puppet(puppetArgs0) {
		var self = this; // reference for closures with rebound this

		this.pPuppet = lodash.defaults({}, puppetArgs0, {
			name     			 : null,
			layers   			 : null,
			handle   			 : null,
			behaviors			 : null,
			warpHash			 : null,
			warpMeshExpansion 	 : 0,
			warpType  			 : Container.WarpDomain.BOX,
		});

		this.pTree = {
			// TODO: try renaming these and adapting via mixin label parameters
			parent     : null,
			subpuppets : []
		};

		this.getAttachToHandle = lodash.memoize(findAttachToHandle, StageId.hash);

		// states for mixins
		self.DisplayContainerStorage(false, true, this.getName());

		self.getLayers().forEach(function (l) {
			var puppetChild = l.getPuppet();
			if (puppetChild) {self.addChild(puppetChild); }
			l.parentPuppet = self; // so we can walk up the tree (TODO: update this up-pointer when cloning)
		});

		if (self.getHandleTreeRoot()) {
			self.setHandleTree(self.getHandleTreeRoot());
		}
	}

	utils.mixin(Puppet, DisplayContainerStorage,
		treeLeaf({ getTreeInfo: getTreeData, parentLabel : "parent" }),
		treeContainer({ getTreeInfo: getTreeData, childrenLabel : "subpuppets" }),
		{
			getName: function () {
				return this.pPuppet.name;
			},

			getLayers: function () {
				return this.pPuppet.layers;
			},

			getLayerByID : function (bindingID) {
				var layer;
				
				// could build a map to make this faster
				this.getLayers().forEach(function (l) {
					if (l.getBindingId() === bindingID) {
						layer = l;
					}
				});
				
				return layer;
			},

			// TODO: rename to setLocalHandleTree
			setHandleTree : function (handle) {
				var cPuppet;

				// invalidate cache
				this.getAttachToHandle.cache = {};

				this.setDisplayContainer(null);

				// setup new handles
				this.pPuppet.handle = handle;

				var puppet = this;
				handle.breadthFirstEach(function (h) {
					h.setPuppet(puppet);
				});

				// configure display container
				cPuppet = new Container(this.getName());
				this.getLayers().forEach(function (l) {
					cPuppet.addChild(l.getDisplayContainer());
				});
				this.setDisplayContainer(cPuppet);
			},

			getHandleTreeRoot: function () {
				return this.pPuppet.handle;
			},
			
			getBehaviors: function () {
				return this.pPuppet.behaviors;
			},

			privateWarp: function () {
				var puppet = this;
				this.getLayers().forEach(function (subLayer) {
					if (subLayer.getWarpWithParent()) {
						subLayer.warp();
					} else {
						var warperLayer = puppet.getWarperLayer(),
							warperSetsContainerMatrix = warperLayer.warperSetsContainerMatrix();
						if (warperSetsContainerMatrix) {
							subLayer.warp();
						} else {
							// use attachTo tag to update subLayer transform
							var tree = warperLayer.getHandleTreeArray(),
								handle = puppet.getAttachToHandle(subLayer),
								origin = puppet.getHandleTreeRoot();

							if (handle === origin) {
								// if independent layer attaches to root, move it over to __Origin__Leaf__
								if (layerWarpsIndependently(subLayer)) {
									var inboardJoint = lodash.find(origin.getChildren(), function (h) {
											return h.getName() === kOriginLeaf;
										});
									utils.assert(inboardJoint);
									handle = inboardJoint;
								}
							}

							var	handleRef = tree.getHandleRef(handle),
								matWarper_Handle = tree.getAccumulatedHandle("puppet", handleRef);
							subLayer.warp(matWarper_Handle);
						}
					}
				});
			},

			getView : function () {
				var trackItem = this.getTrackItem();
				if (trackItem) {
					return trackItem.getView();
				}
				return null;
			},
		
		
			// i.e. the layer that refers to this puppet -- kosher in JS land because the DAG has
			//	been turned into a tree here (but not so in Lua-ville)
			getParentLayer : function () {
				var parentLayer = this.parentLayer; // Layers jam this field in their source

				if (parentLayer === undefined) { 	// if undefined, it must be a TrackItem
					parentLayer = this.getParent();
				}

				return parentLayer;
			},

			// returns this if no parent puppets.
			getRootPuppet: function () {
				var	parentPuppetP0 = this, foundParent = null;
				while (parentPuppetP0) {
					if (parentPuppetP0.constructor === Puppet) {
						foundParent = parentPuppetP0;
					}
					parentPuppetP0 = parentPuppetP0.getParent();	// note: this walks up the puppet tree, skipping layers
				}
				return foundParent;
			},

			// TODO: move into Layer
			getTrackItem: function () {
				var parentPuppet = this.getRootPuppet();
				return parentPuppet.getParent();
			},

			// TODO: move into Layer
			getTrack: function () {
				var ti = this.getTrackItem();
				if (ti) {
					return ti.getParent();
				}
				return null;
			},

			// TODO: move into Layer
			getScene: function () {
				var t = this.getTrack();
				if (t) {
					return t.getParent();
				}
				return null;
			},

			displayInView: function () {
				// propagate display event to *puppet* layers
				this.getLayers().forEach(function (layer) {
					if (layer.getPuppet()) {
						layer.displayInView();
					}
				});
			},

			getBounds: function () {
				var bbox = [0, 0, 0, 0];

				this.breadthFirstEach(function (p) {
					var c = p.getDisplayContainer(), cBBox = [0, 0, 0, 0];
					if (c) {
						cBBox = c.getBounds();
						if (cBBox[0] < bbox[0] || bbox[2] === 0) { bbox[0] = cBBox[0]; }
						if (cBBox[1] < bbox[1] || bbox[3] === 0) { bbox[1] = cBBox[1]; }
						if ((cBBox[0] + cBBox[2]) > (bbox[0] + bbox[2])) { bbox[2] = cBBox[0] + cBBox[2] - bbox[0]; }
						if ((cBBox[1] + cBBox[3]) > (bbox[1] + bbox[3])) { bbox[3] = cBBox[1] + cBBox[3] - bbox[1]; }
					}
				});

				return bbox;
			},


			/**
			* Clone puppet.
			* @params {boolean} [clone_children=true] - determines whether cloning includes children.
			* @params [result={}] - destination object to receive cloned result
			*/
			clone: function (clone_children, result) {
				var data = getPuppetData(this),
					options = {
						name : data.name,
						layers : null,
						handle : null,
						behaviors : null
					};

				clone_children = typeof clone_children === "boolean" ? clone_children : true;
				utils.assert(clone_children === true, "NYI: shallow Puppet clone.");

				// clone layers
				if (data.layers) {
					options.layers = lodash.map(data.layers, function (layer) {
						return layer.clone();
					});
				}

				// clone behaviors				
				if (data.behaviors) {
					utils.clone(true, options.behaviors, data.behaviors);
				}

				if (result) {
					// init
					Puppet.call(result, options);
				} else {
					// alloc and init
					result = new Puppet(options);
				}

				// clone handles
				if (data.handle) {
					result.setHandleTree(this.getHandleTreeRoot().clone(true));
				}

				return result;
			},

			getWarperLayer : function () {
				return this.getParentLayer().getWarperLayer();
			},

			getWarpType : function () {
				return this.pPuppet.warpType.slice(0);
			},

			getWarpMeshExpansion : function () {
				return this.pPuppet.warpMeshExpansion;
			},

			privateGatherHandleTreeArray : function (tree0, tagHandle0) {
				var self = this,
					layer = this.getParentLayer(),
					tree = tree0 || new HandleTreeArray();

				tree.addLayer(layer, tagHandle0);

				forEachSelectedPuppetLayer(this, layerWarpsWithParent, function (subLayer) {
					var handle = self.getAttachToHandle(subLayer),
						handleRef = tree.getHandleRef(handle);

					subLayer.getPuppet().privateGatherHandleTreeArray(tree, handleRef);
				});

				return tree;
			},

			privateGetWarpHash : function () {
				return this.pPuppet.warpHash;
			},

			privateIsIndependentLayerAttachedToOrigin : function () {
				var puppet = this,
					attachedToOrigin = false;

				forEachSelectedPuppetLayer(this, layerWarpsIndependently, function (subLayer) {
					var handle = puppet.getAttachToHandle(subLayer);
					if (handle === puppet.getHandleTreeRoot()) {
						attachedToOrigin = true;
						return false;
					}
				});

				return attachedToOrigin;
			}
		}
	);

	return Puppet;
});
