/*global define, require */
/*jslint white: true */

/*
	Path:

	This object represents a Path used for vector graphics.
*/

define([    "src/build/CommandPlayer",  "src/math/Mat3",	"src/math/Vec2",	"src/math/mathUtils", 	"src/math/curveUtils",  "src/utils"],
function(   CommandPlayer,              mat3,				vec2,				mathUtils,				curveUtils,             utils) {
	'use strict';

	function transformPoints (mtx, inPts) {
		if (inPts && inPts.length !== 0) {
			var i;

			for (i = 0; i < inPts.length; i += 1) {
				vec2.transformAffine(mtx, inPts[i], inPts[i]);
			}
		}
		return inPts;
	}

	return function Path() {
		var that = this, PrintPlayer, isPenDown;

		// In order for this object to support cloning, the private members must be
		// exposed and accessible outside of the closure.  -jacquave
		that.pPathData = { name: undefined, commands : [] };

		that.setName = function (inName) {
			this.pPathData.name = inName;
		};

		that.getName = function (inName) {
			return this.pPathData.name;
		};

		that.moveTo = function (inPt1) {
			this.pPathData.commands.push({ cmdName : "moveTo", parameters : { pts : [inPt1] } });
		};

		that.lineTo = function (inPt1) {
			this.pPathData.commands.push({ cmdName : "lineTo", parameters : { pts : [inPt1] } });
		};

		that.bezierTo = function (inPt1, inPt2, inPt3) {
			this.pPathData.commands.push({ cmdName : "bezierTo", parameters : { pts : [inPt1, inPt2, inPt3] } });
		};

		that.addBeziers = function (inBeziers) {
			var i, firstPt, lastPt;
			for (i = 0; i < inBeziers.length; i += 1) {
				firstPt = inBeziers[i][0];
				if (firstPt !== lastPt) {
					this.moveTo(firstPt);
				}
				lastPt = inBeziers[i][3];
				this.bezierTo(inBeziers[i][1], inBeziers[i][2], lastPt);
			}
		};

		that.addRect = function (inLTWH) {
			this.moveTo([inLTWH[0], inLTWH[1]]);
			this.lineTo([inLTWH[0] + inLTWH[2], inLTWH[1]]);
			this.lineTo([inLTWH[0] + inLTWH[2], inLTWH[1] + inLTWH[3]]);
			this.lineTo([inLTWH[0], inLTWH[1] + inLTWH[3]]);
			this.close();
		};

		// Converts addArc into a sequence of moveTo/lineTo/bezierTo commands
		that.addArc = function (inPt1, inRadius, inStartAngle, inEndAngle, inCounterClockwise) {
			var mtx = mat3.identity(), sweep, startPt = [0, 1],
				curvePts = [], bezDegreesF = 0, firstB = true, positiveB = false,
				sweepRemF = 0, sweepRem2F = 0, partialB = false,
				quadCurvePts = [ [0, 0], [0, 0], [0, 0], [0, 0] ],
				quadCurve2Pts = [ [0, 0], [0, 0], [0, 0], [0, 0] ], tF = 0,
				quadTransform, i, j, evalPt;

			sweep = inEndAngle - inStartAngle;

			if (inCounterClockwise) {
				sweep = -sweep;
				inStartAngle = inEndAngle;
			}

			if (sweep < 0) {
				sweep = 360 + sweep;
			}

			mtx = mtx.translate(inPt1).rotate(mathUtils.rad(inStartAngle)).scale([inRadius, inRadius]);

			if (sweep === 0) {
				vec2.transformAffine(mtx, startPt, startPt);
				if (isPenDown(this)) {
					this.lineTo(startPt);
				} else {
					this.moveTo(startPt);
				}

				return;
			}

			positiveB = sweep > 0;

			do {
				sweepRemF = sweep - bezDegreesF;
				partialB = Math.abs(sweepRemF) < 90;

				quadCurvePts[0][0] = 1.0;
				quadCurvePts[0][1] = 0.0;
				quadCurvePts[1][0] = 1.0;
				quadCurvePts[1][1] = curveUtils.ellipseHandleScale;
				quadCurvePts[2][0] = curveUtils.ellipseHandleScale;
				quadCurvePts[2][1] = 1.0;
				quadCurvePts[3][0] = 0.0;
				quadCurvePts[3][1] = 1.0;

				if (partialB) {
					// approximate t.
					tF = Math.abs(sweepRemF) / 90.0;

					// refine t since we can't analytically know how to map angle to t.  -jacquave
					for (j = 0; j < 3; j += 1) {
						evalPt = curveUtils.evaluate2DCurve(quadCurvePts, tF);
						sweepRem2F = Math.atan2(evalPt[1], evalPt[0]) * 180 / Math.PI;
						tF = tF - (sweepRem2F - sweepRemF) / 90.0;
					}

					// This curve subdivision is based on t, which is not the same as angle and distance.  
					// But, it should provide a decent approximation. -jacquave 8/7/2007
					curveUtils.subdivide2DCurve(quadCurvePts, tF, quadCurvePts, quadCurve2Pts);
				}

				quadTransform = mtx.clone();

				quadTransform.prerotate(mathUtils.rad(Math.abs(bezDegreesF)));
				if (!positiveB) {
					quadTransform.prescale([1.0, -1.0]);
				}
				transformPoints(quadTransform, quadCurvePts);

				if (firstB) {
					curvePts.push(quadCurvePts[0].slice(0));
					firstB = false;
				}
				curvePts.push(quadCurvePts[1].slice(0));
				curvePts.push(quadCurvePts[2].slice(0));
				curvePts.push(quadCurvePts[3].slice(0));

				bezDegreesF += positiveB ? 90 : -90;
			} while (positiveB ? bezDegreesF < sweep : bezDegreesF > sweep);

			if (curvePts.length !== 0) {

				if (isPenDown(this)) {
					this.lineTo(curvePts[0]);
				} else {
					this.moveTo(curvePts[0]);
				}
				for (i = 1; i < curvePts.length; i += 3) {
					this.bezierTo(curvePts[i], curvePts[i + 1], curvePts[i + 2]);
				}
			}
		};

		that.addEllipse = function (inLTWH) {
			// Handle length for bezier representation of ellipses, as % of radius
			// 4/3 * (sqrt(2) - 1) (Copied from OMp_BEZ_ELLIPSE_HANDLE_SCALE) -jacquave
			var halfW = 0.5 * inLTWH[2], halfH = 0.5 * inLTWH[3], tanWidth, tanHeight, centerX, centerY,
				left = inLTWH[0], top = inLTWH[1], w = inLTWH[2], h = inLTWH[3];

			tanWidth = halfW * curveUtils.ellipseHandleScale;
			tanHeight = halfH * curveUtils.ellipseHandleScale;
			centerX = inLTWH[0] + halfW;
			centerY = inLTWH[1] + halfH;

			this.moveTo([centerX, top]);

			this.bezierTo([centerX + tanWidth, top],
						  [left + w, centerY - tanHeight],
						  [left + w, centerY]);

			this.bezierTo([left + w, centerY + tanHeight],
						  [centerX + tanWidth, top + h],
						  [centerX, top + h]);

			this.bezierTo([centerX - tanWidth, top + h],
						  [left, centerY + tanHeight],
						  [left, centerY]);

			this.bezierTo([left, centerY - tanHeight],
						  [centerX - tanWidth, top],
						  [centerX, top]);

			this.close();
		};

		that.close = function () {
			this.pPathData.commands.push({ cmdName : "close", parameters : {} });
		};

		that.clone = function () {
			var newPath = utils.clone(true, {}, this);
			return newPath;
		};

		that.append = function (inPath) {
			var otherCommands = utils.clone(true, [], inPath.getCommands());
			this.pPathData.commands = this.pPathData.commands.concat(otherCommands);
		};

		that.getCommands = function () {
			return this.pPathData.commands;
		};

		PrintPlayer = function (inPath) {
			var that = this, m_path = inPath, m_prints = "", printPoint, printPoints, printCmd;

			printPoint = function (inPt) {
				var p;

				m_prints += "[";
				m_prints += inPt[0].toString();
				m_prints += ",";
				m_prints += inPt[1].toString();
				m_prints += "]";
			};

			printPoints = function (inPts) {
				var p;

				if (inPts.length > 0) {
					if (inPts.length === 1) {
						printPoint(inPts[0]);
						return;
					}
					m_prints += "[";
					for (p = 0; p < inPts.length; p += 1) {
						if (p !== 0) {
							m_prints += ",";
						}
						printPoint(inPts[p]);
					}
					m_prints += "]";
				}
			};

			printCmd = function (inFuncName, inPts) {
				if (m_prints !== "") {
					m_prints += ", ";
				}
				m_prints += inFuncName;

				if (inPts !== undefined) {
					m_prints += " : ";
					printPoints(inPts);
				}
			};

			that.moveTo = function (inParams) {
				printCmd("moveTo", inParams.pts);
			};

			that.lineTo = function (inParams) {
				printCmd("lineTo", inParams.pts);
			};

			that.bezierTo = function (inParams) {
				printCmd("bezierTo", inParams.pts);
			};

			that.close = function () {
				printCmd("close");
			};

			that.print = function () {
				var cmdPlayer = new CommandPlayer(m_path);
				m_prints = "";
				cmdPlayer.play(this);
				return m_prints;
			};

			return that;
		};

		that.print = function () {
			var player = new PrintPlayer(this);
			return player.print();
		};

		isPenDown = function (inPath) {
			var cmds = inPath.getCommands();

			if (cmds.length === 0) {
				return false;
			}

			return cmds.cmdName !== "close";
		};

		return that;
	};
});
