define(["src/utils", "src/build/DisplayContainerStorage", "src/math/Mat3", "src/build/Container", "src/build/Puppet", "src/build/SdkLayer",
		"src/math/Vec2", "lodash", "src/build/StageId", "src/build/Handle"],
function (utils, DisplayContainerStorage, mat3, Container, Puppet, SdkLayer,
		vec2, lodash, StageId, Handle) {
	"use strict";

	var kInitializeWithArtworkState = false; // set to true to initialize warper with the true artwork state.  otherwise, it will use current handle state whatever it may be.
	var kDropLeafForExplicitOrigin = false; // set to true if you want to go back to the heuristic of dropping collocated origin leaves for each explicitly specified origin
	var kOriginLeaf = "__Origin__Leaf__"; 	// keep in synch with the string in Puppet.lua 

	function makeJointHandle (layer) {
		var joint = new Handle({name : kOriginLeaf, puppet : layer.getPuppet()});
		layer.getHandleTreeRoot().addChild(joint);
		return joint;
	}

	function setupJoint (layer) {
		// look for a leaf origin handle used for attaching one puppet to another
		var origin = layer.getHandleTreeRoot(),
			joint = lodash.find(origin.getChildren(), function (h) {
				return h.getName() === kOriginLeaf;
			});

		if (kDropLeafForExplicitOrigin) {
			// if found
			if (joint) {
				if (layer.getWarpWithParent()) {
					joint.auto({
						translation : true,
						linear : true
					});
				} else {
					// enforce desired motion dofs for layers that warp independently
					joint.auto({
						translation : layer.motionMode.position,
						linear : layer.motionMode.scaling
					});
				}
			}
		} else {
			if (layer.getWarpWithParent()) {
				var layerAttachedToOrigin = layer.getPuppet().privateIsIndependentLayerAttachedToOrigin();
				if (layerAttachedToOrigin) {
					// either reuse or create new one
					if (! joint) joint = makeJointHandle(layer);
					joint.auto({translation : true, linear : true});
				} else {
					// don't drop a leaf, and remove if there
					if (joint) origin.removeChild(joint);
				}
			} else {
				// either reuse or create new one
				if (! joint) joint = makeJointHandle(layer);
				joint.auto({ translation : layer.motionMode.position, linear : layer.motionMode.scaling});
			}
		}
	}


	function Layer(source, args0) {
		this.StageId("Layer_");
		this.source = source;
		lodash.defaults(this, args0, {
			name : null,
			bindingId : null,
			tag : [0, 0],
			tagBackstageId : null,
			matLayer_Source : mat3.identity(),
			motionMode : { position : true,	scaling : true },
			warpWithParent : true,
			bUserVisible : true, // actually more like SDK-visible on the JS side (we use this to hide Locations layers)
			warpHash : null
		});
		this.handleTreeArray = null;
		this.matSource_Layer = mat3.invert(this.matLayer_Source);

		// parentPuppet is set by the parent puppet when the layer is installed in the puppet
		this.parentPuppet = null;
		this.sdkLayer = new SdkLayer(this);

		// create display item containers for this layer:
		// tag -> matrix -> source
		this.DisplayContainerStorage(false, false, this.name);

		var cLayerAttacher = new Container("__LayerAttacher__" + this.name);
		cLayerAttacher.setMatrix(mat3.translation(this.tag));

		var cSourceAttacher = new Container("__SourceAttacher__" + this.name);
		cLayerAttacher.addChild(cSourceAttacher, 0);
		cSourceAttacher.setMatrix(this.matLayer_Source);

		this.source.parentLayer = this; // HACK, so that layer params can resolve to the root layer (one up from the root puppet)
									// and so we can go from stagePuppet to stageLayer (similar to parentPuppet for going up from layer)

		var puppet = this.getPuppet();									
		if (puppet) {
			var cPuppet = puppet.getDisplayContainer();
			cSourceAttacher.addChild(cPuppet);
			setupJoint(this);
		} else {
			// or use the source itself as the display item
			cSourceAttacher.addChild(this.getSource());
		}

		this.setDisplayContainer(cLayerAttacher);
	}

	utils.mixin(Layer, DisplayContainerStorage, StageId, {

		// alternatively, we could get rid of .sdkLayer and have this function
		//	just do "return new SdkLayer(this);"
		getSdkLayer : function () { return this.sdkLayer; },
		
		getName : function () { return this.name; },	// note: matching container also has this name
		
		getSource : function () { return this.source; },

		getSourceMatrixRelativeToLayer : function () {
			return mat3.clone(this.matLayer_Source);
		},

		getLayerMatrixRelativeToSource : function () {
			return mat3.clone(this.matSource_Layer);
		},

		getPuppet : function () {
			var p = this.getSource();
			return p.constructor === Puppet ? p : null;
		},

		getTag : function () {
			return vec2.clone(this.tag);			
		},

		getTagBackstageId : function () {
			var id = this.tagBackstageId;
			return id ? id.slice(0) : null;
		},
		
		getBindingId: function () {
			return this.bindingId;
		},

		/**
		 * Warp layer according to the state of its handles.
		 * @params matTag_Layer Transform matrix for this layer relative to its attachment tag.
		 */		
		warp : function (matSupPuppet_Tag0) {
			var cLayerAttacher = this.getDisplayContainer();
			if (matSupPuppet_Tag0) cLayerAttacher.setMatrix(matSupPuppet_Tag0);

			var	puppet = this.getPuppet();
			if ( !puppet ) return;

			var cWarper = this.getWarperContainer();
			if (cWarper) {
				// warp and scatter to subpuppets that warp together
				var	tree = this.getHandleTreeArray(),
					aLeafRef = tree.getLeafRefArray(),
					aMatPuppet_Handle = tree.getAccumulatedTree("puppet"),
					aMatPuppet_Leaf = lodash.at(aMatPuppet_Handle, aLeafRef),
					aLeafAutomate = lodash.at(tree.getTreeAutomation(), aLeafRef);

				cWarper.warp(aMatPuppet_Leaf, aLeafAutomate);

				aLeafRef.forEach(function (leafRef, index) {
					aMatPuppet_Handle[leafRef] = aMatPuppet_Leaf[index];
				});
				tree.setAccumulatedTree("puppet", aMatPuppet_Handle);
			}

			// propagate warp to all subpuppets
			puppet.privateWarp();
		},

		warperSetsContainerMatrix : function () {
			var cWarper = this.getWarperContainer();
			if ( !cWarper ) return false;

			return cWarper.warperSetsContainerMatrix();
		},

		setVisible : function (bEnabled) {
			var c = this.getDisplayContainer();
			c.setVisibleEnabled(bEnabled);
		},

		getVisible : function () { 
			var c = this.getDisplayContainer();
			return c.getVisibleEnabled();
		},

		setWarpWithParent : function (bEnabled, aMatPuppet_HandleAtRest0) {
			var cSourceAttacher, cWarper, cPuppet,
				didWarpWithParent = this.getWarpWithParent(),
				puppet = this.getPuppet();

			this.warpWithParent = bEnabled;

			if ( !puppet ) utils.assert(this.warpWithParent, "NYI: skin layers that warp on their own.");

			// no change
			if ( !puppet || (didWarpWithParent === this.warpWithParent) ) return;

			// layer no longer warps on its own
			if ( !didWarpWithParent && this.warpWithParent) {
				setupJoint(this);
				// remove warper
				cSourceAttacher = this.getSourceAttachContainer();
				cPuppet = puppet.getDisplayContainer();
				cSourceAttacher.removeChildAtIndex(0);
				cSourceAttacher.addChild(cPuppet, 0);
				return;
			}

			// layer starts to warp on its own
			if ( didWarpWithParent && !this.warpWithParent ) {
				setupJoint(this);
				// make warper
				cSourceAttacher = this.getSourceAttachContainer();
				cWarper = new Container("__LayerWarper__" + this.name);
				cPuppet = puppet.getDisplayContainer();
				cSourceAttacher.removeChildAtIndex(0);
				cSourceAttacher.addChild(cWarper);

				cWarper.setWarpDomain(puppet.getWarpType());
				cWarper.setWarpHash(this.warpHash);
				cWarper.setWarpMeshExpansion(puppet.getWarpMeshExpansion());
				cWarper.setParentCanWarpMe(this.warpWithParent);
				cWarper.addChild(cPuppet);

				var tree = this.getHandleTreeArray();
				if (aMatPuppet_HandleAtRest0) {
					this.aMatPuppet_HandleAtRest = aMatPuppet_HandleAtRest0;
				} else {
					if (kInitializeWithArtworkState) 
						this.aMatPuppet_HandleAtRest = tree.getAccumulatedTreeAtRest("puppet");
					else
						this.aMatPuppet_HandleAtRest = tree.getAccumulatedTree("puppet");
				}
				this.aMatPuppet_LeafAtRest = lodash.at(this.aMatPuppet_HandleAtRest, tree.getLeafRefArray());
				cWarper.setWarpRest(this.aMatPuppet_LeafAtRest);
			}

		},

		getWarpWithParent : function () { 
			return this.warpWithParent;
		},
		
		getUserVisible : function () {
			return this.bUserVisible;
		},

		clone: function (result) {
			var source = this.source.clone(),
				args = {
					name : this.name ? this.name.slice(0) : null,
					tag : this.getTag(),
					tagBackstageId : this.getTagBackstageId(),
					matLayer_Source : mat3.clone(this.matLayer_Source),
					motionMode : lodash.clone(this.motionMode),
					bindingId : this.bindingId,
					bUserVisible : this.bUserVisible
				};

			if (result) {
				Layer.call(result, source, args);
			} else {
				result = new Layer(source, args);
			}

			// clone display container properties
			result.setOpacity(this.getOpacity());
			result.setVisible(this.getVisible());
			result.setWarpWithParent(this.getWarpWithParent(), lodash.cloneDeep(this.aMatPuppet_HandleAtRest));

			return result;
		},

		getSourceAttachContainer : function () {
			var cLayerAttacher = this.getDisplayContainer();
			return cLayerAttacher.getChildren()[0];
		},

		getWarperContainer : function () {
			if (this.getWarpWithParent()) return null;

			var cSourceAttacher = this.getSourceAttachContainer(),
				cWarper = cSourceAttacher.getChildren()[0];

			utils.assert(cWarper, "getWarperContainer(): warper container not found.");
			return cWarper;
		},

		displayInView : function () {
			// initialize warp
			var cWarper = this.getWarperContainer();
			if (cWarper) {
				cWarper.initWarp();
				if ( cWarper.shouldInitializeWarp() && !cWarper.canWarp() ) {
					console.logToUser("A layer binding puppet '" + this.getPuppet().getName() + "' has several handles but no artwork to warp.");
				}
			}
			// and propagate display event
			var puppet = this.getPuppet();
			if (puppet) {
				puppet.displayInView();
			}
		},

		getTrackItem : function () {
			// HACK we should not go to source puppet for this.
			// TODO revise when we stamp out reliance on Puppet instead of Layer.
			var puppet = this.getPuppet();
			utils.assert(puppet, "NYI: getView on skin layers.");
			return puppet.getTrackItem();
		},

		getView : function () {
			return this.getTrackItem().getView();
		},

		getTrack : function () {
			var ti = this.getTrackItem();
			if (ti) {
				return ti.getParent();
			}
			return null;
		},

		getScene : function () {
			var t = this.getTrack();
			if (t) {
				return t.getParent();
			}
			return null;
		},

		setOpacity : function (opacity)	{	
			var c = this.getDisplayContainer();
			utils.assert(opacity >= 0 && opacity <= 1.0, "opacity must be between 0 and 1");
		 	return c.setAlpha(opacity);
		},

		getOpacity : function ()	{	
			var c = this.getDisplayContainer();
		 	return c.getAlpha();
		},

		getHandleTreeRoot : function () {
			var puppet = this.getPuppet();
			utils.assert(puppet, "getHandleTreeRoot(): puppet not found.");
			return puppet.getHandleTreeRoot();
		},

		/**
		 * Gather leaf handles that warp together.
		 * Co-recursive with the corresponding Puppet function.
		 * @return Pre-order array of handles.
		 */
		gatherHandleLeafArray : function () {
			var tree = this.getHandleTreeArray(),
				aLeafRef = tree.getLeafRefArray();

			return lodash.at(tree.aHandle, aLeafRef);
		},

		/**
		 * Gather transforms of leaf handles that warp together.
		 * Co-recursive with the corresponding Puppet function.
		 * @return Pre-order array of matrices relative to this Layer's coordinate frame.
		 */
		gatherHandleLeafMatrixArray : function (kRelativeTo) {
			var tree = this.getHandleTreeArray(),
				aMatLayer_Handle = tree.getAccumulatedTree(kRelativeTo),
				aLeafRef = tree.getLeafRefArray();

			return lodash.at(aMatLayer_Handle, aLeafRef);
		},

		/**
		 * Scatter transforms of leaf handles that warp together.
		 * Co-recursive with the corresponding Puppet function.
		 * @param aMatLayer_Leaf Pre-order array of matrices relative to this Layer's coordinate frame.
		 * @param aAutomation0 Pre-order array of automation instructions.
		 */
		scatterHandleLeafMatrixArray : function (kRelativeTo, aMatLayer_Leaf, aLeafAutomation0) {
			var tree = this.getHandleTreeArray(),
				aLeafRef = tree.getLeafRefArray(),
				aMatLayer_Handle = tree.getAccumulatedTree(kRelativeTo);

			aLeafRef.forEach(function (leafRef, index) {
				aMatLayer_Handle[leafRef] = aMatLayer_Leaf[index];
			});

			var aHandleAutomation = null;
			if (aLeafAutomation0) {
				aHandleAutomation = tree.getTreeAutomation();
				aLeafRef.forEach(function (leafRef, index) {
					aHandleAutomation[leafRef] = aLeafAutomation0[index];
				});
			}

			tree.setAccumulatedTree(kRelativeTo, aMatLayer_Handle, aHandleAutomation);
		},

		getHandleMatrix : function (handle, result0) {
			var tree = this.getHandleTreeArray(),
				handleRef = tree.getHandleRef(handle);
			utils.assert(handleRef !== null, "getHandleMatrix(): handle not found.");

			return tree.getAccumulatedHandle("layer", handleRef, result0);
		},

		/**
		 * Update transform for the given handle.
		 * @private Requires manully setting automation attributes.
		 * @param handle
		 * @param aMatLayer_Handle Matrix relative to Layer's coordinate frame.
		 */
		setHandleMatrix : function (handle, matLayer_Handle, automation0) {
			var tree = this.getHandleTreeArray(),
				handleRef = tree.getHandleRef(handle);
			utils.assert(handleRef !== null, "setHandleMatrix(): handle not found.");

			tree.setAccumulatedHandle(handleRef, matLayer_Handle, automation0);
			if (automation0) tree.propagateHandleAutomation(handleRef, automation0);
		},


		// HACK: refactor along with parentLayer and parentPuppet fields.
		getWarperLayer : function () {
			var warperLayer = this;
			while ( warperLayer && warperLayer.getWarpWithParent() ) {
				warperLayer = warperLayer.parentPuppet.getWarperLayer();
			}

			utils.assert(warperLayer, "getWarperLayer(): warper layer not found.");
			return warperLayer;
		},

		/**
		 * @private
		 */		
		getHandleTreeArray : function () {
			if (this.handleTreeArray !== null) return this.handleTreeArray;

			this.handleTreeArray = this.getPuppet().privateGatherHandleTreeArray();
			return this.handleTreeArray;
		}

	});

	return Layer;
});
