define(["lodash", "src/utils", "src/math/Mat3", "src/math/Vec2"],
function(lodash, utils, mat3) {
	"use strict";

	var arrayPush = Array.prototype.push;

	function fill (value, length) {
		var result = [],
			index = -1;

		while (++index < length) result[index] = value;

		return result;
	}

	/**
	 * Test if the argument is a valid handle ref.
	 * Note: Don't rely on simple falsey test because zero is falsey and we use it as a valid handle ref.
	 */
	function isValidRef (ref) {
		return typeof ref === "number" && ref >= 0;
	}

	/**
	 * Test if the matrix is defined.
	 * Note: Don't rely on simple falsey test because null is falsey and we use to identify singular and missing matrices.
	 */
	function isUndefinedMatrix (mat) {
		return typeof mat === "undefined";
	}

	function invertOrNull (mat) {
		var result;
		try {
			result = mat3.invert(mat);
		} catch (e) {
			result = null;
		}
		return result;
	}

	function enableRelativeTo (kRelativeTo, tree) {

		if (kRelativeTo === "puppet") {

			var matLayerParent_Puppet = tree.aMatLayerParent_Puppet[0],
				matPuppet_LayerParent = tree.aMatPuppet_LayerParent[0];

			tree.aMatLayerParent_Puppet[0] = null;
			tree.aMatPuppet_LayerParent[0] = null;

			return function () {
				tree.aMatLayerParent_Puppet[0] = matLayerParent_Puppet;
				tree.aMatPuppet_LayerParent[0] = matPuppet_LayerParent;
			};

		}

		if (kRelativeTo === "layer") {
			return function () {};
		}

		utils.assert(false, "enableRelativeTo(): unknown frame.");
	}

	function getLocalMatrixAtRest (handle, result0) {
		return handle.getMatrixAtRestRelativeToParent(result0);
	}

	function getLocalMatrix (handle, result0) {
		return handle.getMatrixRelativeToParent(result0);
	}

	function getAccumulatedTree (kRootFrame, getMatrixRelativeToParent) {
		/*jshint validthis: true*/
		var disableRelativeTo = enableRelativeTo(kRootFrame, this);

		var handle,
			mat_Handle = mat3(),
			aMat_Handle = [];

		var	handleRef = 0,
			length = this.aHandle.length,
			matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];

		// accumulate root matrix
		handle = this.aHandle[handleRef];
		getMatrixRelativeToParent(handle, mat_Handle);
		if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
		aMat_Handle.push(mat_Handle.clone());

		// walk downward (to leaves) to accumulate matrices
		while (++handleRef < length) {
			var parentRef = this.aParentRef[handleRef];

			matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];
			handle = this.aHandle[handleRef];
			getMatrixRelativeToParent(handle, mat_Handle);
			if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
			mat3.multiply(aMat_Handle[parentRef], mat_Handle, mat_Handle);
			aMat_Handle.push(mat_Handle.clone());
		}

		disableRelativeTo();

		return aMat_Handle;
	}


	function HandleTreeArray () {
		this.aHandle = [];
		this.aParentRef = [];
		this.aMatLayerParent_Puppet = [];
		this.aMatPuppet_LayerParent = [];
		this.aLeafRef = null;
		this.size = 0;
	}

	utils.mixin(HandleTreeArray, {
		getHandleRef : function (handle) {
			var handleRef = this.aHandle.indexOf(handle);
			if ( !isValidRef(handleRef) ) handleRef = null;

			return handleRef;
		},

		addLayer : function (layer, tagRef0) {
			var layerTree = layer.getHandleTreeRoot().gatherHandleTreeArray();

			arrayPush.apply(this.aHandle, layerTree.aHandle);

			var handleRef = 0, length = layerTree.aParentRef.length;

			// add root...
			if (isValidRef(tagRef0)) {
				this.aParentRef.push(tagRef0);
			} else {
				this.aParentRef.push(null);
			}
			this.aMatLayerParent_Puppet.push(layer.getSourceMatrixRelativeToLayer());
			this.aMatPuppet_LayerParent.push(layer.getLayerMatrixRelativeToSource());

			// and remaining descendents
			while (++handleRef < length) {
				this.aParentRef.push(this.size + layerTree.aParentRef[handleRef]);
				this.aMatLayerParent_Puppet.push(null);
				this.aMatPuppet_LayerParent.push(null);
			}

			this.size += length;
			this.aLeafRef = null;
		},

		getLeafRefArray : function () {
			if (this.aLeafRef === null) {

				var aLeafRef = [],
					aIsLeaf = fill(true, this.size);

				lodash.forEach(this.aParentRef, function (parentRef) {
					aIsLeaf[parentRef] = false;
				});
				lodash.forEach(aIsLeaf, function (isLeafRef, leafRef) {
					if (isLeafRef) aLeafRef.push(leafRef);
				});

				this.aLeafRef = aLeafRef;

			}

			return this.aLeafRef;
		},

		getAccumulatedHandle : function (kRootFrame, handleRef, result0) {
			var disableRelativeTo = enableRelativeTo(kRootFrame, this),
				mat_Handle = mat3.identity(result0);

			// walk upward (to root) to accumulate matrix
			while ( isValidRef(handleRef) ) {
				var handle = this.aHandle[handleRef],
					matLayerParent_Puppet = this.aMatLayerParent_Puppet[handleRef];
				mat3.multiply(handle.getMatrixRelativeToParent(), mat_Handle, mat_Handle);
				if (matLayerParent_Puppet) mat3.multiply(matLayerParent_Puppet, mat_Handle, mat_Handle);
				handleRef = this.aParentRef[handleRef];
			}

			disableRelativeTo();
			return mat_Handle;
		},

		getAccumulatedTree : function (kRootFrame) {
			return getAccumulatedTree.call(this, kRootFrame, getLocalMatrix);
		},

		getAccumulatedTreeAtRest : function (kRootFrame) {
			return getAccumulatedTree.call(this, kRootFrame, getLocalMatrixAtRest);
		},

		setAccumulatedHandle : function (handleRef, matLayerRoot_Handle, automation0) {
			var	handle = this.aHandle[handleRef],
				parentRef = this.aParentRef[handleRef],
				matPuppet_LayerParent = this.aMatPuppet_LayerParent[handleRef],
				mat_Handle = [];

			if ( isValidRef(parentRef) ) {

				var matParent_LayerRoot = invertOrNull(this.getAccumulatedHandle("layer", parentRef));
				if (matParent_LayerRoot) {  // if invertible

					mat3.multiply(matParent_LayerRoot, matLayerRoot_Handle, mat_Handle);
					if (matPuppet_LayerParent) mat3.multiply(matPuppet_LayerParent, mat_Handle, mat_Handle);

				}

			} else {

				mat3.multiply(matPuppet_LayerParent, matLayerRoot_Handle, mat_Handle);

			}

			handle.privateSetMatrixRelativeToParent(mat_Handle, automation0);
		},

		setAccumulatedTree : function (kRootFrame, aMatLayerRoot_Handle, aAutomation0) {
			var disableRelativeTo = enableRelativeTo(kRootFrame, this);

			var aMatHandle_LayerRoot = [];

			var length = this.aHandle.length,
				handleRef = 0,
				handle = this.aHandle[handleRef],
				matPuppet_LayerParent = this.aMatPuppet_LayerParent[handleRef],
				automate = aAutomation0 ? aAutomation0[handleRef] : null,
				mat_Handle = mat3.clone(aMatLayerRoot_Handle[handleRef]);


			if (matPuppet_LayerParent) mat3.multiply(matPuppet_LayerParent, aMatLayerRoot_Handle[handleRef], mat_Handle);
			handle.privateSetMatrixRelativeToParent(mat_Handle, automate);

			while (++handleRef < length) {
				var parentRef = this.aParentRef[handleRef],
					matParent_LayerRoot = aMatHandle_LayerRoot[parentRef];

				automate = aAutomation0 ? aAutomation0[handleRef] : null;

				if ( isUndefinedMatrix(matParent_LayerRoot) ) {
					matParent_LayerRoot = invertOrNull(aMatLayerRoot_Handle[parentRef]);
					aMatHandle_LayerRoot[parentRef] = matParent_LayerRoot;
				}

				if (matParent_LayerRoot !== null) { // if invertible
					var matLayerRoot_Handle = aMatLayerRoot_Handle[handleRef];

					matPuppet_LayerParent = this.aMatPuppet_LayerParent[handleRef];
					handle = this.aHandle[handleRef];
					mat3.multiply(matParent_LayerRoot, matLayerRoot_Handle, mat_Handle);
					if (matPuppet_LayerParent) mat3.multiply(matPuppet_LayerParent, mat_Handle, mat_Handle);
					handle.privateSetMatrixRelativeToParent(mat_Handle, automate);
				} else {
					// if parent is not invertible then neither are its descendent handles
					aMatHandle_LayerRoot[handleRef] = null;
				}
			}

			disableRelativeTo();
		},

		propagateHandleAutomation : function (handleRef, automate) {
			var automateKid = {};
			if ( automate.translation === false ) {
				automateKid.translation = false;
			}

			if ( automate.linear === false ) {
				automateKid.translation = false;
				automateKid.linear = false;
			}

			// walk handle subtree and update automation of each descendent
			var kidRef = handleRef + 1,
				length = this.aHandle.length;

			while (
				kidRef < length &&							// there are more kids to test
				(this.aParentRef[kidRef] >= handleRef) 		// and kids are descendents of handle
			) {
				var kid = this.aHandle[kidRef];
				kid.auto(automateKid);
				++kidRef;
			}
		},

		getTreeAutomation : function () {
			var	handleRef = -1,
				length = this.aHandle.length,
				aAutoAttribute = [];

			// walk downward (to leaves) to accumulate matrices
			while (++handleRef < length) {
				var handle = this.aHandle[handleRef];
				aAutoAttribute.push(handle.getAutoAttribute());
			}

			return aAutoAttribute;
		}

	});

	return HandleTreeArray;
});
