/*
	Handle:

	This object implements the handle hierarchy.
*/


// TODO: create a mixin to remove repetition between Node and Handle object
define(["src/math/Mat3", "src/math/Vec2", "src/utils",
			"src/build/treeLeaf", "src/build/treeContainer", "lodash", "src/build/StageId"],
function (m3, v2, utils,
			treeLeaf, treeContainer, lodash, StageId) {
	"use strict";

	// set to true to enable various sanity checks
	// set to false for faster performance wihout sanity checks
	var kVERIFY = true,
		arrayPush = Array.prototype.push,
		genericSetParent;

	function fnConstant (value) {
		return function () { return value; };
	}

	function rotationAngle (rotation) {
		var angle = Math.atan2(rotation[1], rotation[0]);
		return angle;
	}

	function manualPosition_(handle) {
		var i, ci;

		// manually specifies position for itself...
		handle.auto({ translation : false });
		// and its children	
		for (i = 0; i < handle.children_.length; i += 1) {
			ci = handle.children_[i];
			if (ci.puppet_ === handle.puppet_) {
				manualPosition_(ci);
			}
		}
	}

	function manualAll_(handle) {
		var i, ci;

		// manually specifies all for itself...
		handle.auto({ translation : false, linear : false});
		// and its children	
		for (i = 0; i < handle.children_.length; i += 1) {
			ci = handle.children_[i];
			if (ci.puppet_ === handle.puppet_) {
				manualAll_(ci);
			}
		}
	}

	function manualScaleAndAngle_(handle) {
		var i, ci;
		// manually specifies rotation/scale for itself...
		handle.auto({ linear : false });
		// and position, rotation/scale for its children		
		for (i = 0; i < handle.children_.length; i += 1) {
			ci = handle.children_[i];
			if (ci.puppet_ === handle.puppet_) {
				manualAll_(ci);
			}
		}
	}

	function verifyMatrix_(mat) {
		var i,
			n = mat.length;

		if (n !== 9) {
			throw new Error("verifyMatrix: local not 3x3 matrix.");
		}

		for (i = 0; i < n; i += 1) {
			if (typeof mat[i] !== "number") {
				throw new Error("verifyMatrix: non-number entry.");
			}

			if (Number.isNaN(mat[i])) {
				throw new Error("verifyMatrix: NaN entry.");
			}
		}
	}

	function decomposeAffine(mat, position, scale, shear, rotation) {
		m3.decomposeAffine(mat, position, scale, shear, rotation);

		// verify
		if (kVERIFY) {
			var angle, recomposed;
			angle = rotationAngle(rotation);
			recomposed = m3.affine(position, scale, shear, angle, []);
			if (!m3.equalsApproximately(mat, recomposed)) {
				throw new Error("decomposeAffine: recompose matrix doesn't match input matrix");
			}
		}
	}

	function Handle(options0) {
		var options = lodash.defaults({}, options0, {
			name : "unnamed Handle",
			puppet : null,
			originParent_Local : [0, 0],
			origin : [0, 0], 
			scale : [1, 1], 
			shear : [0, 0], 
			angle : 0,
			automate : { translation : true, linear : true},
			tagBackstageId : null
		});

		this.name_ = options.name;
		this.tagBackstageId_ = options.tagBackstageId;
		this.puppet_ = options.puppet;
		this.origin_ = options.origin;
		this.matParent_Local_ = m3.affine(options.originParent_Local, [1, 1], [0, 0], 0);

		// represent transform as a local matrix, BUT
		// this may be inconvenient for keeping track of wind-up rotation which could be needed for motion blur or speedlines 
		this.local_ = m3.affine(options.origin, options.scale, options.shear, options.angle);
		if (kVERIFY) { verifyMatrix_(this.local_); }

		this.pAutomation = options.automate;
		this.parent_ = null;
		this.children_ = [];

		this.StageId("Handle_");
	}

	utils.mixin(Handle, treeLeaf(), treeContainer(), StageId);
	genericSetParent = Handle.prototype.setParent;

	utils.mixin(Handle, {
		setName: function (name) {
			this.name_ = name;
		},

		// const return value
		getName: function () {
			return this.name_;
		},

		getTagBackstageId : function () {
			var id = this.tagBackstageId_;
			return id ? id.slice(0) : null;
		},

		setPuppet: function (p) {
			this.puppet_ = p;
		},

		getPuppet: function () {
			return this.puppet_;
		},

		// HACK: refactor along with parentLayer and parentPuppet fields.
		getWarperLayer : function () {
			return this.getPuppet().getWarperLayer();
		},

		// TODO: remove as it no longer appears to need this override
		// overrides generic Tree default
		setParent: function (parent) {
			genericSetParent.call(this, parent);
		},

		getMatrix: function (result0) {
			return m3.initWithArray(this.local_, result0 || []);
		},

		getMatrixAtRestRelativeToParent : function (result0) {
			var result = m3.translation(this.origin_, result0);
			return m3.multiply(this.matParent_Local_, result, result);
		},

		getMatrixRelativeToParent : function (result0) {
			var result = result0 || [];
			m3.multiply(this.matParent_Local_, this.getMatrix(), result);
			return result;
		},

		setPosition: function (position) {
			m3.setTranslation(position, this.local_);
			if (kVERIFY) { verifyMatrix_(this.local_); }
			manualPosition_(this);
			// manualParent_(this);
		},

		getPosition: function (result0) {
			return m3.getTranslation(this.local_, result0);
		},

		setScale: function (scale) {
			var oldPosition = [], oldScale = [], oldShear = [], oldRotation = [];
			decomposeAffine(this.local_, oldPosition, oldScale, oldShear, oldRotation);

			m3.affine(oldPosition, scale, oldShear, rotationAngle(oldRotation), this.local_);
			if (kVERIFY) { verifyMatrix_(this.local_); }

			manualScaleAndAngle_(this);
			// manualParent_(this);
		},

		getScale: function () {
			var position = [], scale = [], shear = [], rotation = [];
			decomposeAffine(this.local_, position, scale, shear, rotation);

			return scale;
		},


		setShear: function (shear) {
			var oldPosition = [], oldScale = [], oldShear = [], oldRotation = [];
			decomposeAffine(this.local_, oldPosition, oldScale, oldShear, oldRotation);

			m3.affine(oldPosition, oldScale, shear, rotationAngle(oldRotation), this.local_);
			if (kVERIFY) { verifyMatrix_(this.local_); }

			manualScaleAndAngle_(this);
			// manualParent_(this);
		},

		getShear: function () {
			var position = [], scale = [], shear = [], rotation = [];
			decomposeAffine(this.local_, position, scale, shear, rotation);

			return shear;
		},

		setAngle: function (angle) {
			var oldPosition = [], oldScale = [], oldShear = [], oldRotation = [];
			decomposeAffine(this.local_, oldPosition, oldScale, oldShear, oldRotation);

			m3.affine(oldPosition, oldScale, oldShear, angle, this.local_);
			if (kVERIFY) { verifyMatrix_(this.local_); }

			manualScaleAndAngle_(this);
			// manualParent_(this);
		},

		setAngleWithMatrix: function (mat) {
			var oldPosition = [], oldScale = [], oldShear = [], oldRotation = [];
			decomposeAffine(this.local_, oldPosition, oldScale, oldShear, oldRotation);

			m3.affine(oldPosition, oldScale, oldShear, rotationAngle(mat), this.local_);
			if (kVERIFY) { verifyMatrix_(this.local_); }

			manualScaleAndAngle_(this);
			// manualParent_(this);
		},

		getAngle: function () {
			var position = [], scale = [], shear = [], rotation = [];
			decomposeAffine(this.local_, position, scale, shear, rotation);

			return rotationAngle(rotation);
		},

		auto: function (setAuto0) {
			if (setAuto0) {
				lodash.assign(this.pAutomation, setAuto0);
				utils.assert(lodash.size(this.pAutomation) === 2, "auto(): invalid attribute.");
			}
			return lodash.clone(this.pAutomation);
		},

		/**
		 * Return automation property as a logically ORed attribute.
		 */
		getAutoAttribute : function () {
			/*jshint bitwise: false */
			var bit = 0,
				attribute = 0;

			attribute |= this.pAutomation.translation << bit++;
			attribute |= this.pAutomation.linear << bit++;

			return attribute;
		},

		// is some aspects of this handle automatically determined
		isAutomated: function () {
			return this.pAutomation.translation || this.pAutomation.linear;
		},

		// is this handle under full manual control
		// note: isManual is not opposite of isAutomated when handle is partially manual and partially automated
		isManual: function () {
			return !(this.pAutomation.translation && this.pAutomation.linear);
		},

		/**
		 * Set local matrix
		 */
		setMatrix: function (mat) {
			m3.initWithArray(mat, this.local_);
			manualAll_(this);
			// manualParent_(this);
		},

		/**
		 * Set local handle matrix with accumulated matrix relative to the parent.
		 * @param mat Desired value for the accumulated matrix.
		 * @param automate0
		 */
		privateSetMatrixRelativeToParent: function (mat, automate0) {
			// TODO: precompute this inverse
			var matLocal_Parent = m3.invert(this.matParent_Local_);

			m3.multiply(matLocal_Parent, mat, this.local_);
			if (automate0) this.auto(automate0);
			if (kVERIFY) { verifyMatrix_(this.local_); }
		},

		clone : function (clone_children, result) {
			if (result) {
				// init
				Handle.call(result);
			} else {
				// alloc and init
				result = new Handle();
			}

			result.name_ = this.name_.slice(0);
			result.matParent_Local_ = m3.clone(this.matParent_Local_);
			result.origin_ = v2.clone(this.origin_);
			result.local_ = m3.clone(this.local_);
			result.pAutomation = lodash.clone(this.pAutomation);
			result.tagBackstageId_ = this.getTagBackstageId();

			// puppet must be set explicitly after construction
			result.puppet_ = null; 

			// parent must be set explicitly after construction
			delete result.parent_;	// remove the parent...
			result.setParent(null);	// ...so that this call works correctly
			
			// deep clone (by default) all descendent handles
			clone_children = typeof clone_children === "boolean" ? clone_children : true;
			if (clone_children) {
				result.children_ = lodash.map(this.children_, function (hi) {
					var handleChild = hi.clone(true);
					handleChild.setParent(result);
					return handleChild;
				});
			}

			return result;
		},

		gatherHandleTreeArray : function () {
			var aHandle = [],
				aParentRef = [];

			// Just like preOrderEach but modified to create array of parent indices
			// DRY (?): if needed, refactor into preOrderEach by providing additional
			// arguments to the callback: arg, argIndex, parentIndex, ...
			var arg = this,
				parentRef = null,
				stackArg = [],
				stackRef = [];

			while (arg) {
				aHandle.push(arg);
				parentRef = aParentRef.push(parentRef) - 1;

				var kids = arg.getChildren && arg.getChildren().slice(0);
				arg = kids.shift();
				if (arg) {
					arrayPush.apply(stackArg, kids.reverse());
					arrayPush.apply(stackRef, kids.map(fnConstant(parentRef)));
				} else {
					arg = stackArg.pop();
					parentRef = stackRef.pop();
				}
			}

			return {
				aHandle : aHandle,
				aParentRef : aParentRef
			};
		}
	});

	return Handle;

});
