define([	"lib/Zoot",		"src/math/Random",	"lodash"],
function(	Z,				Random,				lodash) {
	"use strict";

/** 
 *
 * NOTES:
 * We remove scale from any transform passed to simulation.
 * That way scale only affects rendering and warping but not simulation.
 *
 * For now this is a low level behavior like any other and so requires prioritization in order. 
 * Later promote physics to a "main sim” global behavior that will gather suggested transforms from other behaviors
 * and mediate.
 * 
 * TODO:
 * Scaling up/down should have no effect on simulation, but that's not the case.
 *
**/

	function getMatScene_Container (args, layer) {
		var matScene_Layer = args.getLayerMatrixRelativeToScene(layer);
		return mat3.removeScale(mat3.multiply(matScene_Layer, layer.getSourceMatrixRelativeToLayer()));
	}

	var kAbout = "Dangle, (c) 2014.",
		kTagName = "dangle",
		kTagRE = new RegExp(kTagName, "i"),
		mat3 = Z.Mat3,
		arrayPush = Array.prototype.push;

	return {
		about: kAbout,
		description: "$$$/Animal/Behavior/RigDynamics/Desc=Uses physics simulation to move Dangle handles naturally in reponse to movement",
		uiName: 	"$$$/Animal/Behavior/RigDynamics/UIName=Dangle",
		defaultArmedForRecordOn: true,
	
		defineParams: function () { 
			// free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{
					id: "targets", type: "handle", uiName: "Target Handles",
					dephault: { match: "//dangle" }
				},
				{
					id:"stiffness",	type:"slider", uiName:"Spring Stiffness",
					min:1, max:100, precision:0, dephault:100
				},
				{
					id:"pin", type:"slider", uiName:"Attachment Strength",
					min:0, max:10000, precision:0, dephault: 1000
				},
				{
					id:"iterations", type:"slider", uiName:"Number of Iterations",
					min:0, max:1000, precision:0, dephault: 100,
					hidden:true
				},
				{
					id:"windStrength", type:"slider", uiName: "Wind Strength", min: 0, precision: 1, dephault:0
				},
				{
					id:"windDirection", type:"angle", uiName:"Wind Direction", precision:0, dephault:90
				},
				{
					id:"windVariation", type:"slider", uiName:"Wind Variation", uiUnits:"%", min: 0, max:100, precision:1, dephault:0
				},
				{
                    id:"reset", type:"checkbox", uiName:"Reset",dephault:false,
					hidden:true
				}
			];
		},
		
		onCreateBackStageBehavior: function (/*self*/) {
			// TODO: why does it need to run first?  we were expecting sims to run last in order to mediate handle motion.
			// TODO: there are other behaviors with the same order, would this still work if they had different alphabetical order?
			return {
				order : 1.0,
				importance : 1.0
			};
		},
		
		onCreateStageBehavior: function (self/*, args*/) {
			// here you might initialize simulation state
			self.simulations = [];
			self.rand = new Random(123456789);
		},
		
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			var springStiffness = Math.pow(1.05,args.getParam("stiffness")),
				pinStiffness = args.getParam("pin"),
				windAngle = 2*Math.PI/360*args.getParam("windDirection") - Math.PI/2 + args.getParam("windVariation")/100*(1-2*self.rand.random())*Math.PI/2,
				windMag = args.getParam("windStrength"),
				wind = [windMag*Math.cos(windAngle),windMag*Math.sin(windAngle)],
				robustify = args.getParam("reset"),
				mouseTagRE = new RegExp("mousetrack", "i");

			if(self.simulations.length === 0){
				args.stageLayer.forEachLayerBreadthFirst(function (sdkLayer) {

					var layer = sdkLayer.privateLayer;

					// skip over layers without display container
					var displayContainer = layer.getDisplayContainer();
					if (! displayContainer) return;

					// skip over Layers without warper container
					var warperContainer = layer.getWarperContainer();
					if (! warperContainer) return;

					// skip if no warping
					var warperContainerCanWarp = warperContainer.canWarp();
					if (! warperContainerCanWarp) return;

					// skip over containers without handles
					// TODO: verify if we can now handle puppets with just one handle
					var	aHandle = layer.gatherHandleLeafArray();
					if (! aHandle || aHandle.length < 2) return;

					var aDynamicIds = [], 
						aKinematicIds = [], 
						aDofs = [],
						matScene_Container =  getMatScene_Container(args, layer),
						aMatContainer_Handle = layer.gatherHandleLeafMatrixArray("puppet");

					lodash.forEach(aHandle, function (hi, i) {
						var mi = aMatContainer_Handle[i];
						arrayPush.apply(aDofs, mat3.removeScale(mi));
						if (kTagRE.test(hi.getName())) { // TODO: switch to a real handle param so matching algorithm is whole word
							aDynamicIds.push(i);
						} else {
							aKinematicIds.push(i);
						}
					});

					if (aDynamicIds.length > 0) {

						aHandle.forEach(function (h) {
							if (mouseTagRE.test(h.getName())) {
								h.auto({ translation : false, linear : true });	
							} 
							else {
								h.auto({ translation : false, linear : false });
							}
						});
						

						self.simulations.push({
							rigDynamics : nmlImpl.newRigDynamics(matScene_Container, warperContainer, aDofs, aKinematicIds, {
								// TODO: set with an actual param
								timestep : 1.0 / 12.0,
								stiffness : springStiffness,
								pinStiffness : args.getParam("pin"),
								iterations : args.getParam("iterations")
							}),
							layer : layer,
							kinematicIds : aKinematicIds,
							kinematicDofs : [],
							lastMatScene_Container : [],
							name : layer.getName(),
							aDofs : aDofs
						});
					}
				});
			}	

			function getKinematicDofs (ids, aMatLayer_Handle) {
				var dofs = [];

				lodash.forEach(ids, function (id) {
					var mi = mat3.removeScale(aMatLayer_Handle[id]);
					// REMOVE AFTER DEBUGGING
					// var initial = self.getDofInitial(id, handles);
					// mat3.multiply(initial, mat3().setIdentity().rotate(0.5*Math.PI), mi);

					arrayPush.apply(dofs, mi);
				});

				return dofs;
			}

			function setKinematicDofs (ids, aKinematicDofs, aMatLayer_Handle) {
				lodash.forEach(ids, function (id, i) {
					aMatLayer_Handle[id] = aKinematicDofs.slice(i*9,(i+1)*9);
				});
			}

			self.simulations.forEach(function (sim) {
				var matScene_Container = getMatScene_Container(args, sim.layer),
					aMatContainer_Handle = sim.layer.gatherHandleLeafMatrixArray("puppet"),
					alpha = 0.25; // todo: make weight hidden live param

				if(sim.kinematicDofs.length === 0){
					sim.lastMatScene_Container = mat3.clone(matScene_Container);
				} 
				sim.kinematicDofs = getKinematicDofs(sim.kinematicIds, aMatContainer_Handle);
				// console.log("kinematicDofs: " + sim.kinematicDofs);

				// exponential smooth
				matScene_Container = mat3.removeScale(mat3.scaleAdd(1-alpha, sim.lastMatScene_Container, alpha, matScene_Container));
				// console.log(sim.name + ":" + matScene_Container);
				
				sim.rigDynamics.setKinematicHandles(matScene_Container, sim.kinematicIds, sim.kinematicDofs);

				sim.lastMatScene_Container = mat3.clone(matScene_Container);
				
				sim.rigDynamics.setBodyForce(wind[0],wind[1]);
				if(sim.stiffness  !== springStiffness){
					sim.rigDynamics.updateSpringStiffness(springStiffness);
			    	sim.stiffness = springStiffness;
				}
				if(sim.pinStiffness !== pinStiffness){
					sim.rigDynamics.updatePinStiffness(pinStiffness);
					sim.pinStiffness = pinStiffness;
				}

				if(robustify){
					sim.rigDynamics.resetShapeState();
				}

				sim.rigDynamics.step(aMatContainer_Handle);

				// now that we update kinDofs we project them back after each step to stop drift
				setKinematicDofs(sim.kinematicIds, sim.kinematicDofs, aMatContainer_Handle);

				sim.layer.scatterHandleLeafMatrixArray("puppet", aMatContainer_Handle);
			});

			if (self.simulations.length > 0) args.stageLayer.privateLayer.warp();

		}
	}; // end of object being returned
});
