
define(["lib/Zoot"],
function (Z) {
	'use strict';

	var kAbout = "ObjectTracker, (c) 2014.",
		kId = "target",
		stringToMatch = "objecttrack",
        getTargetFromColor, onAnimateColor, altKeyInputCode = Z.keyCodes.getKeyGraphId(Z.keyCodes.altKey);

	getTargetFromColor = function (inTargetName, inColorName) {
        return inTargetName + "_" + inColorName;
    };
    
    onAnimateColor = function (self, inColorName, args) { // method on behavior that is attached to a puppet, only onstage
        var colorId = getTargetFromColor(kId, inColorName.toLowerCase()), visibleB, objPosition, 
			objScale, scaleStrength, scale, angle, altDown, 
			startPosId = inColorName+'startPosition', handleDownPosId = inColorName+'handleDownPos',
			angle_offset, x_offset, y_offset, baseScale = 0.1; 
			/*baseScale is the size ratio between the object and webcam frame height when the cards are held at normal distance to the camera*/

        if (!self[colorId]) {
            return;
        }


        visibleB = args.getParamEventValue("CameraInput", inColorName + "/Visible");
        if (visibleB) {
			if (args.getParam("EnablePos"+inColorName)) {
                // Event graph stores the position in the [-0.5,0.5] y-up coordinate system. We need to transform the position into the scene coordinates 
                var view = args.stagePuppet.getView();
				var width = view.getWidth();
                var height = view.getHeight();
                var posX = args.getParamEventValue("CameraInput", inColorName + "/PosX") * height + width / 2;
                var posY = height / 2 - args.getParamEventValue("CameraInput", inColorName + "/PosY") * height;
				x_offset = (args.getParam(inColorName.toLowerCase()+"XOffset") || 0)*0.01*height;
				y_offset = (args.getParam(inColorName.toLowerCase()+"YOffset") || 0)*0.01*height;
				objPosition = [posX+x_offset, posY+y_offset];
				if (args.getParam("EnableRelativeMovement")) {
					if (self[startPosId] !== undefined) {
						var delta = Z.Vec2.subtract(objPosition, self[startPosId], []);
						self[colorId].forEach(function (h, i) {
							h.setPositionRelativeTo("kPuppetHandle", Z.Vec2.add(self[handleDownPosId][i], delta, []));
						});
					} else {		
						// set start position
						self[startPosId] = Z.Vec2.clone(objPosition);
						self[handleDownPosId] = self[colorId].map(function (h) {
							return h.getPositionRelativeTo("kPuppetHandle");
						});
					}
                } else {
					self[colorId].forEach(function (h, i) {
                    	h.setPositionRelativeTo("kPuppetHandle", objPosition);
					});
				}
			}

			
			if (args.getParam("EnableScale"+inColorName)) {
				scaleStrength = (args.getParam("scale"+inColorName) || 0)*0.01;
				scale = args.getParamEventValue("CameraInput", inColorName + "/Scale") / baseScale * scaleStrength; /* scale is 100% at normal distance */
			} else {
				scale = 1;
			}
			objScale = [scale, scale];
            
			angle_offset = args.getParam(inColorName.toLowerCase()+"AngleOffset") || 0;
			angle = (args.getParamEventValue("CameraInput", inColorName + "/Rotation") + angle_offset) / 180 * Math.PI;
        } else {
			self[startPosId] = undefined;
		}

		altDown = args.getParamEventValue("CameraInput", altKeyInputCode) || 0;

        visibleB = visibleB && !altDown;

		
        if (visibleB) {      
            if (objScale !== undefined) {
                self[colorId].forEach(function (h, i) {
                    h.setScale(objScale);
                });
            }

            if (angle !== undefined) {
                self[colorId].forEach(function (h, i) {
                    h.setAngle(angle);
                });
            }

        }
    };

    return {
		about: kAbout,
		description: "Track printed pattern",
		uiName: 	"Object Tracker",
		hideInBehaviorList: true,
		defaultArmedForRecordOn: true,

		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			return [
				{	id: "CameraInput",
					type: "eventGraphInput", 
					uiName:"Camera Input",
					inputKeysArray: ["Red/", "Blue/", altKeyInputCode],
					uiToolTip: "Analyzed object tracking data from the camera",
				 	defaultArmedForRecordOn: true
				},
				{
					id: getTargetFromColor(kId, "red"),
					type: "handle",
					uiName: "Red Target",
					dephault: { match: "//" + getTargetFromColor(stringToMatch, "red") }
				},
				{
					id: getTargetFromColor(kId, "blue"),
					type: "handle",
					uiName: "Blue Target",
					dephault: { match: "//" + getTargetFromColor(stringToMatch, "blue") }
				},	
				{
					id:"EnableRelativeMovement",
					type:"checkbox",
					uiName:"Enable Relative Movement",
					dephault:false
				},
				{
					id:"EnableScaleRed",
					type:"checkbox",
					uiName:"Enable Scale Red",
					dephault:true
				},
				{
					id:"scaleRed", 
					type:"slider", 
					uiName:"Scale Strength Red", 
					uiUnits:"%",
					min:0, 
					max:500, 
					precision:1,
					dephault:100
				},
				{
					id:"EnablePosRed",
					type:"checkbox",
					uiName:"Enable Position Red",
					dephault:true
				},
				{
					id:"redXOffset", 
					type:"slider", 
					uiName:"X Offset Red", 
					uiUnits:"%",
					precision:1, 
					dephault:0
				},
				{
					id:"redYOffset", 
					type:"slider", 
					uiName:"Y Offset Red", 
					uiUnits:"%",
					precision:1, 
					dephault:0
				},	
				{
					id:"redAngleOffset", 
					type:"slider", 
					uiName:"Angle Offset Red", 
					precision:1, 
					dephault:0
				},
				{
					id:"EnableScaleBlue",
					type:"checkbox",
					uiName:"Enable Scale Blue",
					dephault:true
				},
				{
					id:"scaleBlue", 
					type:"slider", 
					uiName:"Scale Strength Blue", 
					uiUnits:"%",
					min:0, 
					max:500, 
					precision:0,
					dephault:100
				},
				{
					id:"EnablePosBlue",
					type:"checkbox",
					uiName:"Enable Position Blue",
					dephault:1
				},
				{
					id:"blueXOffset", 
					type:"slider", 
					uiName:"X Offset Blue", 
					uiUnits:"%",
					precision:1, 
					dephault:0
				},
				{
					id:"blueYOffset", 
					type:"slider", 
					uiName:"Y Offset Blue", 
					uiUnits:"%",
					precision:1, 
					dephault:0
				},	
				{
					id:"blueAngleOffset", 
					type:"slider", 
					uiName:"Angle Offset Blue",
					precision:1, 
					dephault:0
				}
			];
		},

		onCreateBackStageBehavior: function (self) {
			// method on backstage behavior just before it is attached to a puppet; self is the new backstage behavior
			// which will contain these properties: boundPuppet, projectBehavior, getParam(id)/setParam(id, value) [or just allow this.id syntax?], addParam(), removeParam(),  
			// here you might walk the handles of your puppet and bind handle parameters or change defaults accordingly
			// the already-bound other behaviors can be examined, and this function can return a suggested index for its order,
			// and also its priority (i.e. how many simulation time-slices per rendered frame)
		},

		onCreateStageBehavior: function (self, args) {
			// method on stage behavior just before it is instanced on stage; self is the new stage behavior which
			// contains these properties: backstageBehavior, getParam(id)/setParam(id, value) [or just allow this.id syntax?], addParam(), removeParam(). 
			// here you might initialize simulation state
            var redId = getTargetFromColor(kId, "red"), blueId = getTargetFromColor(kId, "blue");
			self[redId] = args.getStaticParam(redId);
			self[blueId] = args.getStaticParam(blueId);
		},

		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
            onAnimateColor(self, "Red", args);
            onAnimateColor(self, "Blue", args);
		}

	}; // end of object being returned
});
