define(["src/math/Vec2", "src/math/Mat3", "lodash", "src/utils", "src/build/VisualMarks"],
function (vec2, mat3, lodash, utils, VisualMarks) {
	"use strict";

	var MouseUp = {
		kReturn : 0,
		kHold : 1
	};

	function HandleSelector (aHandle) {
		this.aHandle = aHandle;
		this.tMatScene_Handle = null;
	}

	utils.mixin(HandleSelector, {
		isEmpty : function () {
			return this.aHandle.length < 1;
		},

		resetInitialPose : function (behaviorArg) {
			var tMatScene_Handle = {};
			this.aHandle.forEach(function (handle) {
				var id = handle.getStageId();
				tMatScene_Handle[id] = behaviorArg.getHandleMatrixRelativeToScene(handle);
			});
			this.tMatScene_Handle = tMatScene_Handle;
		},

		getInitialMatrix : function (handle) {
			return this.tMatScene_Handle[handle.getStageId()];
		},

		// find handle closest to the location
		find : function (location) {
			var result = this.aHandle.reduce(function (candidate, handle) {

					var warper = handle.getWarperLayer();
					// skip over targets on hidden layers
					if ( !warper.getVisible() ) return candidate;

					var matScene_Handle = this.getInitialMatrix(handle),
						position = mat3.getTranslation(matScene_Handle, vec2()),
						offset = position.subtract(location),
						proximity = offset.magnitudeSquared();
						
					if ( !candidate || proximity < candidate.proximity) {
						candidate = candidate || {};
						candidate.proximity = proximity;
						candidate.handle = handle;
						candidate.offset = offset;
					}
					return candidate;
				}.bind(this), null);

			return result;
		}
	});

	function showTargets (self, behaviorArg) {
		var markSize = 20,
			markOpacity = 1.0;

		if (behaviorArg.getParam("showTargets")) {
			if (self.targetMarks) {
				// update
				self.targetMarks.forEach(function (mark, index) {
					mark.setMatrix(behaviorArg.getHandleMatrixRelativeToScene(self.targets[index]));
				});
			} else {
				// create
				self.targetMarks = new VisualMarks({ opacity : markOpacity });
				self.targets.forEach(function (hi, index) {
					var mark = self.targetMarks.square(markSize);
					self.targetMarks.insertMark(mark);
					mark.setMatrix(behaviorArg.getHandleMatrixRelativeToScene(self.targets[index]));
				});

				self.targetMarks.addToDisplayItem(behaviorArg.stageLayer.privateLayer.getTrackItem());
			}
		} else {
			// remove
			if (self.targetMarks) {
				self.targetMarks.removeFromDisplayItem(behaviorArg.stageLayer.privateLayer.getTrackItem());
				self.targetMarks = null;
			}
		}
	}


	return {
		about: "MouseTracker, (c) 2014.",
		description: "$$$/Animal/Behavior/MouseTracker/Desc=Tracks mouse motion",
		uiName: 	"$$$/Animal/Behavior/MouseTracker/UIName=Mouse Tracker",
		defaultArmedForRecordOn: true,

		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			var params = [
				{ id: "MouseInput", type: "eventGraphInput", uiName: "Mouse Input", inputKeysArray: ["Mouse/"], uiToolTip: "Mouse input used to transform handles", defaultArmedForRecordOn: true},
				{
					id: "targets",
					type: "handle",
					uiName: "Target Handles",
					dephault: { match: "//mousetrack" }
				},
				{
					id: "onMouseUp",   
					type:"enum",
					uiName: "After Move",
					items: [ 
						{ id: MouseUp.kReturn, uiName: "Return to Rest"}, 
						{ id: MouseUp.kHold, uiName: "Hold in Place"}
					],
					dephault: MouseUp.kReturn
				},
				{
					id : "showTargets",
					type : "checkbox",
					uiName : "Show Targets",
					hidden : true,
					dephault : false
				},
			];

			return params;
		},

		onCreateBackStageBehavior: function (/*self*/) {
		},

		onCreateStageBehavior: function (self, args) {
			self.targets = args.getStaticParam("targets");
			self.selector = new HandleSelector(self.targets);
		},

		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			if (self.selector.isEmpty()) return;

			showTargets(self, args);

			var mouseVec,
				leftDownB = args.getParamEventValue("MouseInput", "Mouse/Down/Left"),
				mousePosition0 = args.getParamEventValue("MouseInput", "Mouse/Position");

			if (mousePosition0) mouseVec = vec2(mousePosition0);

			if (leftDownB) {
				self.selector.resetInitialPose(args);

				if (!self.mouseDrag && mousePosition0) {
					args.setInputParamUsedDuringRecording("MouseInput");
					self.mouseDrag = true;
					self.selection = self.selector.find(mouseVec, args);
				}

				// move selected handle
				var handleSel = self.selection && self.selection.handle;
				if (self.mouseDrag && handleSel && mousePosition0) {
					// REMOVE AFTER DEBUGGING
					// console.log("target: " + handleSel);
					// console.log("offset: " + self.selection.offset);
					// console.log("mouse: " + mouseVec);

					var matScene_HandleSel = self.selector.getInitialMatrix(handleSel),
						ptScene_HandleSel = mat3.getTranslation(matScene_HandleSel),
						delta = mouseVec.add(self.selection.offset).subtract(ptScene_HandleSel);

					// REMOVE AFTER DEBUGGING
					// console.log("delta: " + delta);

					// delta = vec2([10, 0]);

					var layerSel = handleSel.getWarperLayer(),
						matLayerSel_Scene = mat3.invert(args.getLayerMatrixRelativeToScene(layerSel));

					layerSel.setHandleMatrix(
						handleSel, 
						mat3.multiply(matLayerSel_Scene, mat3.multiply(mat3.translation(delta), matScene_HandleSel)),
						{ translation : false }
					);
					args.stageLayer.privateLayer.warp();
				}
			} else {
				if (self.mouseDrag) {
					var onMouseUp = args.getParam("onMouseUp"),
						handle = self.selection && self.selection.handle;
					
					if (handle) {
						self.mouseDrag = false;
						switch (onMouseUp) {
							case MouseUp.kReturn:
								handle.auto({translation : true});
								args.stageLayer.privateLayer.warp();
							break;
							case MouseUp.kHold:
								// do nothing to hold in place
							break;
						}
					}
				}
			}
		}

	}; // end of object being returned
});
