// Adobe.LipSync.js
/*jslint sub: true */

define( ["lib/ReplacerUtilities"],
  function (ReplacerUtilities) {
		"use strict";

	var kReplacerPriority = 1.0,       // TODO: make this user-specified param 
		kReplacerDoesOverride = false, // TODO: make this user-specified param (what name?)

		mouthShapeLabels = [
			"L",
			"K",
			"Ee",
			"F",
			"W-Oo",
			"Oh-Uh", 
			"Th",
			"D",
			"M",
			"Ah"
		],

		// sequences of fall back mouth shape indices
		// whichever mouth shape index the facetracker returns, 
		// chooseMouthReplacements will use the first valid mouth shape in the corresponding sequence
		mouthShapeCascades = {
			0: [0, 6, 2, 1, -1],	// L
			1: [1, -1],				// K
			2: [2, 9, -1],			// Ee
			3: [3, -1],				// F
			4: [4, -1],				// W-Oo
			5: [5, 9, -1],			// Oh-Uh
			6: [6, 1, -1],			// Th
			7: [7, -1],				// D
			8: [8, -1],				// M
			9: [9, 5, 2, -1]		// Ah
		};

	function chooseMouthShape (args) {
		if (args.getParamEventValue("audioInput", "Viseme/InputEnabled")) {
			var visemeVal, visemeDuration, faceTrackerSetMouthSilenceDurationThreshInS = args.getParam("faceTrackerSetMouthSilenceDurationThreshInMilliseconds") / 1000;

			if (!args.isParamEventLive("audioInput"))
			{
				visemeDuration = args.getParamEventValue("audioInput", "Viseme/LookaheadHoldDuration");
				visemeVal = args.getParamEventValue("audioInput", "Viseme/LookaheadIndex");
			}
			else
			{
				visemeDuration = args.getParamEventValue("audioInput", "Viseme/HoldDuration");
				visemeVal = args.getParamEventValue("audioInput", "Viseme/Index");
			}
			args.setInputParamUsedDuringRecording("audioInput");

			// if we've detected silence for longer than faceTrackerSetMouthSilenceDurationThresh,
			// return an undefined visemeVal, which allows face tracker to control the mouth
			if (!mouthShapeCascades.hasOwnProperty(visemeVal) && 
				(visemeDuration < 0 || visemeDuration > faceTrackerSetMouthSilenceDurationThreshInS)) {
				return undefined;
			}

			return Math.round(visemeVal);
		}
		return undefined;
	}
	
	
	function makeValidIdFromLabel (str) {
		return str.replace(/[^\w]/g, "_");
	}
	
	function makeLayerIdFromLabel (str) {
		return "L_" + makeValidIdFromLabel(str);
	}
	
	function defineMouthLayerParams (bSorted) {
		var aParams = [];
		
		mouthShapeLabels.forEach(function (label) {
			// TODO: need to localize the uiName for each mouth shape
			aParams.push({id:makeLayerIdFromLabel(label), type:"layer", uiName:label, dephault:{match:"//"+label}, maxCount:1});
		});
		
		if (bSorted) { // WARNING: repeated in Adobe.FaceTracker.js
			aParams.sort(function (a, b)
							{ if (a.uiName < b.uiName) return -1;
								if (a.uiName > b.uiName) return 1;
								return 0; });
		}

		return aParams;
	}


	// returns array of all the mouth layer params that matched something
	function getMouthLayers (args) {
		var aMouthLayers = [], aMouthParams = defineMouthLayerParams();
		
		aMouthParams.forEach(function (param) {
			var aLayers = args.getParam(param.id),
				layer = aLayers[0]; // we only support one mouth per layer param
			
			if (layer) {
				aMouthLayers.push(layer);
			}
		});
		
		return aMouthLayers;
	}

	function chooseMouthReplacements (args, outNodesToShow, outNodesToHide) {
		var mouthIndexToShow, mouthCandidates, validMouth, visemeTypeVal = undefined;

		// returns actual mouth node & index for requested mouth index, which may not exist (uses fallbacks in mouthShapeCascades)
		function getClosestValidMouthContainerRoot(inMouthIndex) {

			var mouthIndicesToTry = [], i, mouthIndex, mouthShapeLabel,
				aLayers, layer;

			if (mouthShapeCascades.hasOwnProperty(inMouthIndex)) {
				mouthIndicesToTry = mouthShapeCascades[inMouthIndex];
			} else {
				mouthIndicesToTry = [inMouthIndex, -1];
			}

			for (i = 0; i < mouthIndicesToTry.length; i += 1) {
				mouthIndex = mouthIndicesToTry[i];
				if (mouthIndex >= 0 && mouthIndex < mouthShapeLabels.length) {
					mouthShapeLabel = mouthShapeLabels[mouthIndex];
					if (!mouthShapeLabel) {
						// this shouldn't happen, so adding error message if it does to help diagnose
						console.log("invalid mouth index: " + mouthIndex);
					} else {
						aLayers = args.getParam(makeLayerIdFromLabel(mouthShapeLabel));
						layer = aLayers[0];
						
						//console.logToUser(mouthShapeLabel);

						if (layer) {
							return {layer:layer, index:mouthIndex};
						}
					}
				}
			}

			return {layer:null, index:-1};
		}

		// gather all the mouth layer params that are not empty
		mouthCandidates = getMouthLayers(args);

		if (mouthCandidates.length > 1) {
			// get ideal mouth to show
			mouthIndexToShow = chooseMouthShape(args);

			// get valid (found) mouth that is closest to specified mouth to show
			validMouth = getClosestValidMouthContainerRoot(mouthIndexToShow);

			if (validMouth.layer) {
				outNodesToShow.push({"node" : validMouth.layer, "enabled" : true});
			}

			// send message to face tracker; indicate the viseme type detected by lip sync.
			// if it's undefined (i.e, we've detected silence for longer than faceTrackerSetMouthSilenceDurationThresh) 
			// face tracker is allowed to control the mouth replacements. otherwise, lip sync has detected 
			// either a non-silent viseme or short silence, and we should not let face tracker control the mouth.
			if (mouthIndexToShow !== undefined) {
				visemeTypeVal = (validMouth.layer) ? "nonSilent" : "shortSilence";
			}
			args.stageLayer.setSharedFrameData("Adobe.LipSync", { visemeType: visemeTypeVal });

			// hide any mouths that we are not trying to show
			mouthCandidates.forEach(function (c) {
				if (c !== validMouth.layer) {
					outNodesToHide.push({"node" : c, "enabled" : true});
				}
			});
		}

		// otherwise nothing to choose
	}

	function chooseFaceTrackerReplacements(self, args) {
		var nodesToShow = [], nodesToHide = [];
		chooseMouthReplacements(args, nodesToShow, nodesToHide);

		return {"nodesToShow" : nodesToShow, "nodesToHide" : nodesToHide};
	}

	function animateWithFaceTracker(self, args) {
		var replNodes = chooseFaceTrackerReplacements(self, args);

		ReplacerUtilities.applyReplacements(self, args, kReplacerPriority, kReplacerDoesOverride, replNodes.nodesToShow, replNodes.nodesToHide);
	}
	
	return {
		about:			"Lip Sync, (c) 2015.",
		description:	"$$$/Animal/Behavior/LipSync/Desc=Animates mouth replacements based on speech analysis; overrides mouths set by the Face behavior",
		uiName:			"$$$/Animal/Behavior/LipSync/UIName=Lip Sync",
		defaultArmedForRecordOn: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
			var aParams = [];
			aParams.push({id:"audioInput",	type:"eventGraphInput", uiName:"Audio Input", 
						  	inputKeysArray:["Viseme/"], uiToolTip:"Analyzed viseme data from audio input", defaultArmedForRecordOn: true});

			aParams.push({id:"faceTrackerSetMouthSilenceDurationThreshInMilliseconds", type:"slider", uiName:"Silence Duration Thresh", uiUnits:"ms", min:0, max:1000, precision:0, dephault:500,
							uiToolTip:"Minimum duration for silence before we allow the face tracker to control the mouth", hidden: true});

			
			var aMouths = defineMouthLayerParams("sorted");
			
			return aParams.concat(aMouths);
		},
		
		onCreateBackStageBehavior: function (self) {
			self.puppetInitTransforms = {};

			return { order: 0.5, importance : 0.0 }; // must come before FaceTracker
		},
		
		onCreateStageBehavior: function (self, args) {
			ReplacerUtilities.updateReplacerInfo(args);
		},
		
		onAnimate: function (self, args) {
			animateWithFaceTracker(self, args);
		}
		
	}; // end of object being returned
});
