// Adobe.FaceTracker.js
/*jslint sub: true */

define( [	"lib/Zoot",	"lib/ReplacerUtilities", "src/utils", "src/math/Vec2"],
  function (Zoot,		ReplacerUtilities,		utils,		vec2) {
		"use strict";

	var kReplacerPriority = 1.0,       // TODO: make this user-specified param 
		kReplacerDoesOverride = false, // TODO: make this user-specified param (what name?)

		V2 = Zoot.Vec2, // everyN = 0,

		// see https://bitbucket.org/amitibo/pyfacetracker/src/d54866d9b3e23654b1c06adca625dafcbe7629ce/doc/images/3DMonaLisa.png?at=default
		//	for diagram of these indices
//		faceFeatureIndices = {
//			"Head" :                    [0, 16], // jaw line going from puppet right to left
//			"Head/Right Eye" :          [36, 41],
//			"Head/Left Eye" :            [42, 47],
//			"Head/Right Eyebrow" :       [17, 21],
//			"Head/Left Eyebrow" :        [22, 26],
//			"Head/Nose" :                [27, 35], // 27-30 down ridge, 31-35 right to left
//			"Head/Mouth" :               [48, 65]
//		},

		// all possible face features, boolean value indicates whether
		// the feature is present in the puppet
		// TODO: remove "Head/" from this table, and have the FaceTracker behavior applied directly to the Head subpuppet.
		//			this would allow two FaceTrackers to be applied to two seperate head subpuppets with different names (but
		//			the same Right Eye, etc. names inside)
		//	Then again, having "Head/" allows applying the behavior to the top level. BOBW to have a handle param pointing to head
		//	and having all other layers searched for underneath that; then you could direct the handle param to a specific head if you
		//	want, or to N heads, and the behavior could be applied locally or at the root?
		//	Note: the hierarchy in these labels isn't used anymore (just the last component), and can be removed, but needs care because the text of the
		//		labels is also used in places like head14ToPuppetTransformMapping
		faceFeatureLabels = {		// this is a global read-only version; each behavior instance has a self.featureLabels of its own
			"Head" : false,
			"Head/Right Eye" : false, 
			"Head/Right Eye/Right Eyelid Top" : false,
			"Head/Right Eye/Right Eyelid Bottom" : false,
//			"Head/Right Eye/Right Iris" : false,		// only Pupil for now, until we have xpath "or"
			"Head/Right Eye/Right Pupil" : false,
			"Head/Left Eye" : false, 
			"Head/Left Eye/Left Eyelid Top" : false,
			"Head/Left Eye/Left Eyelid Bottom" : false,
//			"Head/Left Eye/Left Iris" : false,
			"Head/Left Eye/Left Pupil" : false,
			//"Head/Right Blink" : false,		    don't need these because there are no transforms related to them (just replacements)
			//"Head/Left Blink" : false,
			"Head/Right Eyebrow" : false, 
			"Head/Left Eyebrow" : false, 
			"Head/Nose" : false, 
			"Head/Mouth" : false
		},
		

		head14Labels = { // note: values are identity; repeated in .cpp code in FaceMetrics constructor (TODO: factor)
			"Head/DX" : 0,
			"Head/DY" : 0,
			"Head/DZ" : 0,
			"Head/Orient/X" : 0,
			"Head/Orient/Y" : 0,
			"Head/Orient/Z" : 0,
			"Head/LeftEyebrow" : 1,
			"Head/RightEyebrow" : 1,
			"Head/LeftEyelid" : 1,
			"Head/RightEyelid" : 1,
			"Head/MouthSX" : 1,
			"Head/MouthSY" : 1,
			"Head/MouthDX" : 0,
			"Head/MouthDY" : 0,
			"Head/Scale" : 1,
			"Head/MouthShape" : -1,
			"Head/LeftEyeGazeX" : 0,
			"Head/LeftEyeGazeY" : 0, 
			"Head/RightEyeGazeX" : 0, 
			"Head/RightEyeGazeY" : 0
		},

		// first three are default mouth shapes (match default mouth shape classifiers in beaker::zoot:FaceClassifiers)
		// TODO: eventually allow users to add to this list with appropriate named mouth shapes for customized facial expression recognition
		mouthShapeLabels = [
			"Neutral",
			"Surprised",
			"Smile"
		],

		// sequences of fall back mouth shape indices
		// whichever mouth shape index the facetracker returns, 
		// chooseMouthReplacements will use the first valid mouth shape in the corresponding sequence
		mouthShapeCascades = {
			0: [0], 
			1: [1, 0],
			2: [2, 0],
			
//			3: [3, 1, 0], commented out along with "Grimace" - "Tongue" above
//			4: [4, 0],
		},

		// stores mapping from head14 params to puppet transformations	
		// all transformation values are unitless; when transforms are applied
		// at runtime, they get multiplied by the appropriate puppet-specific measurements
		// note: no mappings for Left/Right Blink because it's only hidden or shown
		head14ToPuppetTransformMapping = { 
			"Head" :
			[
			{
				"labels" : ["Head/DX"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/DX" : -2 
					},
					"T" : {
						"translate" : [-2, 0], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/DX" : 2
					},
					"T" : {
						"translate" : [2, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			},
			{
				"labels" : ["Head/DY"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/DY" : -2 
					},
					"T" : {
						"translate" : [0, -2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/DY" : 2
					},
					"T" : {
						"translate" : [0, 2],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			},
			{
				"labels" : ["Head/Orient/Z"],
				"translateUnit" : "interocularDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/Orient/Z" : -1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1],
						"angle" : -1.5
					}
				},
				{
					"head14" : {
						"Head/Orient/Z" : 1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1], 
						"angle" : 1.5
					}
				}
				]
			}			
			],

			"Head/Nose" : 
			[
			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "noseDepth", 
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]);
					T.translate[1] += 0.5 * Math.sin(x[1]); 
				}
			}
			],

			"Head/Left Eyebrow" :
			[
			{
				"labels" : ["Head/LeftEyebrow"],
				"translateUnit" : "leftEyeEyebrowDist",
				"samples" :
				[
				{
					"head14" : {
						"Head/LeftEyebrow" : 0.5
					},
					"T" : {
						"translate" : [0, 2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},
				{
					"head14" : {
						"Head/LeftEyebrow" : 2
					},
					"T" : {
						"translate" : [0, -4], 
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}

			],

			"Head/Right Eyebrow" :
			[
			{
				"labels" : ["Head/RightEyebrow"],
				"translateUnit" : "rightEyeEyebrowDist",
				"samples" : [

				{
					"head14" : {
						"Head/RightEyebrow" : 0.5
					},
					"T" : {
						"translate" : [0, 2], 
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyebrow" : 2
					},
					"T" : {
						"translate" : [0, -4],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}

			],

			"Head/Right Eye" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1.2], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}
			],

			"Head/Right Eye/Right Eyelid Top" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"translateUnit" : "rightEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Head/Right Eye/Right Eyelid Bottom" :
			[
			{
				"labels" : ["Head/RightEyelid"],
				"translateUnit" : "rightEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Head/Right Eye/Right Pupil" :
			[
			{
				"labels" : ["Head/RightEyeGazeX"],
				"translateUnit" : "rightEyeWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyeGazeX" : -0.8
					},
					"T" : {
						"translate" : [-0.45, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyeGazeX" : 0.8
					},
					"T" : {
						"translate" : [0.45, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/RightEyeGazeY"],
				"translateUnit" : "rightEyeHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/RightEyeGazeY" : -0.8
					},
					"T" : {
						"translate" : [0, -0.3],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/RightEyeGazeY" : 0.6
					},
					"T" : {
						"translate" : [0, 0.225],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Head/Left Eye" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 1.2], 
						"angle" : 0
					}
				}

				]
			}, 

			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}
			],

			"Head/Left Eye/Left Eyelid Top" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"translateUnit" : "leftEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Head/Left Eye/Left Eyelid Bottom" :
			[
			{
				"labels" : ["Head/LeftEyelid"],
				"translateUnit" : "leftEyelidDist",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyelid" : 0.8
					},
					"T" : {
						"translate" : [0, -0.5],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyelid" : 1.2
					},
					"T" : {
						"translate" : [0, 0.5],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}
				]
			}
			],

			"Head/Left Eye/Left Pupil" :
			[
			{
				"labels" : ["Head/LeftEyeGazeX"],
				"translateUnit" : "leftEyeWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyeGazeX" : -0.8
					},
					"T" : {
						"translate" : [-0.45, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyeGazeX" : 0.8
					},
					"T" : {
						"translate" : [0.45, 0],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/LeftEyeGazeY"],
				"translateUnit" : "leftEyeHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/LeftEyeGazeY" : -0.8
					},
					"T" : {
						"translate" : [0, -0.3],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/LeftEyeGazeY" : 0.6
					},
					"T" : {
						"translate" : [0, 0.225],
						"scale" : [1, 1], 
						"angle" : 0
					}
				}

				]
			}
			],

			"Head/Mouth" :
			[
			{
				"labels" : ["Head/MouthDX"],
				"translateUnit" : "mouthWidth",				
				"samples" : [

				{
					"head14" : {
						"Head/MouthDX" : -1
					},
					"T" : {
						"translate" : [-1, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthDX" : 1
					},
					"T" : {
						"translate" : [1, 0],
						"scale" : [1, 1],
						"angle" : 0
					}
				}

				]
			},
			{
				"labels" : ["Head/MouthDY"],
				"translateUnit" : "mouthHeight",				
				"samples" : [

				{
					"head14" : {
						"Head/MouthDY" : -1
					},
					"T" : {
						"translate" : [0, -1],
						"scale" : [1, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthDY" : 1
					},
					"T" : {
						"translate" : [0, 1],
						"scale" : [1, 1],
						"angle" : 0
					}
				}

				]
			},

			{
				"labels" : ["Head/MouthSX"],
				"samples" : [

				{
					"head14" : {
						"Head/MouthSX" : 0
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [0, 1],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthSX" : 1.5
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1.5, 1],
						"angle" : 0
					}
				}

				]
			},
			{
				"labels" : ["Head/MouthSY"],
				"samples" : [

				{
					"head14" : {
						"Head/MouthSY" : 0
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 0],
						"angle" : 0
					}
				},

				{
					"head14" : {
						"Head/MouthSY" : 2
					},
					"T" : {
						"translate" : [0, 0],
						"scale" : [1, 2],
						"angle" : 0
					}
				}
				]
			}, 
			{
				"labels" : ["Head/Orient/Y", "Head/Orient/X"],
				"translateUnit" : "eyeDepth",
				"custom" : function(x, T) { 
					T.translate[0] -= 0.7 * Math.sin(x[0]); 
					T.translate[1] += 0.5 * Math.sin(x[1]);
				}				
			}			
			]
		};
	// end of var declarations
	
			
	function getHead14(args) {
		var label, v, vals = {};

		// use first label existence as a proxy for all existing
		if (args.getParamEventValue("cameraInput", "Head/InputEnabled")) {
			args.setInputParamUsedDuringRecording("cameraInput");
			for (label in head14Labels) {
				if (head14Labels.hasOwnProperty(label)) {
					v = args.getParamEventValue("cameraInput", label);
					if (v !== undefined) {
						vals[label] = v;
					} else { // else  missing part of head14 data 
						vals[label] = head14Labels[label];
					}
				}
			}
		} else {
			// return identity head14 -- perhaps camera tracking not spun up yet
			for (label in head14Labels) {
				if (head14Labels.hasOwnProperty(label)) {
					vals[label] = head14Labels[label];
				}
			}
		}
		
		return vals;
	}

	function getIdentityTransform() {
		var identityTransform = {};
		identityTransform.translate = [0, 0];
		identityTransform.scale = [1, 1];
		identityTransform.angle = 0;
		return identityTransform;
	}

	// get named puppetMeasurement
	function getPuppetMeasurement(self, inMeasurementName, args) {
		var measurement = null, puppetMeasurements = self.puppetMeasurements;
		
		if (inMeasurementName && puppetMeasurements[inMeasurementName]) {
			measurement = puppetMeasurements[inMeasurementName];

			// TODO: it's not ideal that some parameter adjustments happen here, while others
			// happen in computePuppetTransforms. should consolidate.
			// apply relevant parameter adjustments
			if (inMeasurementName === "noseDepth" || inMeasurementName === "eyeDepth") {
				measurement *= args.getParam("parallaxFactor") / 100;
			}
		}

		return measurement;
	} 

	// given a set of head14 params and a mapping entry with list of samples
	// determine the appropriate transform to return
	function computePuppetTransform(self, inHead14, inMappingEntry, args) {
		var finalTransform = getIdentityTransform(), transform, labels, translateUnit, samples, custom, customArgs, 
		translateRange, scaleRange, angleRange,
		i, j, v, inVec, vec1, vec2, range, offset, rangeNorm, alpha;

		for (i = 0; i < inMappingEntry.length; i += 1) {

			labels = inMappingEntry[i].labels;
			samples = inMappingEntry[i].samples;
			custom = inMappingEntry[i].custom;
			inVec = []; vec1 = []; vec2 = []; range = []; offset = [];
			translateRange = [0, 0];
			scaleRange = [0, 0];
			angleRange = 0;

			transform = getIdentityTransform();

			// interpolate based on mapping samples
			if (samples && samples.length >= 2) {

				V2.subtract(samples[samples.length-1].T.translate, samples[0].T.translate, translateRange);
				V2.subtract(samples[samples.length-1].T.scale, samples[0].T.scale, scaleRange);
				angleRange = samples[samples.length-1].T.angle - samples[0].T.angle;

				for (j = 0; j < labels.length; j += 1) {
					if (inHead14.hasOwnProperty(labels[j])) {
						inVec.push(inHead14[labels[j]]);
					}

					if (samples[0].head14.hasOwnProperty(labels[j])) {
						vec1.push(samples[0].head14[labels[j]]);
					}
					if (samples[samples.length-1].head14.hasOwnProperty(labels[j])) {
						vec2.push(samples[samples.length-1].head14[labels[j]]);
					}
				}

				if (inVec.length === labels.length && vec1.length === labels.length && vec2.length === labels.length) {

					rangeNorm = 0;
					for (j = 0; j < labels.length; j +=1 ) {
						range[j] = vec2[j] - vec1[j];
						offset[j] = inVec[j] - vec1[j];
						rangeNorm += (range[j] * range[j]);
					}

					rangeNorm = Math.sqrt(rangeNorm);

					// get dot product of offset onto range
					alpha = 0;
					for (j = 0; j < labels.length; j += 1) {
						alpha += offset[j] * (range[j] / rangeNorm);
					}
					alpha /= rangeNorm;

					// clamp
					if (alpha < 0) { alpha = 0; }
					if (alpha > 1) { alpha = 1; }

					// compute interpolated transform values
					V2.add(transform.translate, V2.add(samples[0].T.translate, V2.scale(alpha, translateRange, []), []), transform.translate);
					V2.xmy(transform.scale, V2.add(samples[0].T.scale, V2.scale(alpha, scaleRange, []), []), transform.scale);
					transform.angle += (samples[0].T.angle + (alpha * angleRange));
				}

			}
			else if (custom) {
				customArgs = [];
				for (j = 0; j < labels.length; j += 1) {
					v = inHead14[labels[j]];
					if (v === undefined) {
						console.log("missing FaceTracker custom value for " + labels[j]);
					} else {
						customArgs.push(v);
					}
				}

				custom(customArgs, transform);
			}

			// get translateUnit and scale transform
			translateUnit = getPuppetMeasurement(self, inMappingEntry[i].translateUnit, args);

			if (translateUnit) {
				V2.scale(translateUnit, transform.translate, transform.translate);
			}

			// add to finalTransform
			V2.add(finalTransform.translate, transform.translate, finalTransform.translate);
			V2.xmy(finalTransform.scale, transform.scale, finalTransform.scale);
			finalTransform.angle += transform.angle;
		}

		return finalTransform;
	}

	// get corresponding entry in head14ToPuppetTransformMapping
	function getHead14ToPuppetTransformEntry(inLabel) {
		var entry = null;
		if (head14ToPuppetTransformMapping.hasOwnProperty(inLabel)) {
			entry = head14ToPuppetTransformMapping[inLabel];
		}

		return entry;
	}

	// 1.0 means no change to the scale, 0.0 means scale will be identity (either 1.0 or -1.0), and 0.5
	//	means the scale will be half the strength (4 -> 2, 0.25 -> 0.5)
	function adjustScaleByFactor(scale, factor) {
		var negative, scaleDown;

		// handle 0.0 scale
		if (scale === 0) {
			if (factor > 0) {
				return scale;
			} else {
				return 1.0;
			}
		}

		if (factor === 1) {
			return scale;
		}
		
		negative = scale < 0;
		
		if (negative) {
			scale = -scale; // make it positive for a moment
		}
		
		scaleDown = scale < 1;
		
		if (scaleDown) {
			scale = 1/scale; // make it > 1 for a moment
		}
		
		// now we only need to deal with scale > 1 here
		scale = 1 + (scale - 1) * factor;

		if (scaleDown) {
			scale = 1/scale;
		}

		if (negative) {
			scale = -scale; // back to negative
		}
		
		return scale;
	}

	function adjustTransformByFactor(translate, factor) {
		return [translate[0] * factor, translate[1] * factor];
	}
	
	function applyFactorToTransform(inPosFactor, inScaleFactor, inRotFactor, inoutTransform) {
		// if getIdentityTransform() ever returns something other than actual identity, this function
		//	would need to be updated to do a LERP between that and the passed param
		inoutTransform.translate = adjustTransformByFactor(inoutTransform.translate, inPosFactor);
		inoutTransform.scale[0] = adjustScaleByFactor(inoutTransform.scale[0], inScaleFactor);
		inoutTransform.scale[1] = adjustScaleByFactor(inoutTransform.scale[1], inScaleFactor);
		inoutTransform.angle *= inRotFactor;
	}

	// customized version of applyParamFactorToNamedTransform that allows for adjusting pos, scale and rot factors.
	// the values in factors scale the effect of the param on the corresponding component of the transform.
	// e.g., factors.pos = 1 modifies translation by the param full param values, 
	//		and factors.pos = 0 does not modify translation at all
	function applyParamFactorToNamedTransformCustom(self, paramName, args, transforms, transformName, factors)
	{
		var t = transforms[transformName], factor;

		if (t) {
			factor = args.getParam(paramName) / 100;
			applyFactorToTransform(1 + (factor-1) * factors.pos, 1 + (factor-1) * factors.scale, 1 + (factor-1) * factors.rot, t);
		}		
	}

	// paramName is for a (currently root-level) param that has 100 as a "neutral" gain
	function applyParamFactorToNamedTransform(self, paramName, args, transforms, transformName) {
		var factors = { pos : 1, scale : 1, rot : 1 };
		applyParamFactorToNamedTransformCustom(self, paramName, args, transforms, transformName, factors);
	}

	function setTransformScale(transforms, transformName, scale) {
		var t = transforms[transformName];
		if (t) {
			t.scale = scale;
		}
	}

	function setTransformTranslate(transforms, transformName, translate) {
		var t = transforms[transformName];
		if (t) {
			t.translate = translate;
		}
	}

	function addTransformTranslate(transforms, transformName, translate) {
		var t = transforms[transformName];
		if (t) {
			t.translate[0] += translate[0];
			t.translate[1] += translate[1];
		}
	}

	// compute mouse eye gaze offset
	function computeNormalizedMouseEyeGazeOffset(self, args) {
		var	mouseVec, mouseEyeGazeOffset = [0, 0], 
			sceneCenter, deltaFromSceneCenter, distFromSceneCenter, angle, 			
			epsilon = 0.00001,
			stageView = args.stagePuppet.getView(),
            leftDownB = args.getParamEventValue("mouseEyeGaze", "Mouse/Down/Left"),
            mousePosition0 = args.getParamEventValue("mouseEyeGaze", "Mouse/Position");

        if (leftDownB && mousePosition0) {
			mouseVec = vec2(mousePosition0);
			args.setInputParamUsedDuringRecording("mouseEyeGaze");

        	// eye gaze determined by mouse position wrt a circle with 200px radius
        	// at the center of the scene. the perimeter of the circle represents 
        	// the boundary of the eye.

        	sceneCenter = vec2.initWithEntries(0.5 * stageView.getWidth(), 0.5 * stageView.getHeight(), vec2());
        	deltaFromSceneCenter = vec2.subtract(mouseVec, sceneCenter, vec2());
        	distFromSceneCenter = vec2.magnitude(deltaFromSceneCenter);
        	deltaFromSceneCenter = vec2.normalize(deltaFromSceneCenter, vec2());

        	// note: Math.atan seems to return a valid result (PI/2) even if you 
        	// call it with NaN (e.g., 1/0), but just to be safe, we're checking
        	// for deltaFromSceneCenter[0] > epsilon here and explicitly setting the angle. -wilmotli
			if (deltaFromSceneCenter[0] > epsilon) {
				angle = Math.atan(deltaFromSceneCenter[1]/deltaFromSceneCenter[0]);	
			} else if (deltaFromSceneCenter[0] < -epsilon) {
				angle =  -Math.PI + Math.atan(deltaFromSceneCenter[1]/deltaFromSceneCenter[0]);
			} else {
				angle = (deltaFromSceneCenter[1] > 0) ? 0.5 * Math.PI : -0.5 * Math.PI;
			}

        	// clamp to a 200px radius circle
        	distFromSceneCenter /= 200;
        	if (distFromSceneCenter > 1) distFromSceneCenter = 1;

        	// compute offset vector
        	mouseEyeGazeOffset[0] = distFromSceneCenter * Math.cos(angle);
        	mouseEyeGazeOffset[1] = distFromSceneCenter * Math.sin(angle);
        }

        return mouseEyeGazeOffset;
	}

	// compute transforms for puppet
	function computePuppetTransforms(self, args, head14) {
		var transforms = {}, transform, headTransform,
			faceFeatureLabel, head14ToPuppetTransformEntry,
			featureLabels = self.featureLabels, 
			normalizedMouseEyeGazeOffset, leftMouseEyeGazeOffset, rightMouseEyeGazeOffset, 
			leftEyeGazeRangeX = getPuppetMeasurement(self, "leftEyeGazeRangeX", args),
			leftEyeGazeRangeY = getPuppetMeasurement(self, "leftEyeGazeRangeY", args),
			rightEyeGazeRangeX = getPuppetMeasurement(self, "rightEyeGazeRangeX", args),
			rightEyeGazeRangeY = getPuppetMeasurement(self, "rightEyeGazeRangeY", args),
			mouseEyeGazeFactor = args.getParam("mouseEyeGazeFactor") / 100; 

		// apply parameter adjustments to puppet measurements
		for (faceFeatureLabel in featureLabels) {
			if (featureLabels.hasOwnProperty(faceFeatureLabel)) {
				head14ToPuppetTransformEntry = getHead14ToPuppetTransformEntry(faceFeatureLabel);
				if (featureLabels[faceFeatureLabel] && head14ToPuppetTransformEntry) {
					transform = computePuppetTransform(self, head14, head14ToPuppetTransformEntry, args);
					transforms[faceFeatureLabel] = transform;
				} 
			}
		}

		// 
		// adjust eyes
		//
		if (featureLabels["Head/Right Eye/Right Eyelid Top"] && featureLabels["Head/Right Eye/Right Eyelid Bottom"]) {
			setTransformScale(transforms, "Head/Right Eye", [1, 1]);
		} else {
			// not clear why this clause is needed, though harmless
			transforms["Head/Right Eye/Right Eyelid Top"] = getIdentityTransform();
			transforms["Head/Right Eye/Right Eyelid Bottom"] = getIdentityTransform();
		}
		if (featureLabels["Head/Left Eye/Left Eyelid Top"] && featureLabels["Head/Left Eye/Left Eyelid Bottom"]) {
			setTransformScale(transforms, "Head/Left Eye", [1, 1]);
		} else {
			transforms["Head/Left Eye/Left Eyelid Top"] = getIdentityTransform();
			transforms["Head/Left Eye/Left Eyelid Bottom"] = getIdentityTransform();
		}
		
		applyParamFactorToNamedTransform(self, "eyebrowFactor", args, transforms, "Head/Left Eyebrow");
		applyParamFactorToNamedTransform(self, "eyebrowFactor", args, transforms, "Head/Right Eyebrow");

		// when blink puppet is available, we don't want to scale the eye; TODO: this implies we should disable the eyeFactor param,
		//  but we currently don't have a way to do that (i.e. onCreateStageBehavior should return param UI hint that the param should
		//  be disabled.
		if (self.leftBlinkLayers.length > 0) {
			setTransformScale(transforms, "Head/Left Eye", [1, 1]);
			setTransformTranslate(transforms, "Head/Left Eye/Left Eyelid Top", [0, 0]);
			setTransformTranslate(transforms, "Head/Left Eye/Left Eyelid Bottom", [0, 0]);
		} else {
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Head/Left Eye", { pos : 0, scale : 1, rot : 0 });
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Head/Left Eye/Left Eyelid Top", { pos : 1, scale : 0, rot : 0 });
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Head/Left Eye/Left Eyelid Bottom", { pos : 1, scale : 0, rot : 0 });			
		}
		
		if (self.rightBlinkLayers.length > 0) {
			setTransformScale(transforms, "Head/Right Eye", [1, 1]);
			setTransformTranslate(transforms, "Head/Right Eye/Right Eyelid Top", [0, 0]);
			setTransformTranslate(transforms, "Head/Right Eye/Right Eyelid Bottom", [0, 0]);
		} else {
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Head/Right Eye", { pos : 0, scale : 1, rot : 0 });
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Head/Right Eye/Right Eyelid Top", { pos : 1, scale : 0, rot : 0 });
			applyParamFactorToNamedTransformCustom(self, "eyeFactor",	args, transforms, "Head/Right Eye/Right Eyelid Bottom", { pos : 1, scale : 0, rot : 0 });			
		}

		//
		// adjust eye gaze
		//
		applyParamFactorToNamedTransformCustom(self, "eyeGazeFactor",	args, transforms, "Head/Left Eye/Left Pupil", { pos : 1, scale : 0, rot : 0 });
		applyParamFactorToNamedTransformCustom(self, "eyeGazeFactor",	args, transforms, "Head/Right Eye/Right Pupil", { pos : 1, scale : 0, rot : 0 });

		// add mouse-based eye gaze
		normalizedMouseEyeGazeOffset = computeNormalizedMouseEyeGazeOffset(self, args);
		leftMouseEyeGazeOffset = vec2.initWithEntries(normalizedMouseEyeGazeOffset[0] * leftEyeGazeRangeX * mouseEyeGazeFactor, normalizedMouseEyeGazeOffset[1] * leftEyeGazeRangeY * mouseEyeGazeFactor, vec2());
		rightMouseEyeGazeOffset = vec2.initWithEntries(normalizedMouseEyeGazeOffset[0] * rightEyeGazeRangeX * mouseEyeGazeFactor, normalizedMouseEyeGazeOffset[1] * rightEyeGazeRangeY * mouseEyeGazeFactor, vec2());
		//
		addTransformTranslate(transforms, "Head/Right Eye/Right Pupil", leftMouseEyeGazeOffset);
		addTransformTranslate(transforms, "Head/Left Eye/Left Pupil", rightMouseEyeGazeOffset);

		//
		// adjust mouth
		//
		applyParamFactorToNamedTransform(self, "mouthFactor",	args, transforms, "Head/Mouth");

		//
		// adjust head
		//
		setTransformScale(transforms, "Head", [head14["Head/Scale"], head14["Head/Scale"]]);

		headTransform = transforms["Head"];
		if (headTransform) {
			applyFactorToTransform(
				args.getParam("headPosFactor") / 100, 
				args.getParam("headScaleFactor") / 100, 
				args.getParam("headRotFactor") / 100, 
				headTransform);
		}

		// DEBUG
		/* (uncomment var everyN far above)
		everyN += 1;
		if (everyN >= 1) {
			everyN = 0;
			// Put debugging output here
		}
		*/

		return transforms;
	}


	function applyPuppetTransforms(self, args, inTransforms) {
		var faceFeatureLabel, transform, initState, faceFeatureNode, 
			dof = [], position = [], scale = [], shear = [0, 0], angle;

		for (faceFeatureLabel in inTransforms) {
			if (inTransforms.hasOwnProperty(faceFeatureLabel)) {
				transform = inTransforms[faceFeatureLabel];
				initState = self.puppetInitTransforms[faceFeatureLabel];

				faceFeatureNode = getHandle(args, getLastComponentOfFeatureLabel(faceFeatureLabel));

				if (transform && initState && faceFeatureNode) {

					if (faceFeatureLabel === "Head") {
						V2.add( initState.position, V2.scale( 1, transform.translate, [] ), position );
						// TODO: will initState ever have a scale != 1 that we have to take into account?
						scale = transform.scale;
						angle = initState.angle + transform.angle;
						Zoot.Mat3.affine(position, scale, shear, angle, dof);
						faceFeatureNode.setMatrix(dof);
					} else {
						faceFeatureNode.setPosition( V2.add( initState.position, V2.scale( 1, transform.translate, [] ), [] ) );
						// TODO: will initState ever have a scale != 1 that we have to take into account?
						faceFeatureNode.setScale( transform.scale );
						faceFeatureNode.setAngle(initState.angle + transform.angle);
					}
				}
			}
		}
	}	

	// returns mouth shape index or -1 if already set by LipSync behavior
	function chooseMouthShape (args, head14) {
		// if no valid viseme, return geometry-based mouth shape ...
		var sharedData = args.stageLayer.getSharedFrameData("Adobe.LipSync"),
			lipSyncVisemeType = (sharedData) ? sharedData.visemeType : undefined;

		// if no valid lip sync viseme, just return mouth shape index
		if (lipSyncVisemeType === undefined) {
			return Math.max(head14["Head/MouthShape"], 0);
		}
		// if lip sync detects a short silence, return neutral index (0)
		else if (lipSyncVisemeType === "shortSilence") {
			return 0;
		}
		// otherwise, let lip sync set the viseme
		else {
			return -1;
		}
	}
	
	
	function makeValidIdFromLabel (str) {
		return str.replace(/[^\w]/g, "_");
	}
	
	function makeHandleIdFromLabel (str) {
		return "H_" + makeValidIdFromLabel(str);
	}
	
	function makeLayerIdFromLabel (str) {
		return "L_" + makeValidIdFromLabel(str);
	}
	
	function getLastComponentOfFeatureLabel (label) {
		return label.split("/").pop();	// for handle params, we only use the last component of the path
	}
	
	function addHiddenLayerParam (aParams, label, tooltip) {
		aParams.push({id:makeLayerIdFromLabel(label), type:"layer", uiName:label, dephault:{match:"//"+label}, maxCount:1,
					 	uiToolTip:tooltip, hidden:true});
	}
	
	function defineHandleParams () {
		var aParams = [];
 
		for (var label in faceFeatureLabels) {
			if (faceFeatureLabels.hasOwnProperty(label)) {
				var last = getLastComponentOfFeatureLabel(label), bHidden = false; // (label !== "Head");	// previously, we only showed Head

				aParams.push({id:makeHandleIdFromLabel(last), type:"handle", uiName:last+" Handle", dephault:{match:"//"+last}, maxCount:1, hidden:bHidden});
				// TODO: allow Iris/Pupil synomym -- needs support for | or "or" in xpath match
			}
		}

		// plus extra layer params
		addHiddenLayerParam(aParams, "Left Eyeball", "Sets the range of the Left Pupil; if missing, Left Eye is used instead");
		addHiddenLayerParam(aParams, "Right Eyeball", "Sets the range of the Right Pupil; if missing, Right Eye is used instead");
		addHiddenLayerParam(aParams, "Left Eye", "Sets the range of the Left Pupil if Left Eyeball not found");
		addHiddenLayerParam(aParams, "Right Eye", "Sets the range of the Right Pupil if Right Eyeball not found");
		
		// and layer params for the pupils
		addHiddenLayerParam(aParams, "Left Pupil", "Used to compute range of the Left Pupil");
		addHiddenLayerParam(aParams, "Right Pupil", "Used to compute range of the Right Pupil");

		// and layer params for the eyelids (TODO: consolidate to only have layer params, not handles?)
		addHiddenLayerParam(aParams, "Left Eyelid Top", "Sets the vertical range of the Left Eyelid; if missing, Left Eyelid Top _handle_ is used instead");
		addHiddenLayerParam(aParams, "Left Eyelid Bottom", "Sets the vertical range of the Left Eyelid; if missing, Left Eyelid Bottom _handle_ is used instead");
		addHiddenLayerParam(aParams, "Right Eyelid Top", "Sets the vertical range of the Right Eyelid; if missing, Right Eyelid Top _handle_ is used instead");
		addHiddenLayerParam(aParams, "Right Eyelid Bottom", "Sets the vertical range of the Right Eyelid; if missing, Right Eyelid Bottom _handle_ is used instead");
		
		return aParams;
	}


	function defineMouthLayerParams (bSorted) {
		var aParams = [];
		
		mouthShapeLabels.forEach(function (label) {
			// TODO: need to localize the uiName for each mouth shape
			aParams.push({id:makeLayerIdFromLabel(label), type:"layer", uiName:label, dephault:{match:"//"+label}, maxCount:1});
		});
		
		if (bSorted) { // WARNING: repeated in Adobe.LipSync.js
			aParams.sort(function (a, b)
							{ if (a.uiName < b.uiName) return -1;
								if (a.uiName > b.uiName) return 1;
								return 0; });
		}
		
		return aParams;
	}


	// returns array of all the mouth layer params that matched something
	function getMouthLayers (args) {
		var aMouthLayers = [], aMouthParams = defineMouthLayerParams();
		
		aMouthParams.forEach(function (param) {
			var aLayers = args.getParam(param.id),
				layer = aLayers[0];
			
			if (layer) {
				aMouthLayers.push(layer);
			}
		});

		return aMouthLayers;
	}

	function chooseMouthReplacements (args, allMouths, head14, inoutTransforms, outNodesToShow, outNodesToHide) {
		var mouthIndexToShow, validMouth = null;

		// returns actual mouth node & index for requested mouth index, which may not exist (uses fallbacks in mouthShapeCascades)
		function getClosestValidMouthContainerRoot(inMouthIndex) {

			var mouthIndicesToTry = [], i, mouthIndex, mouthShapeLabel,
				aLayers, layer;

			if (mouthShapeCascades.hasOwnProperty(inMouthIndex)) {
				mouthIndicesToTry = mouthShapeCascades[inMouthIndex];
			}
			else {
				mouthIndicesToTry = [inMouthIndex, 0];
			}

			for (i = 0; i < mouthIndicesToTry.length; i += 1) {
				mouthIndex = mouthIndicesToTry[i];
				if (mouthIndex >= 0 && mouthIndex < mouthShapeLabels.length) {
					mouthShapeLabel = mouthShapeLabels[mouthIndex];
					if (!mouthShapeLabel) {
						// this shouldn't happen, so adding error message if it does to help diagnose
						console.log("invalid mouth index: " + mouthIndex);
					} else {
						aLayers = args.getParam(makeLayerIdFromLabel(mouthShapeLabel));
						layer = aLayers[0];
						
						if (layer) {
							return {layer:layer, index:mouthIndex};
						}
					}
				}
			}

			return {layer:null, index:-1};
		}

		if (allMouths.length > 0) {

			// get mouth to show
			mouthIndexToShow = chooseMouthShape(args, head14);

			// get valid (found) mouth that is closest to specified mouth to show
			if (mouthIndexToShow >= 0) {
				validMouth = getClosestValidMouthContainerRoot(mouthIndexToShow);

				// either show the new mouth replacement....
				if (validMouth.layer) {
					outNodesToShow.push({"node" : validMouth.layer, "enabled" : true});
				} else {
					// alternatively, we could output error msg saying that we can't find a Neutral mouth
					//console.logToUser("choseMouthReplacements(): no Neutral mouth found");
				}
			} else {
				// lip sync behavior already set a mouth, just leave our ones hidden
				
				// don't scale visemes -- it looks strange; note that this trumps "mouthFactor" param
				inoutTransforms["Head/Mouth"] = getIdentityTransform();
				// NOTE: need to figure out how to do the above in other replacer behaviors (e.g., key replacer)
			}

			// hide any mouths that we are not trying to show
			allMouths.forEach(function (c) { 
				if (!validMouth || c !== validMouth.layer) {
					outNodesToHide.push({"node" : c, "enabled" : true});
				}
			});
		}

		// otherwise nothing to choose
	}

	function chooseBlinkReplacements(blinkLayer, siblingLayers, eyeOpenness, threshold, eyeFactor, outNodesToShow, outNodesToHide) {
		var showBlink = true;
		
		if (blinkLayer) {
			if ((eyeOpenness > threshold) || (eyeFactor === 0))  { // the eyelid measurment in head14 never seems to go below about .3
				showBlink = false;
			}

			siblingLayers.forEach(function (s) {
				if (showBlink) {
					outNodesToHide.push({"node" : s, "enabled" : true});
				} else {						
					outNodesToShow.push({"node" : s, "enabled" : false});
				}
			});

			if (showBlink) {
				outNodesToShow.push({"node" : blinkLayer, "enabled" : true});
			} else {
				outNodesToHide.push({"node" : blinkLayer, "enabled" : false});
			}
		}
	}

	function getHandle (args, label) {
		return args.getParam(makeHandleIdFromLabel(label))[0]; // only works for a single handle for each label right now
	}

	function computeInitTransforms (self, args) {
		var faceFeatureLabel, handleParam, featureLabels = self.featureLabels;

		self.puppetInitTransforms = {};

		for (faceFeatureLabel in featureLabels) {
			if (featureLabels.hasOwnProperty(faceFeatureLabel)) {
				handleParam = getHandle(args, getLastComponentOfFeatureLabel(faceFeatureLabel));
				if (handleParam) {
					// WL: we are explicitly recording whether a face feature label 
					// is present ... as an alternative, we could also just check 
					// if puppetInitTransforms has an entry for the label.
					featureLabels[faceFeatureLabel] = true;

					self.puppetInitTransforms[faceFeatureLabel] = {
						"position": handleParam.getPosition(),
						"scale"   : handleParam.getScale(),
						"angle"   : handleParam.getAngle()
					};
				}
			}
		}
	}
	
	function getLayer (args, label) {
		var match = args.getParam(makeLayerIdFromLabel(label))[0];
		
		if (!match) {
			match = null;					// the old code returned null
		}
		
		return match;
	}

	function computeEyeGazeRanges (self, args) {
		var puppetMeasurements = self.puppetMeasurements, 
			leftEyeGazeRangeX = 0, leftEyeGazeRangeY = 0, rightEyeGazeRangeX = 0, rightEyeGazeRangeY = 0,
			leftEyeWidth = getPuppetMeasurement(self, "leftEyeWidth", args),
			leftEyeHeight = getPuppetMeasurement(self, "leftEyeHeight", args),
			rightEyeWidth = getPuppetMeasurement(self, "rightEyeWidth", args),
			rightEyeHeight = getPuppetMeasurement(self, "rightEyeHeight", args),
			leftPupilWidth = getPuppetMeasurement(self, "leftPupilWidth", args),
			leftPupilHeight = getPuppetMeasurement(self, "leftPupilHeight", args),
			rightPupilWidth = getPuppetMeasurement(self, "rightPupilWidth", args),
			rightPupilHeight = getPuppetMeasurement(self, "rightPupilHeight", args);

		if (leftEyeWidth) {
			leftEyeGazeRangeX = (leftPupilWidth) ? 0.5 * (leftEyeWidth - leftPupilWidth) : 0.5 * leftEyeWidth;
		}
		if (leftEyeHeight) {
			leftEyeGazeRangeY = (leftPupilHeight) ? 0.5 * (leftEyeHeight - leftPupilHeight) : 0.5 * leftEyeHeight;
		}
		if (rightEyeWidth) {
			rightEyeGazeRangeX = (rightPupilWidth) ? 0.5 * (rightEyeWidth - rightPupilWidth) : 0.5 * rightEyeWidth;
		}
		if (rightEyeHeight) {
			rightEyeGazeRangeY = (rightPupilHeight) ? 0.5 * (rightEyeHeight - rightPupilHeight) : 0.5 * rightEyeHeight;
		}

		puppetMeasurements["leftEyeGazeRangeX"] = leftEyeGazeRangeX;
		puppetMeasurements["leftEyeGazeRangeY"] = leftEyeGazeRangeY;
		puppetMeasurements["rightEyeGazeRangeX"] = rightEyeGazeRangeX;
		puppetMeasurements["rightEyeGazeRangeY"] = rightEyeGazeRangeY;
	}
	
	function computePuppetMeasurements (self, args) {
		// "pos" measures the x or y dist between two rest positions
		// "bbox" measures the x or y dimension of a rest bbox
		// "scaledCopy" copies another named measurement scaled by factor
		var measurementsList = [
				["headWidth", "bbox", self.headPuppet, "X"],
				["headHeight", "bbox", self.headPuppet, "Y"],
			
				["leftEyeEyebrowDist", 	"pos", getHandle(args, "Left Eye"), 	getHandle(args, "Left Eyebrow"), 	"Y"],
				["leftEyeEyebrowDist", 	"scaledCopy", 0.15, "headHeight"], // fallback if both "Left Eye" and "Left Eyebrow" handles don't exist
																			// maybe remove this -- is it useful to have eyebrows without eyes?

				["rightEyeEyebrowDist", "pos", getHandle(args, "Right Eye"), 	getHandle(args, "Right Eyebrow"), 	"Y"],
				["rightEyeEyebrowDist", "scaledCopy", 0.15, "headHeight"], // fallback if both "Right Eye" and "Right Eyebrow" handles don't exist
			
				["interocularDist",		"pos", getHandle(args, "Left Eye"),		getHandle(args, "Right Eye"), 		"X"],
				["interocularDist",		"scaledCopy", 0.25, "headWidth"],
			
				["noseDepth", 		"scaledCopy", 0.15, "interocularDist"],
				["eyeDepth",		"scaledCopy", 0.09, "interocularDist"],
			
				// controls distance that the left eyelid top moves down, and bottom moves up; primary: space between top & bottom bounds
				["leftEyelidDist", 	"bboxDiff", getLayer(args, "Left Eyelid Top"), getLayer(args, "Left Eyelid Bottom"), "Y"],
				// 	fallback: distance between top & bottom handles
				["leftEyelidDist", 	"pos", getHandle(args, "Left Eyelid Top"), getHandle(args, "Left Eyelid Bottom"), "Y"],
			
				["rightEyelidDist", "bboxDiff", getLayer(args, "Right Eyelid Top"), getLayer(args, "Right Eyelid Bottom"), "Y"],
				["rightEyelidDist", "pos", getHandle(args, "Right Eyelid Top"), getHandle(args, "Right Eyelid Bottom"), "Y"],
			
				["leftEyeWidth", 	"bbox", getLayer(args, "Left Eyeball"), "X"],
				["leftEyeWidth", 	"bbox", getLayer(args, "Left Eye"), "X"],
			
				["rightEyeWidth", 	"bbox", getLayer(args, "Right Eyeball"), "X"],
				["rightEyeWidth", 	"bbox", getLayer(args, "Right Eye"), "X"],
			
				["leftEyeHeight", 	"bbox", getLayer(args, "Left Eyeball"), "Y"],
				["leftEyeHeight", 	"bbox", getLayer(args, "Left Eye"), "Y"],
			
				["rightEyeHeight", 	"bbox", getLayer(args, "Right Eyeball"), "Y"],
				["rightEyeHeight", 	"bbox", getLayer(args, "Right Eye"), "Y"],
			
				["leftPupilWidth",	"bbox", getLayer(args, "Left Pupil"), "X"],
				["leftPupilHeight", "bbox", getLayer(args, "Left Pupil"), "Y"],

				["rightPupilWidth",	"bbox", getLayer(args, "Right Pupil"), "X"],
				["rightPupilHeight", "bbox", getLayer(args, "Right Pupil"), "Y"],				

				["mouthWidth", "bbox", getLayer(args, "Neutral"), "X"], // TODO: change to mouth puppet?			
				["mouthHeight", "bbox", getLayer(args, "Neutral"), "Y"] 
			],

			puppetMeasurements = self.puppetMeasurements,
			i, j, value, m, bbox1, bbox2;

		for (i = 0; i < measurementsList.length; i += 1) {
			m = measurementsList[i];

			if (!(puppetMeasurements.hasOwnProperty(m[0]) && puppetMeasurements[m[0]])) {
				value = null;

				if (m[1] === "pos" && m[2] && m[3] && m[4]) {
					// rootMatrix is a 3x3 transformation (in homogeneous coords)
					// rootMatrix[6] = x translation, rootMatrix[7] = y translation 
					if (m[4] === "X") { 
						j = 6; 
					}
					else { 
						j = 7; 
					}
					// TODO: may not work as intended when Puppets are scaled: ten times smaller or ten times larger
					value = Math.abs(args.getHandleMatrixRelativeToScene(m[2])[j]-args.getHandleMatrixRelativeToScene(m[3])[j]);
				} 
				else if (m[1] === "bbox" && m[2] && m[3]) {
					bbox1 = m[2].getBounds();

					if (m[3] === "X") { 
						value = bbox1[2]; 
					}
					else { 
						value = bbox1[3]; 
					}
				}
				else if (m[1] === "bboxDiff" && m[2] && m[3] && m[4]) {

					bbox1 = m[2].getBounds();
					bbox2 = m[3].getBounds();

					if (m[4] === "X") {
						value = bbox2[0] - (bbox1[0] + bbox1[2]);
					}
					else {
						value = bbox2[1] - (bbox1[1] + bbox1[3]);
					}
					if (value <= 0) {
						value = null;	// will fall back on "pos" instead
					}
				}
				else if (m[1] === "scaledCopy" && m[2] && m[3]) {
					value = m[2] * puppetMeasurements[m[3]];
				}

				puppetMeasurements[m[0]] = value;

				// DEBUG
				//console.log(m[0] + "_" + m[1] + " = " + puppetMeasurements[m[0]]);
			}
		}

		// add additional compound measurements
		computeEyeGazeRanges(self, args);

	}

	function computeReplacementMap(self, args) {
		self.replacementMap = {};

		function getBlinkSiblingLayersToReplace(blinkLayer) {
			var siblingLayersToReplace = [], parentLayer = blinkLayer.getParentLayer();

			parentLayer.forEachDirectChildLayer(function (c) {
				// only add siblings that are not muted to replacement set
				if (c !== blinkLayer && c.getVisible()) { 
					siblingLayersToReplace.push(c);
				}
			});

			return siblingLayersToReplace;
		}

		// add replacement layers for left/right blinks
		self.leftBlinkLayers.forEach(function (lay) {
			self.replacementMap[lay.getId()] = getBlinkSiblingLayersToReplace(lay);
		});

		self.rightBlinkLayers.forEach(function (lay) {
			self.replacementMap[lay.getId()] = getBlinkSiblingLayersToReplace(lay);
		});

		// add replacement mouth layers
		self.replacementMap["mouths"] = getMouthLayers(args);

		// DEBUG
		//console.logToUser("==== FaceTracker replacement map ====");
		//for (var key in self.replacementMap) {
		//	console.logToUser("\t" + key + ":");
		//	self.replacementMap[key].forEach(function (r) {
		//		console.logToUser("\t\t" + r.getName() + "--" + r.getId());
		//	});
		//}
	}

	/*function printPuppetContainerHandleHierarchies(inPuppet) {

		inPuppet.breadthFirstEach(function (p) {
			var cTree, hTree;
			console.log("p = " + p.getName());
			cTree = p.getContainerTree();

			if (cTree) {
				cTree.breadthFirstEach(function (c) {
					console.log("\tc = " + c.getName());
				});
			}

			hTree = p.getHandleTreeRoot();
			if (hTree) {
				console.log("\thTree:");				
				hTree.breadthFirstEach(function (h) {
					console.log("\t\th = " + h.getName());
				});
			}
		});
	}*/

	function chooseFaceTrackerReplacements(self, args, head14, inoutTransforms) {
		var eyeFactor = args.getParam("eyeFactor"),
			eyeClosedThresh = 0.45,
			nodesToShow = [], nodesToHide = [], 
			replacementMap = self.replacementMap;
		
		chooseMouthReplacements(args, replacementMap["mouths"], head14, inoutTransforms, nodesToShow, nodesToHide);
		
		var leftEyelid = head14["Head/LeftEyelid"],
			rightEyelid = head14["Head/RightEyelid"];
		
		if (args.getParam("blinkOnly")) {
			leftEyelid = rightEyelid = (leftEyelid + rightEyelid) / 2;
		}

		self.leftBlinkLayers.forEach(function (lay) {
			chooseBlinkReplacements(lay, replacementMap[lay.getId()], leftEyelid, eyeClosedThresh, eyeFactor, nodesToShow, nodesToHide);
		});

		self.rightBlinkLayers.forEach(function (lay) {
			chooseBlinkReplacements(lay, replacementMap[lay.getId()], rightEyelid, eyeClosedThresh, eyeFactor, nodesToShow, nodesToHide);
		});

		return {"nodesToShow" : nodesToShow, "nodesToHide" : nodesToHide};
	}

	function animateWithFaceTracker(self, args) {
		var transforms, head14 = getHead14(args), replNodes;

		transforms = computePuppetTransforms(self, args, head14);

		replNodes = chooseFaceTrackerReplacements(self, args, head14, transforms);

		ReplacerUtilities.applyReplacements(self, args, kReplacerPriority, kReplacerDoesOverride, replNodes.nodesToShow, replNodes.nodesToHide);

		applyPuppetTransforms(self, args, transforms);
	}
	
	return {
		about:			"Face Tracker, (c) 2015.",
		description:	"$$$/Animal/Behavior/FaceTracker/Desc=Controls head, eyes, eyebrows, nose, and mouth via your webcam",
		uiName:			"$$$/Animal/Behavior/FaceTracker/UIName=Face",
		defaultArmedForRecordOn: true,
	
		defineParams: function () { // free function, called once ever; returns parameter definition (hierarchical) array
		  return [
			{id:"cameraInput",		type:"eventGraphInput", uiName:"Camera Input", inputKeysArray:["Head/"], uiToolTip:"Analyzed face data from the camera",
				defaultArmedForRecordOn: true},
			{id:"mouseEyeGaze",		type:"eventGraphInput", uiName:"Eye Gaze via Mouse Input", inputKeysArray:["Mouse/"],
			 	uiToolTip:"$$$/animal/behavior/face/param/EyeGazeViaMouse/tooltip=Control eye gaze direction with your mouse"},
			{id:"headPosFactor",	type:"slider", uiName:"Head Position Strength",	uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how much the head moves when you move your head"},
			{id:"headScaleFactor",	type:"slider", uiName:"Head Scale Strength",	uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how much the head scales when you move your head closer or farther from the camera"},
			{id:"headRotFactor",	type:"slider", uiName:"Head Tilt Strength",		uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how much the head rotates when you tilt the top of your head to the right and left"},
			{id:"eyebrowFactor",	type:"slider", uiName:"Eyebrow Strength",		uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how much the eyebrows move when you raise and lower your eyebrows"},
			{id:"eyeFactor",		type:"slider", uiName:"Eyelid Strength",		uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how far the eyelids move (if present) or the eyes scale (if not) when you blink"},
			{id:"eyeGazeFactor",	type:"slider", uiName:"Eye Gaze Strength",		uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how far the pupils move when you look around"},
			{id:"mouseEyeGazeFactor",	type:"slider", uiName:"Eye Gaze Mouse Strength",	uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize how far the pupils move when you move mouse around"},
			{id:"mouthFactor",		type:"slider", uiName:"Mouth Strength",			uiUnits:"%", min:0, max:500, precision:0, dephault:0,
				uiToolTip:"Exaggerate or minimize how much the mouth scales and moves to correspond with your mouth movements"},
			{id:"parallaxFactor",	type:"slider", uiName:"Parallax Strength",		uiUnits:"%", min:0, max:500, precision:0, dephault:100,
				uiToolTip:"Exaggerate or minimize the eyes and nose movement when you rotate your head left and right"},

			{id:"handlesGroup", type:"group", uiName: "Handles", groupChildren: defineHandleParams()},
			  
			{id:"replacementsGroup", type:"group", uiName:"Replacements", groupChildren: [
				{id:"leftBlinkLayers",	type:"layer",	uiName:"Left Blink",	dephault:{match:"//Left Blink"},	maxCount:1},
				{id:"rightBlinkLayers",	type:"layer",	uiName:"Right Blink",	dephault:{match:"//Right Blink"},	maxCount:1},
				{id:"blinkOnly",		type:"checkbox", uiName:"Blink Eyes Together",	dephault:true},
				{id:"mouthGroup",		type:"group",	uiName:"Mouths",		groupChildren: defineMouthLayerParams("sorted")}
			]}
		  ];
		},
		
		onCreateBackStageBehavior: function (self) {
			self.puppetInitTransforms = {};

			return { order: 1.0, importance : 0.0 }; // must come after LipSync
		},
		
		onCreateStageBehavior: function (self, args) {
			var headHandle, newHead = null;
			args.getParam = args.getStaticParam;	// @@@HACK! until we rename this function
			
			self.puppetMeasurements = {};
			
			self.featureLabels = utils.copy(faceFeatureLabels);

			headHandle = args.getParam(makeHandleIdFromLabel("Head"))[0];
			if (headHandle) {
				newHead = headHandle.getPuppet(); // get puppet that has the Head handle -- avoids the need for an extra Head layer param
			}
			
			self.headPuppet = newHead;	// used for fallback eyebrow measurement, might be null

			// find blink replacements
			self.leftBlinkLayers = args.getStaticParam("leftBlinkLayers");
			self.rightBlinkLayers = args.getStaticParam("rightBlinkLayers");
			
			computeInitTransforms(self, args);
			computePuppetMeasurements(self, args);
			computeReplacementMap(self, args);
			ReplacerUtilities.updateReplacerInfo(args);

			// debug -- show puppet/container hierarchy
			// printPuppetContainerHandleHierarchies(stagePuppet);
		},
		
		onAnimate: function (self, args) { // method on behavior that is attached to a puppet, only onstage
			animateWithFaceTracker(self, args);
		}
		
	}; // end of object being returned
});
