// Cycle Layers behavior

define( ["lib/Zoot"],
  function (Z) {
	"use strict";
	
	var kStart_Immediately = 0,
		kStart_WaitForTriggerC = 1,
		kStart_WaitForTriggerCustom = 2,
		
		kCycle_Continuously = 0,
		kCycle_Once = 1,
		kCycle_OnceAndHoldOnEnd = 2;
		
	
	function resetCycle (self) {
		self.activeLayerIndex = 0;
		self.idleCount = 0;
	}
	
	// returns triggerKey string or false
	function getKeyFromLayer (nameRegExp, layer) {
		var name, customKey;
		
		name = layer.getName();
		if (nameRegExp.test(name)) {
			customKey = nameRegExp.exec(name)[1];
			if (customKey.length === 1) {
				return customKey.toUpperCase();
			}
		}
		
		return false;
	}
	
	function getTriggerKey (args) {
		var triggerKey = "C", nameRegExp, customKey; // default to "C" for kStart_WaitForTriggerC & if custom can't be found
		
		if (args.getParam("startWhen") === kStart_WaitForTriggerCustom) {
			nameRegExp = new RegExp(/\((\S)(\S?)\)/);	// COPIED from KeyReplacer, but temporary until we have a real UI
			
			customKey = getKeyFromLayer(nameRegExp, args.stageLayer);
			
			if (customKey) {
				triggerKey = customKey;
			} else {
				args.stageLayer.forEachDirectChildLayer(function (lay) {
					customKey = getKeyFromLayer(nameRegExp, lay);

					if (customKey) {
						triggerKey = customKey;
						return false; // stop iteration
					}
				});
			}
		}
		
		return triggerKey;
	}
	
	function immediateAndContinuous (args) {
		return (args.getParam("startWhen") === kStart_Immediately) &&
				(args.getParam("cycle") === kCycle_Continuously);
	}
	

	return {
		about: 			"Cycle Layers, (c) 2015.",
		description: 	"$$$/Animal/Behavior/CycleLayers/Desc=Steps through all layers in a puppet and shows each, one-by-one",
		uiName:  		"$$$/Animal/Behavior/CycleLayers/UIName=Cycle Layers",
		defaultArmedForRecordOn: true,		// so the key trigger works without fiddling around

		defineParams : function () {
			return [
				{ id: "frameCount",	type: "slider", uiName: "Cycle Every", uiUnits: "frames", min: 0, max:1000, dephault:2,
					uiToolTip:"$$$/animal/behavior/cyclelayers/framecount/tooltip=Affects cycling speed; larger values cycle more slowly"},
				{ id: "startWhen",	type: "enum", uiName: "Start",  dephault: kStart_Immediately,
					items: [	{id:kStart_Immediately,				uiName: "Immediately"},
								{id:kStart_WaitForTriggerC,			uiName: "Wait for 'C'"},
								{id:kStart_WaitForTriggerCustom,	uiName: "Wait for (<key>) in Layer Name"}, // hack until we have a key trigger UI
						   ],
					uiToolTip:"$$$/animal/behavior/cyclelayers/startwhen/tooltip=Choose between starting immediately or waiting for a specific keypress"},
				{ id: "cycle",		type: "enum", uiName: "Cycle",  dephault: kCycle_Continuously,
					items: [	{id:kCycle_Continuously,			uiName:"Continously"},
								{id:kCycle_Once,					uiName:"Once"},
								{id:kCycle_OnceAndHoldOnEnd,		uiName:"Once, Hold on Last Frame"}
						   ],
				},
				{ id:"keyboardInput", type:"eventGraphInput", uiName:"Keyboard Input", inputKeysArray:["Keyboard/"], uiToolTip:"Keyboard input used for triggering",
				 	defaultArmedForRecordOn: true}
			];
		},

		onCreateBackStageBehavior : function (/*self*/) {
			return { order: 0.0, importance : 0.0 };
		},

		onCreateStageBehavior : function (self, args) {
			resetCycle(self);
			
			// two problems with kStart_Immediately right now: 1) getStaticParam is currently
			//	not reading track item behavior params, only puppet params; 2) switching
			//	this popup after starting the stage is ignored -- but can't just slam bActive
			//	to true in the main loop if this mode is on, otherwise cycle Once wouldn't work.
			//	perhaps check for Immediate & Continuous, and if so, slam bActive to true in main loop
			self.bActive = (args.getStaticParam("startWhen") === kStart_Immediately);
			self.bTriggerPreviouslyDown = false;
		},

		onAnimate : function (self, args) {
			var i, layerToShow, bTrigger, triggerKey = getTriggerKey(args), cycle = args.getParam("cycle");

			bTrigger = args.getParamEventValue("keyboardInput", Z.keyCodes.getKeyGraphId(triggerKey));
			
			//console.log(triggerKey + " / " + bTrigger);
			
			if (self.bTriggerPreviouslyDown) {
				if (!bTrigger) { // trigger just released
					self.bTriggerPreviouslyDown = false;
				}
			} else {
				if (bTrigger) { // trigger just pulled
					args.setInputParamUsedDuringRecording("keyboardInput");
					self.bTriggerPreviouslyDown = true;
					resetCycle(self);
					self.bActive = true;
				}
			}
			
			if (self.bActive || immediateAndContinuous(args)) {
				layerToShow = self.activeLayerIndex;
				
				//console.log("layerToShow = " + layerToShow);

				i = 0;
				args.stageLayer.forEachDirectChildLayer(function (lay) {
					if (i++ === layerToShow) {
						lay.setOpacity(1);
	//					lay.setVisible(true);	// WHY DOESN'T THIS WORK? -- Mediator is blasting TODO: figure out real plan
					} else {
						lay.setOpacity(0);
	//					lay.setVisible(false);	// WHY DOESN'T THIS WORK? -- Mediator is blasting
					}
				});

				if (++self.idleCount >= args.getParam("frameCount")) {
					self.idleCount = 0;
					++self.activeLayerIndex;
				}

				if (self.activeLayerIndex === i && cycle === kCycle_OnceAndHoldOnEnd) {
					--self.activeLayerIndex;
				}
				
				if (self.activeLayerIndex >= i) {
					resetCycle(self); // wrap back to first layer
					if (cycle === kCycle_Once) {
						self.bActive = false;
					}
				}
			} else {
				args.stageLayer.forEachDirectChildLayer(function (lay) {
					lay.setOpacity(0);
				});
			}
		}

	}; // end of object being returned
});
