/**************************************************************************************************
*
* ADOBE SYSTEMS INCORPORATED
* Copyright 2013 Adobe Systems Incorporated
* All Rights Reserved.
*
* NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the
* terms of the Adobe license agreement accompanying it.  If you have received this file from a
* source other than Adobe, then your use, modification, or distribution of it requires the prior
* written permission of Adobe.
*
**************************************************************************************************/

/** NGLInterface - v9.2.0 */


/**
 * @class NGLInterface
 * This is the entry point to the Adobe Identity Management Service (IMS) toolkit library.
 * Instantiate this class, and use the object to:
 * <ul>
 * <li>Retrieve a subscription access token for a particular service.</li>
 * <li>Retrieve a list of users who are registered on this device.</li>
 * <li>Invoke Adobe Application Manager (AAM) to interact with user if needed.</li>
 * </ul>
 * Most requests include a session reference, which is returned from a connection request,
 * and an NGL client ID, which uniquely identifies your service and service context. When
 * developing a CS Extension, you must register with the NGL team to receive your client ID.
 *
 * @return The \c NGLInterface object for an IMS session.
 */
function NGLInterface()
{
}

/**
 * Registers a  handler for a CS event. The handler is a callback function that is
 * passed a single parameter, the \c #CSEvent object containing detailed information
 * about the event.
 *
 * You must register callback functions to handle the results of asynchronous calls
 * to the \c NGLInterface methods.
 *
 * @param type      The event-type  constant.
 * @param listener  The JavaScript handler function for the event.
 */
NGLInterface.prototype.addEventListener = function(type, listener)
{
    window.__adobe_cep__.addEventListener(type, listener);
};

/**
 * Removes the handler associated with a CS event.
 *
 * @param type      The event-type  constant.
 * @param listener  The JavaScript handler function that was registered for the event.
 */
NGLInterface.prototype.removeEventListener = function(type, listener)
{
    window.__adobe_cep__.removeEventListener(type, listener);
};


/**
 * Requests an access token from IMS for a given user and service.
 *
 * This function is asynchronous. To handle the result, register a callback for the event
 * NGLInterface.events.imsFetchAccessTokenWithStatus
 * For event NGLInterface.events.imsFetchAccessTokenWithStatus, the event data is a JSON string
 * with format {"status":"0","details":{"toua":"...",...,"emailVerified":"false"}}.
 *
 * @see addEventListener()
 * @param clientID       The IMS client ID for your extension, assigned by the IMS team on registration.
 *                       This is the service-code value in the Renga Service Account Object for your client.
 * @param clientSecret   The client secret associated with the provided client ID
 * @param scope          (optional) A comma delimited list of services for which the refresh and access tokens are requested.
                         By default, this is 'openid,AdobeID'. Scope strings are case sensitive.
                         If the cached version of the refresh token does not match the
                         requested service scopes, a new refresh token is fetched.
                         To request your service's own SAO, add your service to the list;
                         for example,  'openid,AdobeID,browserlab'.
 *
 * @return               A boolean status. Returning true means only the call to imsFetchAccessToken function itself is successful.
 *						 Returning false means that failed to fetch access token.
 */
NGLInterface.prototype.imsFetchAccessToken = function(clientId, clientSecret, scope)
{
    return window.__adobe_cep__.nglImsFetchAccessToken(clientId, clientSecret, scope);
};

/**
 * Requests the IMSProfile from NGL for a given access token.
 *
 * This function is asynchronous. To handle the result, register a callback for the event
 * NGLInterface.events.imsFetchProfile
 *
 * @see addEventListener() 
 * @param accessToken    The access token issued by IMS.
 * @return               A boolean status. Returning true means only the call to imsFetchProfile function itself is successful.
 *						 Returning false means that failed to fetch access token.
 */
NGLInterface.prototype.imsFetchProfile = function(accessToken)
{
    return window.__adobe_cep__.nglImsFetchProfile(accessToken);
};

/**
 * Requests the continue token for a particular client ID with the provided access token.
 * If the redirectUri parameter is provided then a JumpURL will be created by appending the token to the URL.
 * This enables the support of SSO workflows from desktop to web.
 *
 * This function is asynchronous. To handle the result, register a callback for the event NGLInterface.events.imsFetchContinueToken
 * @see addEventListener()
 *
 * Since 5.2.0
 *
 * @param bearerToken       The access token issued by IMS.
 * @param targetClientId    The client ID needed for the destination.
 * @param redirectUri  (optional) The URL to jump to. It should be URL encoded.
 * @param scope        (optional) A comma delimited list of services for which the refresh token was requested.
 * @param responseType (optional) The IMS integration type the target has.
 * 
 * @return A boolean status. Returning true means only the call to imsFetchContinueToken function itself is successful.
 *							Returning false means that failed to fetch continue token.
 */
NGLInterface.prototype.imsFetchContinueToken = function(bearerToken, targetClientId, redirectUri, scope, responseType)
{
    if ((redirectUri === undefined) || (redirectUri === null))
    {
        redirectUri = "";
    }
    if ((scope === undefined) || (scope === null))
    {
        scope = "";
    }
    if ((responseType === undefined) || (responseType === null))
    {
        responseType = "";
    }
    

    var params = {};
    params.bearerToken = bearerToken;
    params.targetClientId = targetClientId;
    params.redirectUri = redirectUri;
    params.scope = scope;
    params.responseType = responseType;
    return JSON.parse(window.__adobe_cep__.invokeSync("nglImsFetchContinueToken", JSON.stringify(params))).result;
};

/**
 * Retrieves the IMS user id that is currently logged in.
 *
 * Since 4.1.0
 *
 * @return The IMS user id for the currently logged in user.
 */
NGLInterface.prototype.imsGetCurrentUserId = function()
{
    var userId = window.__adobe_cep__.getCurrentImsUserId();
    return userId;
};

/**
 * Retrieves the IMS user id that is currently logged in. This function
 * also takes care of the workarounds specified for some of the products:
 * https://wiki.corp.adobe.com/display/csxs/CEP+5.0+HTML+Extension+Cookbook#CEP5.0HTMLExtensionCookbook-GetthecurrentIMSUserID
 * 
 * @param callback This function is asynchronous so provide a callback to handle the result.
 * 
 * Since 5.2.0
 *
 * @return The IMS user id for the currently logged in user.
 */
NGLInterface.prototype.imsGetCurrentUserIdHelper = function(callback)
{
    // different workflows for Ps, Ai and ID.
    var hostEnvironment = JSON.parse(window.__adobe_cep__.getHostEnvironment());
    var appName = hostEnvironment.appName;
    if (appName === "ILST") {
        window.__adobe_cep__.evalScript('app.userGUID', function(result){
            callback(result);
        });
    } else if (appName === "IDSN") {
        window.__adobe_cep__.evalScript('app.userGuid', function(result){
            callback(result);
        });
    } else if (appName === "PHSP" || appName === "PHXS") {
        window.__adobe_cep__.evalScript("var getUserIdPhotoshop = function() {var userId = '';try {var bhnc = app.charIDToTypeID('bhnc');var ref = new ActionReference();ref.putProperty(app.charIDToTypeID('Prpr'), bhnc);ref.putEnumerated(app.charIDToTypeID('capp'), app.charIDToTypeID('Ordn'), app.charIDToTypeID('Trgt'));var appDesc = app.executeActionGet(ref);if (appDesc.hasKey(bhnc)) {userId = appDesc.getString(bhnc);}} catch (e) {}return userId;};", function(result){
            window.__adobe_cep__.evalScript('getUserIdPhotoshop()', function(result2) {
                callback(result2);
            });
        });
    } else {
        var userId = window.__adobe_cep__.getCurrentImsUserId();
        callback(userId);
    }
};

/**
 * Launches the Adobe Application Manager (AAM) AIR application.
 * If an error occurs when fetching an access token, this allows the user to
 * log in with their Adobe ID credentials, or accept new terms of use.
 *
 * This function is asynchronous. To handle the result, register a callback for the event IMSInterface.events.imsAAMIMSStatus.
 * The returned event contains an \c #AAMStatus object.
 * @see addEventListener()
 *
 * @param clientID       The IMS client ID for your extension, assigned by the IMS team.
 *               Matches the <code>serviceCode</code> value in the Service Account Object (SAO).
 * @param clientSecret       The client secret associated with the provided client ID.
 * @param redirectUri        The redirect URI registered with IMS
 * @param userAccountGuid    The unique person GUID for the Renga account, as returned in the response
                to the <code>fetchAccounts()</code> method.
 * @param serviceAccountGuid (optional, not currently used) A unique identifier for a Service Account Object (SAO).
 * @param scope          (optional) A comma delimited list of services for which the refresh token was requested.
 *
 * @return A boolean status. Returning true means only the call to showAAM function itself is successful. 
 *							Returning false means that failed to show AAM.
 */
NGLInterface.prototype.showAAM = function(clientId, clientSecret, redirectUri, userAccountGuid, serviceAccountGuid, scope)
{
    return window.__adobe_cep__.showAAM(clientId, clientSecret, redirectUri, userAccountGuid, serviceAccountGuid, scope);
};


NGLInterface.events = {
    imsFetchAccessToken                : "com.adobe.csxs.events.internal.ims.FetchAccessTokenWithStatus",
    imsFetchContinueToken              : "com.adobe.csxs.events.internal.ims.FetchAccessTokenWithStatus",
    imsFetchProfile                    : "com.adobe.csxs.events.internal.ims.FetchAccessTokenWithStatus"
};

NGLInterface.status = {
    IMS_SUCCESS                                     : "0", /**< General Success status.*/
    IMS_ERROR_FAILURE                               : "1",/**< General failure status.*/
    IMS_ERROR_INVALID_ARGUMENTS                     : "2", /**< Error returned if invalid arguments are passed to an API.*/
    IMS_ERROR_CANCEL                                : "20", /**< Error returned if the Http request is cancelled.*/
    IMS_ERROR_TIMEOUT                               : "21",/**< Error returned if the Http request is timed out.*/
    IMS_ERROR_HTTPFAILURE                           : "22", /**< Error returned if http request fails.*/
    IMS_ERROR_SSLFAILURE                            : "23",/**< Error returned if http request fails, due to SSL certifiaction error.*/
    IMS_ERROR_AUTH_PROXY_REQUIRED                   : "24", /**< Error returned if http request is being made behind proxy.*/
    IMS_ERROR_AUTH_PROXY_FAILED                     : "25", /**< Error returned if http request failed due to wrong proxy credentials.*/
    IMS_ERROR_IN_ACCESS_IDP                         : "26", /**< Error returned if IMSLib fails in accessing IDP server. Probably server is down.*/
    IMS_ERROR_ANOTHER_REQUEST_IN_PROCESS            : "40", /**< Error returned if async call to IMSLib is made when one async http request is already in progress.*/
    IMS_ERROR_IN_READ_USER_DATA                     : "60", /**< Error returned if IMSLib fails in reading user specific data from the user machine.*/
    IMS_ERROR_IN_SAVE_USER_DATA                     : "61", /**< Error returned if IMSLib fails in saving user specific data on the user machine.*/
    IMS_ERROR_IN_REMOVE_USER_DATA                   : "62", /**< Error returned if IMSLib fails in removing user specific data from the user machine.*/
    IMS_ERROR_USER_DATA_NOT_PRESENT                 : "63", /**< Error returned if IMSLib detects that user data is not present on the user machine.*/
    IMS_ERROR_IN_READ_DEVICE_TOKEN                  : "64", /**< Error returned if IMSLib fails in reading device token from the user machine.*/
    IMS_ERROR_IN_SAVE_DEVICE_TOKEN                  : "65", /**< Error returned if IMSLib fails in saving device token on the user machine.*/
    IMS_ERROR_IN_REMOVE_DEVICE_TOKEN                : "66", /**< Error returned if IMSLib fails in removing device token from the user machine.*/
    IMS_ERROR_DEVICE_TOKEN_NOT_PRESENT              : "67", /**< Error returned if device token is not present on the user machine.*/
    IMS_ERROR_INVALID_DEVICE_TOKEN                  : "68", /**< Error returned if invalid device token is present on the user machine.*/
    IMS_ERROR_CLIENTID_NOT_PRESENT                  : "69", /**< Error returned, if client id information is missing on the user machine.*/
    IMS_ERROR_IN_FETCH_USER_ACCOUNTS                : "70", /**< Error returned, if IMSLib fails in fetching user accounts details from the user machine.*/
    IMS_ERROR_IN_SAVE_USER_FOR_CLIENTID             : "71", /**< Error returned, if IMSLib fails in saving user id for a particular client id.*/
    IMS_ERROR_DEVICE_ID_NOT_PRESENT                 : "72", /**< Error returned if device id is not present on the user machine.*/
    IMS_ERROR_DEFAULT_USER_FOR_CLIENTID_NOT_PRESENT : "73", /**< Error returned if there is no default user for client_id*/
    IMS_ERROR_NO_USER_RECORDS_PRESENT               : "74", /**< Error returned if there is no user records are present in opm.db*/
    IMS_ERROR_ACCESS_DENIED                         : "80", /**< Error returned if IMS server returns error code 'access_denied' while generating token.*/
    IMS_ERROR_INVALID_REQUEST                       : "81", /**< Error returned if IMS server returns error code 'invalid_request' while generating token.*/
    IMS_ERROR_INVALID_CLIENT                        : "82", /**< Error returned if IMS server returns error code 'invalid_client' while generating token.*/
    IMS_ERROR_INVALID_GRANT                         : "83", /**< Error returned if IMS server returns error code 'invalid_grant' while generating token.*/
    IMS_ERROR_UNAUTHORIZED_CLIENT                   : "84", /**< Error returned if IMS server returns error code 'unauthorized_client' while generating token.*/
    IMS_ERROR_UNSUPPORTED_RESPONSE_TYPE             : "85", /**< Error returned if IMS server returns error code 'unsupported_response_type' while generating token.*/
    IMS_ERROR_INVALID_SCOPE                         : "86", /**< Error returned if IMS server returns error code 'invalid_scope' while generating token.*/
    IMS_ERROR_UNSUPPORTED_GRANT_TYPE                : "87", /**< Error returned if IMS server returns error code 'unsupported_grant_type' while generating token.*/
    IMS_ERROR_BAD_REQUEST                           : "88", /**< Error returned if IMS server returns error code 'bad_request' while generating token.*/
    IMS_ERROR_INVALID_CREDENTIALS                   : "89", /**< Error returned if IMS server returns error code 'invalid_credentials' while generating token.*/
    IMS_ERROR_IN_GET_AUTH_DATA_FROM_IDP             : "100", /**< Error returned if IMSLib fails fetching authentication data from server.*/
    IMS_ERROR_IN_GET_DEVICE_TOKEN_FROM_IDP          : "101", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_IN_GET_REFRESH_TOKEN_FROM_IDP         : "102", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_IN_GET_ACCESS_TOKEN_FROM_IDP          : "103", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_IN_GET_PROFILE_DATA_FROM_IDP          : "104", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_TOU_CHANGED                           : "120", /**< Error returned if TOU has changed. */
    IMS_ERROR_IN_REVOKE_DEVICE_TOKEN                : "121", /**< Error returned if IMSLib fails in revoking device token. */
    IMS_ERROR_TOU_NOT_CURRENT                       : "122", /**< Terms of use acceptance is not current. */
    IMS_ERROR_EVS_INVALID                           : "123", /**< Email Verification Status does not meet required policy. */
    IMS_ERROR_ACCT_ACT_REQ                          : "124", /**< Some issue with the user account requires attention/action. i.e. COPPA or some other issue requiring user action, related to the user account rather than a subscription. */
    IMS_ERROR_ACCT_DISABLED                         : "125", /**< Account is currently disabled. */
    IMS_ERROR_SUBS_ACT_REQ                          : "126", /**< Some issue with the subscription requires attention/action. i.e. payment error etc. */
    IMS_ERROR_SUBS_NO_SUB                           : "127", /**< No current subscription exists for the user. i.e. closed/removed. */
    IMS_ERROR_NO_BUDDY_GROUP_FOR_CLIENT             : "150", /**< No buddy_group exists for the client queried. */
    IMS_ERROR_CLIENT_REGISTERED_FOR_OTHER_GROUP     : "151", /**< Client is already registered for some group other than requested. */
    IMS_ERROR_GROUP_ENTRY_NOT_PRESENT               : "152", /**< No entry exixts for the group queried. */
    IMS_ERROR_IN_SAVE_GROUP_DATA                    : "153", /**< Error Occured while saving the buddy-group data */
    IMS_ERROR_CNAME_ENTRY_NOT_PRESENT               : "154", /**< CName entry not present */
    IMS_ERROR_IN_SAVE_BACKOFF_DATA                  : "155", /**< Error Occured while saving the backoff data for the cname */
    /**< For HTTP errors 401 to 600, IMSLib returns them as it is, they are not wrapped by any IMSLib errors */
    /**< CEP Errors */
    IMSMANAGER_ERROR_EXCEPTION                      : "3000", /**< Error returned when an exception is thrown during processing in the event callback */
    IMSMANAGER_ERROR_ENCODING                       : "3001", /**< Error returned if the inDetails information failed to be encoded into a Base64 string */
	IMSMANAGER_SUCCESS_BROWSER_OPENED               : "3002", /**< Status returned if CEP successfully opens the default browser with the SSO URL as part of the AttemptSSOJumpWorkflows API */
    IMSMANAGER_ERROR_BROWSER_FAILED_TO_OPEN         : "3003", /**< Status returned if CEP fails to open the SSO URL in the default browser */

    IMS_UNKNOWN_ERROR                               : "0xFFFF"
};

// @deprecated Use NGLInterface.events directly.
NGLInterface.prototype.events = NGLInterface.events;