precision highp float;

//agf_include "color/rgbd.inc"
//agf_include "color/tonemap_identity.inc"
//agf_include "color/hsv.inc"
//agf_include "color/srgb.inc"
//agf_include "logarithmic_z.inc"
//agf_include "materials/get_F0.inc"
//agf_include "materials/Light.inc"
//agf_include "materials/boost_intensity.inc"
//agf_include "materials/samplers.inc"
//agf_include "materials/shade.inc"
//agf_include "math/env_map_math.inc"
//agf_include "math/perturb_normal.inc"
//agf_include "math/rand.inc"

// material properties
DECLARE_TEXTURE_COLOR_SAMPLER(base_color)
DECLARE_TEXTURE_COLOR_SAMPLER(emission)
DECLARE_TEXTURE_COLOR_SAMPLER(reflection)
DECLARE_TEXTURE_COLOR_SAMPLER(refraction)

DECLARE_TEXTURE_SCALAR_SAMPLER(glossiness)
DECLARE_TEXTURE_SCALAR_SAMPLER(index_of_refraction)
DECLARE_TEXTURE_SCALAR_SAMPLER(normal)
DECLARE_TEXTURE_SCALAR_SAMPLER(opacity)
DECLARE_TEXTURE_SCALAR_SAMPLER(refraction_glossiness)
////

DECLARE_TEXTURE_COLOR_SAMPLER(fog)
// uniform float fog_bias;
// uniform float fog_mult;

// daylight
uniform bool  daylight_sun_enabled;
uniform bool  daylight_sky_enabled;
uniform vec3  daylight_color;
uniform vec3  daylight_direction;
uniform float daylight_intensity;
uniform float daylight_cloudiness;

// ibl
uniform sampler2D ibl_map;
uniform sampler2D brdf_map;

uniform float ibl_intensity;
uniform float ibl_max_intensity;
uniform float ibl_threshold;
uniform vec4 ibl_color;
uniform float stochastic_transparency_rand_seed;

uniform vec4 eye_space_ground_plane_equation;
uniform bool ground_plane_occludes_objects;

in vec3 v_N;
in vec2 v_uv;
in vec3 v_P_clip;
in vec3 v_P_camera;
in vec4 v_color;
in vec3 v_P_shadow;

// shadows
//agf_include "materials/shadow_support.inc"

LOGZ_FRAGMENT_DECLARE_FLOGZ
LOGZ_FRAGMENT_DECLARE_FCOEF_HALF

out vec4 fragColor;

uniform mat4 eye_space_normal_ibl_lookup_transformation_matrix;
#define NUM_LIGHTS 1
Light lights[NUM_LIGHTS];

// ibl
uniform float ibl_max_levels;

#ifndef PI
#define PI 3.14159265359
#endif

float average(vec3 color) {
  // dot is faster than taking the average via addition and division. a dot is a single operation on some gpu's
  return dot(vec3(0.3333), color);
}

vec3 remap_ibl_intensity(vec3 color)  {
  vec3 remapped_color = boost_intensity(color, ibl_intensity, ibl_threshold, ibl_max_intensity);
  return remapped_color;
}

vec3 sample_environment(vec3 N, float roughness) {
  vec3 ibl_Normal = (eye_space_normal_ibl_lookup_transformation_matrix*vec4(N, 0)).xyz;
  vec2 env_uv = normal_to_uv(ibl_Normal);
  vec3 sampledColor = textureLod(ibl_map, env_uv, roughness*ibl_max_levels).rgb;
  vec3 remappedColor = remap_ibl_intensity(sampledColor);
  return remappedColor * ibl_color.rgb; // ToDo [j:3.30.17] instead of a simple multiply, implement the Ps "color" blend mode
}

vec2 get_brdf(float roughness, float NdV) {
  vec2 brdf_uv = vec2(roughness, NdV);
  vec2 brdf = texture(brdf_map, brdf_uv).xy;
  return brdf.xy;
}

vec3 get_specular_ibl(vec3 specular_color, float roughness, vec3 N, vec3 V, vec3 vertex_normal) {
  const float horizon_amount = 1.3;
  float NdV = safe_dot(N, V);
  vec3 R = 2.0 * dot(V, N) * N - V;

  // Horizon correction, per http://marmosetco.tumblr.com/post/81245981087
  vec3 reflected = normalize(reflect(N,V));
  float horizon = clamp(1.0 + horizon_amount * dot(reflected, vertex_normal), 0.0, 1.0);
  horizon *= horizon;

  vec3 color = sample_environment(R, roughness);
  vec2 brdf = get_brdf(roughness, NdV);
  return color * (specular_color * brdf.xxx + brdf.yyy);
}

void main() {

  //########################################## CLIPPING PLANES ################################################################################
  float clipSign = dot(eye_space_ground_plane_equation.xyz, v_P_camera.xyz)+eye_space_ground_plane_equation.w;
  if (ground_plane_occludes_objects && clipSign < 0.0) {
    discard;
  }

  // V, for all intents and purposes, should be Vcamera in a fragment shader
  vec3 Vcamera = normalize(-v_P_camera);
  vec3 V = Vcamera;

  lights[0] = Light(normalize(daylight_direction),
    daylight_color.rgb,
    daylight_intensity,
    daylight_cloudiness
  );

  // used to determine horizon for IBL correction. should be raw fragment normal, not changed by texture maps.
  vec3 Nvert = normalize(v_N);
  vec3 Nmap = vec3(0.0, 0.0, 1.0);
  vec3 N = Nvert;


  vec2 duv1 = dFdx( v_uv );
  vec2 duv2 = dFdy( v_uv );

  bool badUV = ((duv1.x == 0.0) && (duv1.y == 0.0) && (duv2.x == 0.0) && (duv2.y == 0.0));

  if ((badUV == false) && normal_enabled) {
    Nmap = 2.0 * texture(normal_texture, v_uv).rgb - vec3(1.0);
    Nmap.g = -Nmap.g;
    N = perturb_normal(Nvert, Nmap, Vcamera, v_uv);
  }

 // vec4 base_color = SAMPLE_COLOR(base_color, v_uv);
  vec4 base_color = texture(base_color_texture, v_uv);

  vec3 base_color_linear = srgb_to_linear(base_color.rgb);

  // reflectivity is the reflectance at normal incidence. should be ~4-8% for most materials, corresponding to
  // an index_of_refraction 1-3 range
  // https://en.wikipedia.org/wiki/Refractive_index#Reflectivity
  float ior = SAMPLE_SCALAR(index_of_refraction, v_uv);
  float metallicity = clamp((ior-3.0)/12.0, 0.0, 1.0);   // stay dielectric while ior is in a realistic range, and then start going toward metallic
  float reflectivity = pow(abs((1.0 - ior) / (1.0 + ior)), 2.0);
  vec4 reflection = SAMPLE_COLOR(reflection, v_uv); // TODO: cpina: should this be made linear? if it is a color, yes, if data no.
  vec3 F0 = reflection.rgb * reflectivity;

  float glossiness = SAMPLE_SCALAR(glossiness, v_uv);
  float roughness = 1.0 - glossiness;

  // unlit
  vec3 accumulated_light = vec3(0.0, 0.0, 0.0);

  //
  // Directional lighting
  //
  // Only enable if light intensity is on
  if (daylight_sun_enabled || daylight_sky_enabled) {
    Material material = Material(
      metallicity,
      roughness,
      base_color_linear,
      F0
    );
    for(int i=0; i < NUM_LIGHTS; i++) {
      vec3 L = lights[i].dir;
      float NdU = dot(N, vec3(0.0, 1.0, 0.0)) * 0.5 + 0.5;
      vec3 sun = shade(L, lights[i].radius * lights[i].radius, V, N, material) * lights[i].color;
      sun *= daylight_sun_enabled ? 1.0 : 0.0;

      // occlude lighting from sun when in shade
      float NdL = safe_dot(N, L);
      float sun_shadow = get_shadow(NdL, v_P_shadow);
      sun *= 1.0 - sun_shadow;

      vec3 overcast = mix(0.0, lights[i].radius, NdU) * vec3(0.4, 0.4, 0.4) * base_color_linear;
      overcast *= daylight_sky_enabled ? 1.0 : 0.0;
      accumulated_light += lights[i].intensity * mix(sun, overcast, lights[i].radius);
    }
  }

  //
  // IBL contribution
  //
  // mdl diffuse
//  accumulated_light += base_color_linear * (sample_environment(N, 0.9)) * vec3(1.0 - reflectivity) * (1.0 - metallic_scalar) * 2.0 / PI;

  // diffuse
  accumulated_light += base_color_linear * (sample_environment(N, 0.9)) * (vec3(1.0) - F0) * (1.0 - metallicity) * 2.0 / PI;
  // reflection
  vec3 reflected_light = get_specular_ibl(F0, roughness, N, V, Nvert);

  // energy used for opacity calc
  float reflected_energy = clamp(average(reflected_light), 0.0, 1.0);
  accumulated_light += reflected_light;

  //
  // Emission
  //
  vec4 emission = SAMPLE_COLOR(emission, v_uv);
  vec3 emission_linear = srgb_to_linear(emission.rgb);

  // energy used for opacity calc
  float emitted_energy = average(emission_linear);

  //
  // Refraction
  //

  vec4 refr_color = SAMPLE_COLOR(refraction, v_uv);
  vec3 refr_color_linear = srgb_to_linear(refr_color.rgb);
  float refr_opacity = 0.0;

  float refr_enabled = average(refr_color_linear);
  vec3 refr_accumulated_light = vec3(0.0, 0.0, 0.0);
  if (refr_enabled > 0.01) {
    // TODO: use refraction glossiness texture. need to figure out how to mix between scalar and texture.
    // maybe use a mix parameter instead of strength, and use strength to add?

    // refractivity
    float ior2 = ior * ior;
    float refractivity = (ior2-1.0)/(ior2 + 2.0);
    float refr_glossiness = SAMPLE_SCALAR(refraction_glossiness, v_uv);
    float refr_roughness = 1.0 - refr_glossiness;

    // Scattering contribution
    //

    // Fog tints the glass. Let's assume it tints it 100% (it is the limit, after all)
    vec4 fog_color = SAMPLE_COLOR(fog, v_uv);
    vec3 fog_color_linear = srgb_to_linear(fog_color.rgb);

    float NdV = abs(dot(N,V));
    vec2 brdf = get_brdf(roughness, NdV);

    // as fog color moves away from white, some colors must be causing scattering, so here we determine the tint that comes through
    float scattered_energy = 1.0 - average(fog_color_linear);
    scattered_energy *= 0.5; // the fog should not be completely opaque (it's technically volume dependent, but we don't know the volume)
    float facing_angle = 1.0 - NdV;
    facing_angle *= facing_angle;
    scattered_energy = mix(scattered_energy, 1.0, facing_angle); // at glancing angles, we are passing through more volume, let more scattered color show


    // Compensate for light lost due to reflection. BRDF y component is fresnel. This gives us a bit of darkening on edges, seems a little wrong
    // but it looks more right - maybe simulating internal reflection?
    // 1: no fresnel compensation
    // float fresnel_compensation = 1.0;
    // 2: faked fresnel using facing angle
    // float fresnel_compensation = 1.0 - facing_angle;
    // 3: fresnel from brdf
    float fresnel_compensation = (1.0 - brdf.y);

    // reflection
    // vec3 conserve_reflected_energy = vec3(1.0) - clamp(reflected_light, 0.0, 1.0); // 1
    // float conserve_reflected_energy = 1.0 - reflected_energy; // 2
    // float conserve_reflected_energy = 1.0; // 3
    refr_accumulated_light += fog_color_linear * fresnel_compensation;

    // Reflected IBL contribution
    //
    refr_accumulated_light += reflected_light;

    // Refracted IBL - doesn't look that great, I think we can get by without it
    // vec2 flip = vec2(1.0, -1.0);
    // vec3 refr_N = N * flip.xxy; // -N
    // vec3 refr_V = V * flip.xxy; // -V

    //
    // float refraction_amount = 1.0 - clamp(0.0, 1.0, reflected_energy + scattered_energy);
    // refraction_amount *= 0.1;
    // refr_accumulated_light += refraction_amount * PI * get_specular_ibl(refr_color_linear, refr_roughness, refr_N, refr_V, refr_N);

    // Stochastic opacity compensation -
    // For reflections, emitted light, and scattered light, the light does not go through the object, so it should not be transparent
    //
    refr_opacity = reflected_energy + scattered_energy + emitted_energy;
  }

  // Sampling opacity: opacity map is a color texture, so we pick a max out of the components to get a single value
  float base_opacity = SAMPLE_SCALAR(opacity, v_uv);

  // initial opacity is either the base opacity, or the opacity from refraction
  float final_opacity = mix(base_opacity, base_opacity * refr_opacity, refr_enabled);

  if (stochastic_transparency_rand_seed>0.0) {
    //Stochastic transparency. Code can be put anywhere after opacity math is done and result is in 'final_opacity'
    float heisenbergOpacity = rand(gl_FragCoord.xy+v_P_clip.zz+vec2(stochastic_transparency_rand_seed,stochastic_transparency_rand_seed));

    float testOpacity = final_opacity;

    if (testOpacity < 0.05) {
      testOpacity = 0.1*final_opacity;
    }

    if (heisenbergOpacity > testOpacity) {
      discard;
    }

    final_opacity = 1.0;
  }

  // mix between lighting models based on whether refraction is enabled
  accumulated_light = mix(accumulated_light, refr_accumulated_light, refr_enabled);
  accumulated_light += emission_linear;

  fragColor = vec4(linear_to_srgb(tonemap(accumulated_light)), final_opacity);

  // Debug IBL:
  // fragColor = vec4(linear_to_srgb(get_specular_ibl(brdf_map, ibl_map, eye_space_normal_ibl_lookup_transformation_matrix, vec3(1.0), roughness, N, V, Nvert)), 1.0);
  // Debug normal map:
  // fragColor = vec4(Nmap/2.0 + vec3(0.5), 1.0);
  // Debug normals:
  //  fragColor = vec4(N/2.0 + vec3(0.5), 1.0);

  LOGZ_FRAGMENT_ADJUST_DEPTH_WITH_FLOGZ_FCOEF
}
