//agf_include "materials/Material.inc"
//agf_include "materials/Light.inc"
//agf_include "math/safe_dot.inc"

#ifndef PI
#define PI 3.14159
#endif

// Unreal GGX implementation (Unreal SIGGRAPH 2013)
// http://blog.selfshadow.com/publications/s2013-shading-course/karis/s2013_pbs_epic_notes_v2.pdf

// Schlick approximation
vec3 fresnel_term(float VdH, vec3 F0) {
  return F0 + (vec3(1.0, 1.0, 1.0)-F0) * pow(2.0, (-5.55473*VdH-6.98316)*VdH);
}

float geometry_term(float roughness, vec3 V, vec3 N, vec3 H) {
  // remapped roughness, to be used for the geometry term calculations,
  // per Disney [16], Unreal [3]. N.B. don't do this in IBL
  float roughness_remapped = 0.5 + roughness/2.0;

  // ToDo: surfaces with little roughness still catch a lot of light far from the specular center
  float NdV = safe_dot(N, V);

  float k = pow(roughness_remapped + 1.0, 2.0)/8.0;
  return NdV/((NdV)*(1.0-k)+k);
}

float distribution_term(float NdH, float alpha2) {
  float D_denominator = ((NdH*NdH)*(alpha2-1.0)+1.0);
  return alpha2/(PI * pow(D_denominator, 2.0));
}

vec3 shade(Light light, vec3 V, vec3 N, Material material) {
  float alpha = material.roughness * material.roughness + light.radius;
  alpha = min(1.0, alpha);
  float alpha2 = alpha * alpha;

  vec3 L = light.dir;
  vec3 H = normalize(V+L);

  float VdH = safe_dot(V, H);
  float NdH = safe_dot(N, H);
  float NdL = safe_dot(N, L);
  float NdV = safe_dot(N, V);

  float D = distribution_term(NdH, alpha2);
  float Gl = geometry_term(material.roughness, L, N, H);
  float Gv = geometry_term(material.roughness, V, N, H);
  vec3 F = fresnel_term(VdH, material.F0);

  vec3 specular_contribution = F*(Gl*Gv*D/(4.0*NdL*NdV + 0.000001));
  vec3 diffuse_color = material.base_color * (vec3(1,1,1) - material.F0) * (1.0 - material.metalicity) * 2.0 / PI;

  return (specular_contribution + diffuse_color) * NdL;
}

float sun_distribution_term(float NdH, float alpha2, float cloudiness) {
  float D_denominator = ((NdH*NdH)*(alpha2-1.0)+1.0);
  float falloff_rate = mix(3.0, 1.7, cloudiness);
  float brightness = mix(0.1, 1.0, cloudiness);
  return brightness * alpha2/(PI * pow(D_denominator, falloff_rate));
}

vec3 sun_shade(Light light, vec3 V, vec3 N, float horizonEdge, Material material) {
  float cloudiness = light.radius;
  float alpha = max(material.roughness*material.roughness, 0.25*cloudiness*cloudiness);
  alpha = min(1.0, alpha);
  float alpha2 = alpha * alpha;

  vec3 L = light.dir;
  vec3 H = normalize(V+L);

  float VdH = safe_dot(V, H);
  float NdH = safe_dot(N, H);
  float NdL = safe_dot(N, L);
  float NdV = safe_dot(N, V);

  float D = sun_distribution_term(NdH, alpha2, cloudiness);
  float Gl = geometry_term(material.roughness, L, N, H);
  float Gv = geometry_term(material.roughness, V, N, H);
  vec3 F = fresnel_term(VdH, material.F0);

  float horizon_blend = max(0.01, max(material.roughness, (1.0 - material.metalicity)));
  float horizon_mask = smoothstep(-horizon_blend*0.2, horizon_blend*0.2, horizonEdge);
  float horizon_mask_roughness_lerp = mix(horizon_mask, 1.0, horizon_blend);

  float sun_size =  0.001 * mix(0.02, 3.0, material.roughness);
  float sun_diffusion_mult = mix(1, 0.5, cloudiness);
  float sun_intensity = max(light.intensity,pow(light.intensity,16.0)) * 29.0;
  vec3 disc = 1.0 * vec3_splat(smoothstep(1.0 - sun_size, 1.0 - horizon_blend * sun_size, NdH)) * 2.0*max(0.0,min(0.5-cloudiness,0.5-material.roughness));

  vec3 specular_contribution = F*Gl*Gv*D/(4.0*NdL*NdV + 0.000001);
  vec3 diffuse_color = material.base_color * (vec3_splat(1.0) - material.F0) * (1.0 - material.metalicity) * 2.0 / PI;

  vec3 sun_light = (specular_contribution * sun_diffusion_mult + diffuse_color) * NdL;
  sun_light += disc;
  sun_light *= light.color * sun_intensity;
  sun_light *= horizon_mask_roughness_lerp;
  return sun_light;
}
