/***********************************************************************/
/*                                                                     */
/*                      ADOBE CONFIDENTIAL                             */
/*                   _ _ _ _ _ _ _ _ _ _ _ _ _                         */
/*                                                                     */
/*  Copyright 2016 Adobe Systems Incorporated                          */
/*  All Rights Reserved.                                               */
/*                                                                     */
/* NOTICE:  All information contained herein is, and remains           */
/* the property of Adobe Systems Incorporated and its suppliers,       */
/* if any.  The intellectual and technical concepts contained          */
/* herein are proprietary to Adobe Systems Incorporated and its        */
/* suppliers and are protected by all applicable intellectual property */
/* laws, including trade secret and copyright laws.                    */
/* Dissemination of this information or reproduction of this material  */
/* is strictly forbidden unless prior written permission is obtained   */
/* from Adobe Systems Incorporated.                                    */
/*                                                                     */
/***********************************************************************/

function Controller(/*[Function]*/ inGlobalCallback)
{
	Controller.instance = this;

	var workflowStorage = IWorkflowStorage.create();
	var currentWorkflow = null;
	var currentProcessor = null;
	var globalCallback = inGlobalCallback;
	var surface = new PanelSurface();
	var updateWorkflowList = true;
	
	// initialize
	(function()
	{
		switchSurface(PANELSURFACE_WORKFLOW_LIST);
		
		try
		{
			displayAllWorkflowTitles();
		}
		catch(exc)
		{
			exclog(exc);
		}
	})();
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Display start screen
	//
	this.displayWorkflowList = function()
	{
		switchSurface(PANELSURFACE_WORKFLOW_LIST);
		updateWorkflowList = true;

		try
		{
			displayAllWorkflowTitles();
		}
		catch(exc)
		{
			exclog(exc);
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Select a workflow
	//
	this.selectWorkflow = function(/*[String]*/ inID)
	{
		updateWorkflowList = false;
		var thisObj = this;
		currentProcessor = null;
		workflowStorage.getWorkflow(inID, function(inWorkflow)
		{
			currentWorkflow = inWorkflow;
		
			if (isValidProperty(currentWorkflow))
			{
				try
				{
					currentProcessor = new WorkflowProcessor(currentWorkflow, surface, globalCallback);
					switchSurface(PANELSURFACE_WORKFLOW);
					currentProcessor.initialize();
				}
				catch(exc)
				{
					exclog(exc);
					currentProcessor = null;
					// back to start surface
					thisObj.displayWorkflowList();
				}
			}
			else
			{
				updateWorkflowList = true;
				currentWorkflow = null;
			}
		});
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	this.workflowCommand = function(/*[String]*/ inCommand)
	{
		switch(inCommand)
		{
			// Start workflow processing
			//
			case CLICKEVENT_WORKFLOW_START:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					incWorkflowAttempts(currentWorkflow.id);
					currentProcessor.start();
				}
			}
			break;
			
			// Restart current workflow
			//
			case CLICKEVENT_WORKFLOW_RESTART:
			{
				if (isValidProperty(currentWorkflow))
				{
					this.selectWorkflow(currentWorkflow.id);
				}
			}
			break;
			
			// Repeat the current step of a processing workflow
			//
			case CLICKEVENT_WORKFLOW_STEP_REPEAT:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					currentProcessor.refreshStep();
				}
			}
			break;
			
			// Cancel workflow procesing
			//
			case CLICKEVENT_WORKFLOW_CANCEL:
			{
				if (isValidProperty(currentProcessor))
				{
					if (currentProcessor.isStarted())
					{
						switchSurface(PANELSURFACE_WORKFLOW_CANCELED);
						currentProcessor.cancel();
					}
					else
					{
						this.displayWorkflowList();
					}
				}
			}
			break;
			
			case CLICKEVENT_WORKFLOW_STEP_BACK:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					currentProcessor.prevStep();
				}
			}
			break;
			
			case WorkflowProcessor.FINISHED_NEXT_STEP:
			case CLICKEVENT_WORKFLOW_STEP_NEXT:
			case CLICKEVENT_WORKFLOW_STEP_SKIP:
			{
				if (isValidProperty(currentProcessor))
				{
					switchSurface(PANELSURFACE_WORKFLOW_STEP);
					currentProcessor.nextStep();
				}
			}
			break;
			
			case WorkflowProcessor.STARTED:
			case WorkflowProcessor.SWITCH_STEP_NEXT:
			case WorkflowProcessor.SWITCH_STEP_PREV:
			{
				logAction(inCommand, currentProcessor, currentWorkflow);
			}
			break;
			
			// Execute the actual step (if expression is available)
			//
			case CLICKEVENT_WORKFLOW_STEP_EXEC:
			{
				// TODO: execute workflow step
			}
			break;
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Callback function: Workflow processing finished
	//
	this.onWorkflowFinished = function(/*[String]*/ inWhy, /*[Any]*/ inData)
	{
		switch (inWhy)
		{
			case WorkflowProcessor.FINISHED_SUCCESS:
			{
				logAction(WorkflowProcessor.FINISHED_SUCCESS, currentProcessor, currentWorkflow);
				switchSurface(PANELSURFACE_WORKFLOW_SUCCEEDED);
				var display = new WorkflowDisplayAdapter(surface, currentWorkflow);
				display.showSuccess();
			}
			break;
			
			case WorkflowProcessor.FINISHED_FAILURE:
			{
				switchSurface(PANELSURFACE_WORKFLOW_STEP_FAILURE);
				var display = new WorkflowDisplayAdapter(surface, currentWorkflow);
				
				if (isValidProperty(currentProcessor))
				{
					display.showFailure(currentProcessor.getCurrentStep(), inData);
				}
			}
			break;
			
			case WorkflowProcessor.FINISHED_CANCELED:
			{
				logAction(WorkflowProcessor.FINISHED_CANCELED, currentProcessor, currentWorkflow);
			}
			break;

			case CLICKEVENT_WORKFLOW_SUCCESS_COMMIT:
			case CLICKEVENT_WORKFLOW_CANCEL_COMMIT:
			{
				currentWorkflow = null;
				currentProcessor = null;
				this.displayWorkflowList();
			}
			break;
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Callback: Workflow step succeeded
	//
	this.onWorkflowStepSucceeded = function()
	{
		switchSurface(PANELSURFACE_WORKFLOW_STEP_SUCCESS);
		var display = new WorkflowDisplayAdapter(surface, currentWorkflow);

		if (isValidProperty(currentProcessor))
		{
			display.showStepSuccess(currentProcessor.getCurrentStep());
		}
	}

	// private ///////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////////////
	//
	// Called when new Workflows are available
	//
	function onNewWorkflows(/*[Array]*/ inWorkflows)
	{
		if (updateWorkflowList && surface.isCurrent(PANELSURFACE_WORKFLOW_LIST))
		{
			if (isValidProperty(inWorkflows) && isValidProperty(inWorkflows.length))
			{
				var display = new WorkflowDisplayAdapter(surface);
				display.showTitles(CONTENTLOCATOR_WORKFLOW_LIST, inWorkflows, onWorkflowSelect);
			}
		}
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Display all workflow titles
	//
	function displayAllWorkflowTitles()
	{
		workflowStorage.getCachedWorkflows(onNewWorkflows);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Switch panel display
	//
	function switchSurface(/*[String]*/ inName)
	{
		// workflowlist	- list of all available workflows
		// workflow		- a single in process workflow
		surface.displaySurface(inName);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Increase number of attempts of the workflow
	//
	function incWorkflowAttempts(/*[String]*/ inWorkflowID)
	{
		var cs = new CSInterface();
		cs.evalScript('incWorkflowAttempts("' + inWorkflowID + '");');
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Return number of attempts of the workflow
	//
	function getWorkflowAttempts(/*[String]*/ inWorkflowID, /*[Function]*/ inCallback)
	{
		var script = 'getWorkflowAttempts("' + inWorkflowID + '");';
		var cs = new CSInterface();
		cs.evalScript(script, inCallback);
	}
	
	//////////////////////////////////////////////////////////////////////////////
	//
	// Log data
	//
	function logAction(/*[String]*/ inWhy, /*[WorkflowProcessor]*/ inProcessor, /*[Workflow]*/ inWorkflow)
	{
		if (isValidProperty(inProcessor))
		{
			var timeMeasure = inProcessor.getMeasurements();
			
			var workflowActions = [];
			var actions = [];
			var id = inWorkflow.id;
			var name = inWorkflow.name;
			var session = inWorkflow.session;
			var workflowDuration = -1;
			var stepDuration = -1;
			var stepNumber = inProcessor.getCurrentStep();
			var retry = inProcessor.getRetry();
			var failure = inProcessor.isFailure();
			var interactive = false;
			var numSteps = inWorkflow.getStepLength();
			var stepInteractivity = "no step interactivity";
					
			if (isValidProperty(timeMeasure['workflow']))
			{
				workflowDuration = timeMeasure['workflow'].getElapsed();
			}
			
			if (isValidProperty(timeMeasure['step_total' + stepNumber]))
			{
				stepDuration = timeMeasure['step_total' + stepNumber].getElapsed();
			}

			if (!isNaN(stepNumber))
			{
				var step = inWorkflow.getStep(stepNumber);
				if (isValidProperty(step) && step.getTriggers().length > 0)
				{
					stepInteractivity = "step interactivity"

					if (inProcessor.isSuccess())
					{
						stepInteractivity += " success";
					}
					else if (failure)
					{
						stepInteractivity += " failure";
					}
				}
			}
			
			switch (inWhy)
			{
				case WorkflowProcessor.STARTED:
				{
					actions.push('WorkflowStarted');
					workflowActions.push(true);
				}
				break;
				
				case WorkflowProcessor.FINISHED_SUCCESS:
				{
					actions.push('WorkflowStepSuccess');
					workflowActions.push(false);
					actions.push('WorkflowSuccess');
					workflowActions.push(true);
				}
				break;
				
				case WorkflowProcessor.FINISHED_CANCELED:
				{
					actions.push(failure ? 'WorkflowCanceledFailure' : 'WorkflowCanceled');
					workflowActions.push(false);
				}
				break;

				case WorkflowProcessor.SWITCH_STEP_NEXT:
				case WorkflowProcessor.SWITCH_STEP_PREV:
				{
					actions.push('WorkflowStepSuccess');
					workflowActions.push(false);
				}
				break;
			}
			
			getWorkflowAttempts(inWorkflow.id, function(attempts)
			{
				function createScript(/*[String]*/ inAction, /*[String]*/ inWorkflowAction)
				{
					var script = 'logAction(';
					script += inWorkflowAction;
					script += ',';
					script += '"' + inAction + '"';
					script += ',';
					script += '"' + id + '"';
					script += ',';
					script += '"' + name + '"';
					script += ',';
					script += '"' + session + '"';
					script += ',';
					script += attempts;
					script += ',';
					script += stepNumber;
					script += ',';
					script += retry;
					script += ',';
					script += '"' + stepInteractivity + '"';
					script += ',';
					script += workflowDuration;
					script += ',';
					script += stepDuration;
					script += ',';
					script += numSteps;
					script += ');';

					return script;
				}

				var esscript = '';

				for (var i=0; i<actions.length; i++)
				{
					esscript += createScript(actions[i], workflowActions[i]);
				}

				var cs = new CSInterface();
				cs.evalScript(esscript);
			});
		}
	}
}

