/**************************************************************************************************
*
* ADOBE SYSTEMS INCORPORATED
* Copyright 2013 Adobe Systems Incorporated
* All Rights Reserved.
*
* NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the
* terms of the Adobe license agreement accompanying it.  If you have received this file from a
* source other than Adobe, then your use, modification, or distribution of it requires the prior
* written permission of Adobe.
*
**************************************************************************************************/

/** IMSInterface - v6.1.0 */

/**
 * @class AAMStatus
 * Stores status information for the Adobe Application Manager.
 * All values must use UTF-8 encoded chars.
 *
 * @param status   The authorization status.
 * @param callerId The caller ID, a unique ID for your application created by PlugPlug. The ID identifies the service that made the original request.
 * @param payload  For future use.
 *
 * @return An \c AAMStatus object.
 **/
function AAMStatus(status, callerId, payload)
{
    this.status = status;
    this.callerId = callerId;
    this.payload = payload;
}

/**
 * @class IMSInterface
 * This is the entry point to the Adobe Identity Management Service (IMS) toolkit library.
 * Instantiate this class, and use the object to:
 * <ul>
 * <li>Retrieve a subscription access token for a particular service.</li>
 * <li>Retrieve a list of users who are registered on this device.</li>
 * <li>Invoke Adobe Application Manager (AAM) to interact with user if needed.</li>
 * </ul>
 * Most requests include a session reference, which is returned from a connection request,
 * and an IMS client ID, which uniquely identifies your service and service context. When
 * developing a CS Extension, you must register with the IMS team to receive your client ID.
 *
 * @return The \c IMSInterface object for an IMS session.
 */
function IMSInterface()
{
}

/**
 * Registers a  handler for a CS event. The handler is a callback function that is
 * passed a single parameter, the \c #CSEvent object containing detailed information
 * about the event.
 *
 * You must register callback functions to handle the results of asynchronous calls
 * to the \c IMSInterface methods.
 *
 * @param type      The event-type  constant.
 * @param listener  The JavaScript handler function for the event.
 */
IMSInterface.prototype.addEventListener = function(type, listener)
{
    window.__adobe_cep__.addEventListener(type, listener);
};

/**
 * Removes the handler associated with a CS event.
 *
 * @param type      The event-type  constant.
 * @param listener  The JavaScript handler function that was registered for the event.
 */
IMSInterface.prototype.removeEventListener = function(type, listener)
{
    window.__adobe_cep__.removeEventListener(type, listener);
};

/**
 * Establishes an IMS session. Must be called before any IMS access methods.
 * This method is not thread safe.
 *
 * @return An IMS reference string, as returned from IMSLib, which you
 *  can then use to make further IMS calls to this object's methods.
 *
 * @deprecated Please use imsConnectWithEndpoint instead.
 */
IMSInterface.prototype.imsConnect = function()
{
    var imsRef = window.__adobe_cep__.imsConnect();
    return imsRef;
};

/**
 * Disconnects the extension from IMS and disposes of the IMS reference.
 *
 * @param imsRef     An IMS reference returned from the \c IMSConnect() call.
 */
IMSInterface.prototype.imsDisconnect = function(imsRef)
{
    window.__adobe_cep__.imsDisconnect(imsRef);
};

/**
 * Retrieves user account information from IMS/Renga.
 *
 * @param imsRef    An IMS reference returned from the \c IMSConnect() call.
 * @param clientID  The IMS client ID for your extension, assigned by the IMS team on registration.
 *
 * @return Account information in XML string format. This is an example of the returned XML string:\n
 * <listing>\n
 *   &lt;UserAccounts&gt;\n
 *     &lt;UserData default="true"&gt;\n
 *       &lt;UserID&gt;12345&#64;AdobeID&lt;/UserID&gt;\n
 *       &lt;Name&gt;Joe Bloggs&lt;/Name&gt;\n
 *       &lt;FirstName&gt;Joe&lt;/FirstName&gt;\n
 *       &lt;LastName&gt;Bloggs&lt;/LastName&gt;\n
 *       &lt;Email&gt;Joe_Bloggs&#64;testemail.com&lt;/Email&gt;\n
 *       &lt;SAOList&gt;\n
 *         &lt;SAOData id="123456789"&gt;\n
 *           &lt;ServiceCode&gt;my_service&lt;/ServiceCode&gt;\n
 *           &lt;ServiceStatus&gt;ACTIVE&lt;/ServiceStatus&gt;\n
 *           &lt;ServiceLevel&gt;CS_LVL_1&lt;/ServiceLevel&gt;\n
 *         &lt;/SAOData&gt;\n
 *       &lt;/SAOList&gt;\n
 *     &lt;/UserData&gt;\n
 *   &lt;/UserAccounts&gt;\n
 * </listing>\n
 */
IMSInterface.prototype.imsFetchAccounts = function(imsRef, clientId)
{
    var accounts = window.__adobe_cep__.imsFetchAccounts(imsRef, clientId);
    return accounts;
};

/**
 * Requests an access token from IMS for a given user and service.
 *
 * This function is asynchronous. To handle the result, register a callback for the event 
 * IMSInterface.events.imsFetchAccessTokenWithStatus or IMSInterface.events.imsFetchAccessToken.
 * For event IMSInterface.events.imsFetchAccessToken, the event data is a JSON string 
 * with format {"toua":"...",...,"emailVerified":"false"}.
 * For event IMSInterface.events.imsFetchAccessTokenWithStatus, the event data is a JSON string
 * with format {"status":"0","details":{"toua":"...",...,"emailVerified":"false"}}.
 *
 * @see addEventListener()
 *
 * @param imsRef         An IMS reference returned from the \c IMSConnect() call.
 * @param clientID       The IMS client ID for your extension, assigned by the IMS team on registration.
 *               This is the service-code value in the Renga Service Account Object for your client.
 * @param clientSecret       The client secret associated with the provided client ID
 * @param userAccountGuid    The unique person GUID for the Renga account, as returned by
                the <code>fetchAccounts()</code> method. The token is generated for this user.
 * @param serviceAccountGuid (optional, not currently used) A unique identifier for a Service Account Object (SAO).
 * @param scope          (optional) A comma delimited list of services for which the refresh and access tokens are requested.
                By default, this is 'openid,AdobeID'. Scope strings are case sensitive.
                If the cached version of the refresh token does not match the
                requested service scopes, a new refresh token is fetched.
                To request your service's own SAO, add your service to the list;
                for example,  'openid,AdobeID,browserlab'.
 *
 * @return A boolean status. Returning true means only the call to imsFetchAccessToken function itself is successful. 
 *							Returning false means that failed to fetch access token.
 */
IMSInterface.prototype.imsFetchAccessToken = function(imsRef, clientId, clientSecret, userAccountGuid, serviceAccountGuid, scope)
{
    return window.__adobe_cep__.imsFetchAccessToken(imsRef, clientId, clientSecret, userAccountGuid, serviceAccountGuid, scope);
};

/**
 * Requests that IMS revoke a user's device token.
 * On success, this removes the user account from the current device.
 * The device token allows access to all services managed by the Adobe Application Manager (AAM),
 * so its removal affects all IMS-related services running on the device.
 *
 * This function is asynchronous. To handle the result, register a callback for the event 
 * IMSInterface.events.imsFetchAccessTokenWithStatus or IMSInterface.events.imsFetchAccessToken
 * @see addEventListener()
 *
 * @param imsRef        An IMS reference returned from the \c IMSConnect() call.
 * @param clientID      The IMS client ID for your extension, assigned by the IMS team on registration.
 *              This is the service-code value in the Renga Service Account Object for your client.
 * @param clientSecret      The client secret associated with the provided client ID
 * @param userAccountGuid   The unique person GUID for the Renga account, as returned by
                the <code>fetchAccounts()</code> method. The token is generated for this user.
 * @param serviceAccountGuid (optional, not currently used) A unique identifier for a Service Account Object (SAO).
 *
 * @return A boolean status. Returning true means only the call to imsRevokeDeviceToken function itself is successful. 
 *							 Returning false means that failed to revoke device token.
 */
IMSInterface.prototype.imsRevokeDeviceToken = function(imsRef, clientId, clientSecret, userAccountGuid, serviceAccountGuid)
{
    return window.__adobe_cep__.imsFetchAccessToken(imsRef, clientId, clientSecret, userAccountGuid, serviceAccountGuid, "REVOKE");
};

/**
 * Persists proxy credentials for an Adobe ID in the local store.
 *
 * @param proxyUsername The username for the Adobe ID credential. Must not be an empty string.
 * @param proxyPassword The password for the Adobe ID credential. Must not be an empty string.
 */
IMSInterface.prototype.imsSetProxyCredentials = function(proxyUsername, proxyPassword)
{
    window.__adobe_cep__.imsSetProxyCredentials(proxyUsername, proxyPassword);
};

/**
 * Launches the Adobe Application Manager (AAM) AIR application.
 * If an error occurs when fetching an access token, this allows the user to
 * log in with their Adobe ID credentials, or accept new terms of use.
 *
 * This function is asynchronous. To handle the result, register a callback for the event IMSInterface.events.imsAAMIMSStatus.
 * The returned event contains an \c #AAMStatus object.
 * @see addEventListener()
 *
 * @param clientID       The IMS client ID for your extension, assigned by the IMS team.
 *               Matches the <code>serviceCode</code> value in the Service Account Object (SAO).
 * @param clientSecret       The client secret associated with the provided client ID.
 * @param redirectUri        The redirect URI registered with IMS
 * @param userAccountGuid    The unique person GUID for the Renga account, as returned in the response
                to the <code>fetchAccounts()</code> method.
 * @param serviceAccountGuid (optional, not currently used) A unique identifier for a Service Account Object (SAO).
 * @param scope          (optional) A comma delimited list of services for which the refresh token was requested.
 *
 * @return A boolean status. Returning true means only the call to showAAM function itself is successful. 
 *							Returning false means that failed to show AAM.
 */
IMSInterface.prototype.showAAM = function(clientId, clientSecret, redirectUri, userAccountGuid, serviceAccountGuid, scope)
{
    return window.__adobe_cep__.showAAM(clientId, clientSecret, redirectUri, userAccountGuid, serviceAccountGuid, scope);
};

/**
 * Retrieves the IMS user id that is currently logged in.
 *
 * Since 4.1.0
 *
 * @return The IMS user id for the currently logged in user.
 */
IMSInterface.prototype.imsGetCurrentUserId = function()
{
    var userId = window.__adobe_cep__.getCurrentImsUserId();
    return userId;
};

/**
 * Retrieves the IMS user id that is currently logged in. This function
 * also takes care of the workarounds specified for some of the products:
 * https://wiki.corp.adobe.com/display/csxs/CEP+5.0+HTML+Extension+Cookbook#CEP5.0HTMLExtensionCookbook-GetthecurrentIMSUserID
 * 
 * @param callback This function is asynchronous so provide a callback to handle the result.
 * 
 * Since 5.2.0
 *
 * @return The IMS user id for the currently logged in user.
 */
IMSInterface.prototype.imsGetCurrentUserIdHelper = function(callback)
{
    // different workflows for Ps, Ai and ID.
    var hostEnvironment = JSON.parse(window.__adobe_cep__.getHostEnvironment());
    var appName = hostEnvironment.appName;
    if (appName === "ILST") {
        window.__adobe_cep__.evalScript('app.userGUID', function(result){
            callback(result);
        });
    } else if (appName === "IDSN") {
        window.__adobe_cep__.evalScript('app.userGuid', function(result){
            callback(result);
        });
    } else if (appName === "PHSP" || appName === "PHXS") {
        window.__adobe_cep__.evalScript("var getUserIdPhotoshop = function() {var userId = '';try {var bhnc = app.charIDToTypeID('bhnc');var ref = new ActionReference();ref.putProperty(app.charIDToTypeID('Prpr'), bhnc);ref.putEnumerated(app.charIDToTypeID('capp'), app.charIDToTypeID('Ordn'), app.charIDToTypeID('Trgt'));var appDesc = app.executeActionGet(ref);if (appDesc.hasKey(bhnc)) {userId = appDesc.getString(bhnc);}} catch (e) {}return userId;};", function(result){
            window.__adobe_cep__.evalScript('getUserIdPhotoshop()', function(result2) {
                callback(result2);
            });
        });
    } else {
        var userId = window.__adobe_cep__.getCurrentImsUserId();
        callback(userId);
    }
};

/**
 * Establishes an IMS session. Must be called before any IMS access methods.
 *
 * Since 5.2.0
 *
 * @param imsEndpoint The IMS URL which needs to be passed in the format like "ims-na1.adobelogin.com".
 * If imsEndpoint is set to undefined, null or empty string, IMSLib will use production endpoint of IMS.
 *
 * @return An IMS reference string, as returned from IMSLib, which you
 *  can then use to make further IMS calls to this object's methods.
 */
IMSInterface.prototype.imsConnectWithEndpoint = function(imsEndpoint)
{
    var params = {};
    params.imsEndpoint = (imsEndpoint === undefined || imsEndpoint === null || imsEndpoint === "") ? "undefined" : imsEndpoint;
    return JSON.parse(window.__adobe_cep__.invokeSync("imsConnectWithEndpoint", JSON.stringify(params))).result;	
};

/**
 * Requests the continue token for a particular client ID with the provided access token. 
 * If the redirectUri parameter is provided then a JumpURL will be created by appending the token to the URL. 
 * This enables the support of SSO workflows from desktop to web.
 *
 * This function is asynchronous. To handle the result, register a callback for the event IMSInterface.events.imsFetchContinueToken
 * @see addEventListener()
 *
 * Since 5.2.0
 *
 * @param imsRef         An IMS reference returned from the \c IMSConnect() call.
 * @param bearerToken       The access token issued by IMS.
 * @param targetClientId    The client ID needed for the destination.
 * @param redirectUri  (optional) The URL to jump to. It should be URL encoded.
 * @param scope        (optional) A comma delimited list of services for which the refresh token was requested.
 * @param responseType (optional) The IMS integration type the target has.
 * @param locale       (optional) The locale in the format language_country.
 *
 * @return A boolean status. Returning true means only the call to imsFetchContinueToken function itself is successful. 
 *							Returning false means that failed to fetch continue token.
 */
IMSInterface.prototype.imsFetchContinueToken = function(imsRef, bearerToken, targetClientId, redirectUri, scope, responseType, locale)
{
    if ((redirectUri === undefined) || (redirectUri === null))
    {
        redirectUri = "";
    }
    if ((scope === undefined) || (scope === null))
    {
        scope = "";
    }
    if ((responseType === undefined) || (responseType === null)) 
    {
        responseType = "";
    }
    if ((locale === undefined) || (locale === null))
    {
        locale = "";
    }    
        
    var params = {};
    params.imsRef = imsRef;
    params.bearerToken = bearerToken;
    params.targetClientId = targetClientId;
    params.redirectUri = redirectUri;
    params.scope = scope;
    params.responseType = responseType;
    params.locale = locale;
    return JSON.parse(window.__adobe_cep__.invokeSync("imsFetchContinueToken", JSON.stringify(params))).result;
};

/**
 * This API will remove a particular user as default for a client_id. In case the user has logged in only for a specific client,
 * this API will revoke user's deviceToken at server end.
 *
 * This function is asynchronous. To handle the result, register a callback for the event IMSInterface.events.imsLogoutUser.
 * @see addEventListener()
 *
 * Since 6.0.0
 *
 * @param imsRef              An IMS reference returned from the \c IMSConnect() call.
 * @param userAccountGuid     The unique person GUID for the Renga account, as returned in the response
                              to the <code>fetchAccounts()</code> or <code>GetCurrentUserId</code> methods.
 * @param clientId            The IMS client ID for your extension, assigned by the IMS team.
 *                            Matches the <code>serviceCode</code> value in the Service Account Object (SAO).
 *
 * @return An IMS reference string, as returned from IMSLib, which you
 *  can then use to make further IMS calls to this object's methods.
 */
IMSInterface.prototype.imsLogoutUser = function(imsRef, userAccountGuid, clientId)
{
    var params = {};
    params.imsRef = imsRef;
    params.userAccountGuid = userAccountGuid;
    params.clientId = clientId;
    return JSON.parse(window.__adobe_cep__.invokeSync("imsLogoutUser", JSON.stringify(params))).result;
};

/**
 * Fetches the user profile data for a given person guid.
 *
 * @param imsRef             An IMS reference returned from the \c IMSConnect() call.
 * @param userAccountGuid    The unique person GUID for the Renga account, as returned in the response
 *                           to the <code>fetchAccounts()</code> or <code>GetCurrentUserId</code> methods.
 * @return User Profile Data in XML string format.
 */
IMSInterface.prototype.imsFetchUserProfileData = function(imsRef, userAccountGuid)
{
    var params = {};
    params.imsRef = imsRef;
    params.userAccountGuid = userAccountGuid;
    return window.__adobe_cep__.invokeSync("imsFetchUserProfileData", JSON.stringify(params));
};

/**
 * A helper function to simplify supporting Single-Sign-On (SSO) workflows in services. There are three different levels of customization:
 * <ul>
 *  <li>The most basic is to pass true for openBrowser and include a valid URL. CEP will open the URL in the default browser using the existing API cep.util.openURLInDefaultBrowser(). This does not attempt any SSO workflows.</li>
 *  <li>By supplying valid entries for all the other parameters but passing false for openBrowser, CEP will complete the SSO workflows but will return the generated URL to the client so that they can complete the last step. The data returned to the client will be the same as the response currently returned in API IMSInterface.imsFetchContinueToken.</li>
 *  <li>The most comprehensive workflow is to include all parameters and pass true to OpenBrowser. CEP will complete the SSO workflows and open the generated URL in the default browser so that the user is automatically signed into the web service.</li>
 * </ul>
 *
 * This function is asynchronous so to handle the result you will need to register a callback for the event IMSInterface.events.imsSSOStatus
 * The event data is a JSON string with format {"status":"0","details":{}} where the details could contain the results of:
 * <ul>
 *   <li>FetchAccessToken request if attempting this step in the process fails.</li>
 *   <li>FetchContinueToken request if this step failed or you passed false to openBrowser</li>
 *   <li>A status of IMSInterface.status.IMSMANAGER_SUCCESS_BROWSER_OPENED if you passed true for openBrowser</li>
 * </ul>
 *
 * @since 6.1.0
 * @see addEventListener()
 *
 * @param openBrowser        Boolean value to determine if the SSO token should be opened in the browser. Default is false.
 * @param url                The URL to jump to. It should be URL encoded.
 * @param imsRef             (optional) An IMS reference returned from the \c IMSConnect() call.
 * @param clientID           (optional) The IMS client ID for your extension, assigned by the IMS team on registration.
 * @param clientSecret       (optional) The client secret associated with the provided client ID
 * @param scope              (optional) A comma delimited list of services for which the refresh and access tokens are requested, for example 'openid,AdobeID,browserlab'.
 * @param userAccountGuid    (optional) The unique person GUID for the Renga account, as returned by
 *                           the <code>fetchAccounts()</code>, <code>imsGetCurrentUserId</code> or <code>imsGetCurrentUserIdHelper</code> methods. If you do not pass a userAccountGuid but 
 *                           openBrowser is true, the Jump URL will be opened without attempting SSO workflows.
 * @param targetClientId     (optional) The client ID needed for the destination.
 * @param targetScope        (optional) A comma delimited list of services for which the refresh token should be authorized to use to ensure the user can successful sign-in to the service.  
 * @param targetResponseType (optional) The IMS integration type the target has.
 * @param targetLocale       (optional) The locale in the format language_country, for example "en_US".
 *
 * @return A boolean status. Returning false means that we failed to trigger the SSO workflows. This could be caused by invalid parameters.
 */
IMSInterface.prototype.imsAttemptSSOJumpWorkflows = function(openBrowser, url, imsRef, clientID, clientSecret, scope, userAccountGuid, targetClientId, targetScope, targetResponseType, targetLocale)
{
    
    if (!openBrowser) {
        openBrowser = false;
    }
    if ((clientID === undefined) || (clientID === null)) {
        clientID = "";
    }
    if ((clientSecret === undefined) || (clientSecret === null)) {
        clientSecret = "";
    }
    if ((scope === undefined) || (scope === null)) {
        scope = "";
    }
    if ((targetClientId === undefined) || (targetClientId === null)) {
        targetClientId = "";
    }
    if ((targetScope === undefined) || (targetScope === null)) {
        targetScope = "";
    }
    if ((targetResponseType === undefined) || (targetResponseType === null)) {
        targetResponseType = "";
    }
    if ((imsRef === undefined) || (imsRef === null)) {
        imsRef = "";
    }
    if ((userAccountGuid === undefined) || (userAccountGuid === null)) {
        userAccountGuid = "";
    }

    var params = {};
    params.imsRef = imsRef;
    params.openBrowser = openBrowser;
    params.url = url;
    params.clientId = clientID;
    params.clientSecret = clientSecret;
    params.scope = scope;
    params.userAccountGuid = userAccountGuid;
    params.targetClientId = targetClientId;
    params.targetScope = targetScope;
    params.targetResponseType = targetResponseType;
    params.targetLocale = targetLocale;

    return JSON.parse(window.__adobe_cep__.invokeSync("imsAttemptSSOJumpWorkflows", JSON.stringify(params))).result;
};

IMSInterface.events = {
    imsFetchAccessTokenWithStatus      : "com.adobe.csxs.events.internal.ims.FetchAccessTokenWithStatus",
    imsFetchAccessToken                : "com.adobe.csxs.events.internal.ims.FetchAccessToken",
    imsRevokeAccessToken               : "com.adobe.csxs.events.internal.ims.FetchAccessTokenWithStatus",
    imsFetchContinueToken              : "com.adobe.csxs.events.internal.ims.FetchAccessTokenWithStatus",
    imsAAMIMSStatus                    : "vulcan.SuiteMessage.com.adobe.aam.AAMIMSStatus",
    imsLogoutUser                      : "com.adobe.csxs.events.internal.ims.LogoutUser",
    imsSSOStatus                       : "com.adobe.csxs.events.internal.ims.SSOStatus"
};

IMSInterface.status = {
    IMS_SUCCESS                                     : "0", /**< General Success status.*/
    IMS_ERROR_FAILURE                               : "1",/**< General failure status.*/
    IMS_ERROR_INVALID_ARGUMENTS                     : "2", /**< Error returned if invalid arguments are passed to an API.*/
    IMS_ERROR_CANCEL                                : "20", /**< Error returned if the Http request is cancelled.*/
    IMS_ERROR_TIMEOUT                               : "21",/**< Error returned if the Http request is timed out.*/
    IMS_ERROR_HTTPFAILURE                           : "22", /**< Error returned if http request fails.*/
    IMS_ERROR_SSLFAILURE                            : "23",/**< Error returned if http request fails, due to SSL certifiaction error.*/
    IMS_ERROR_AUTH_PROXY_REQUIRED                   : "24", /**< Error returned if http request is being made behind proxy.*/
    IMS_ERROR_AUTH_PROXY_FAILED                     : "25", /**< Error returned if http request failed due to wrong proxy credentials.*/
    IMS_ERROR_IN_ACCESS_IDP                         : "26", /**< Error returned if IMSLib fails in accessing IDP server. Probably server is down.*/
    IMS_ERROR_ANOTHER_REQUEST_IN_PROCESS            : "40", /**< Error returned if async call to IMSLib is made when one async http request is already in progress.*/
    IMS_ERROR_IN_READ_USER_DATA                     : "60", /**< Error returned if IMSLib fails in reading user specific data from the user machine.*/
    IMS_ERROR_IN_SAVE_USER_DATA                     : "61", /**< Error returned if IMSLib fails in saving user specific data on the user machine.*/
    IMS_ERROR_IN_REMOVE_USER_DATA                   : "62", /**< Error returned if IMSLib fails in removing user specific data from the user machine.*/
    IMS_ERROR_USER_DATA_NOT_PRESENT                 : "63", /**< Error returned if IMSLib detects that user data is not present on the user machine.*/
    IMS_ERROR_IN_READ_DEVICE_TOKEN                  : "64", /**< Error returned if IMSLib fails in reading device token from the user machine.*/
    IMS_ERROR_IN_SAVE_DEVICE_TOKEN                  : "65", /**< Error returned if IMSLib fails in saving device token on the user machine.*/
    IMS_ERROR_IN_REMOVE_DEVICE_TOKEN                : "66", /**< Error returned if IMSLib fails in removing device token from the user machine.*/
    IMS_ERROR_DEVICE_TOKEN_NOT_PRESENT              : "67", /**< Error returned if device token is not present on the user machine.*/
    IMS_ERROR_INVALID_DEVICE_TOKEN                  : "68", /**< Error returned if invalid device token is present on the user machine.*/
    IMS_ERROR_CLIENTID_NOT_PRESENT                  : "69", /**< Error returned, if client id information is missing on the user machine.*/
    IMS_ERROR_IN_FETCH_USER_ACCOUNTS                : "70", /**< Error returned, if IMSLib fails in fetching user accounts details from the user machine.*/
    IMS_ERROR_IN_SAVE_USER_FOR_CLIENTID             : "71", /**< Error returned, if IMSLib fails in saving user id for a particular client id.*/
    IMS_ERROR_DEVICE_ID_NOT_PRESENT                 : "72", /**< Error returned if device id is not present on the user machine.*/
    IMS_ERROR_DEFAULT_USER_FOR_CLIENTID_NOT_PRESENT : "73", /**< Error returned if there is no default user for client_id*/
    IMS_ERROR_NO_USER_RECORDS_PRESENT               : "74", /**< Error returned if there is no user records are present in opm.db*/
    IMS_ERROR_ACCESS_DENIED                         : "80", /**< Error returned if IMS server returns error code 'access_denied' while generating token.*/
    IMS_ERROR_INVALID_REQUEST                       : "81", /**< Error returned if IMS server returns error code 'invalid_request' while generating token.*/
    IMS_ERROR_INVALID_CLIENT                        : "82", /**< Error returned if IMS server returns error code 'invalid_client' while generating token.*/
    IMS_ERROR_INVALID_GRANT                         : "83", /**< Error returned if IMS server returns error code 'invalid_grant' while generating token.*/
    IMS_ERROR_UNAUTHORIZED_CLIENT                   : "84", /**< Error returned if IMS server returns error code 'unauthorized_client' while generating token.*/
    IMS_ERROR_UNSUPPORTED_RESPONSE_TYPE             : "85", /**< Error returned if IMS server returns error code 'unsupported_response_type' while generating token.*/
    IMS_ERROR_INVALID_SCOPE                         : "86", /**< Error returned if IMS server returns error code 'invalid_scope' while generating token.*/
    IMS_ERROR_UNSUPPORTED_GRANT_TYPE                : "87", /**< Error returned if IMS server returns error code 'unsupported_grant_type' while generating token.*/
    IMS_ERROR_BAD_REQUEST                           : "88", /**< Error returned if IMS server returns error code 'bad_request' while generating token.*/
    IMS_ERROR_INVALID_CREDENTIALS                   : "89", /**< Error returned if IMS server returns error code 'invalid_credentials' while generating token.*/
    IMS_ERROR_IN_GET_AUTH_DATA_FROM_IDP             : "100", /**< Error returned if IMSLib fails fetching authentication data from server.*/
    IMS_ERROR_IN_GET_DEVICE_TOKEN_FROM_IDP          : "101", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_IN_GET_REFRESH_TOKEN_FROM_IDP         : "102", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_IN_GET_ACCESS_TOKEN_FROM_IDP          : "103", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_IN_GET_PROFILE_DATA_FROM_IDP          : "104", /**< Error returned if IMSLib fails fetching device token from server.*/
    IMS_ERROR_TOU_CHANGED                           : "120", /**< Error returned if TOU has changed. */
    IMS_ERROR_IN_REVOKE_DEVICE_TOKEN                : "121", /**< Error returned if IMSLib fails in revoking device token. */
    IMS_ERROR_TOU_NOT_CURRENT                       : "122", /**< Terms of use acceptance is not current. */
    IMS_ERROR_EVS_INVALID                           : "123", /**< Email Verification Status does not meet required policy. */
    IMS_ERROR_ACCT_ACT_REQ                          : "124", /**< Some issue with the user account requires attention/action. i.e. COPPA or some other issue requiring user action, related to the user account rather than a subscription. */
    IMS_ERROR_ACCT_DISABLED                         : "125", /**< Account is currently disabled. */
    IMS_ERROR_SUBS_ACT_REQ                          : "126", /**< Some issue with the subscription requires attention/action. i.e. payment error etc. */
    IMS_ERROR_SUBS_NO_SUB                           : "127", /**< No current subscription exists for the user. i.e. closed/removed. */
    IMS_ERROR_NO_BUDDY_GROUP_FOR_CLIENT             : "150", /**< No buddy_group exists for the client queried. */
    IMS_ERROR_CLIENT_REGISTERED_FOR_OTHER_GROUP     : "151", /**< Client is already registered for some group other than requested. */
    IMS_ERROR_GROUP_ENTRY_NOT_PRESENT               : "152", /**< No entry exixts for the group queried. */
    IMS_ERROR_IN_SAVE_GROUP_DATA                    : "153", /**< Error Occured while saving the buddy-group data */
    IMS_ERROR_CNAME_ENTRY_NOT_PRESENT               : "154", /**< CName entry not present */
    IMS_ERROR_IN_SAVE_BACKOFF_DATA                  : "155", /**< Error Occured while saving the backoff data for the cname */	
    /**< For HTTP errors 401 to 600, IMSLib returns them as it is, they are not wrapped by any IMSLib errors */
    /**< CEP Errors */
    IMSMANAGER_ERROR_EXCEPTION                      : "3000", /**< Error returned when an exception is thrown during processing in the event callback */
    IMSMANAGER_ERROR_ENCODING                       : "3001", /**< Error returned if the inDetails information failed to be encoded into a Base64 string */
	IMSMANAGER_SUCCESS_BROWSER_OPENED               : "3002", /**< Status returned if CEP successfully opens the default browser with the SSO URL as part of the AttemptSSOJumpWorkflows API */
    IMSMANAGER_ERROR_BROWSER_FAILED_TO_OPEN         : "3003", /**< Status returned if CEP fails to open the SSO URL in the default browser */
    
    IMS_UNKNOWN_ERROR                               : "0xFFFF"
};

// @deprecated Use IMSInterface.events directly.
IMSInterface.prototype.events = IMSInterface.events;