﻿//
// Revit IFC Import library: this library works with Autodesk(R) Revit(R) to import IFC files.
// Copyright (C) 2013  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
//

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.IFC;
using Revit.IFC.Common.Utility;
using Revit.IFC.Common.Enums;
using Revit.IFC.Import.Enums;
using Revit.IFC.Import.Geometry;
using Revit.IFC.Import.Utility;

namespace Revit.IFC.Import.Data
{
    public class IFCMappedItem : IFCRepresentationItem
    {
        IFCCartesianTransformOperator m_MappingTarget = null;

        IFCRepresentationMap m_MappingSource = null;

        /// <summary>
        /// The transform, potentially including mirroring and non-uniform scaling.
        /// </summary>
        public IFCCartesianTransformOperator MappingTarget
        {
            get { return m_MappingTarget; }
            protected set { m_MappingTarget = value; }
        }

        /// <summary>
        /// The representation map containing the shared geometry.
        /// </summary>
        public IFCRepresentationMap MappingSource
        {
            get { return m_MappingSource; }
            protected set { m_MappingSource = value; }
        }

        protected IFCMappedItem()
        {
        }

        override protected void Process(IFCAnyHandle item)
        {
            base.Process(item);

            // We will not fail if the transform is not given, but instead assume it to be the identity.
            IFCAnyHandle mappingTarget = IFCImportHandleUtil.GetRequiredInstanceAttribute(item, "MappingTarget", false);
            if (mappingTarget != null)
                MappingTarget = IFCCartesianTransformOperator.ProcessIFCCartesianTransformOperator(mappingTarget);
            else
                MappingTarget = IFCCartesianTransformOperator.ProcessIFCCartesianTransformOperator();

            IFCAnyHandle mappingSource = IFCImportHandleUtil.GetRequiredInstanceAttribute(item, "MappingSource", false);
            if (mappingSource == null)
                return;
                
            MappingSource = IFCRepresentationMap.ProcessIFCRepresentationMap(mappingSource);
        }

        /// <summary>
        /// Create geometry for a particular representation item.
        /// </summary>
        /// <param name="shapeEditScope">The geometry creation scope.</param>
        /// <param name="lcs">Local coordinate system for the geometry.</param>
        /// <param name="forceSolid">True if we require a solid.</param>
        /// <param name="guid">The guid of an element for which represntation is being created.</param>
        protected override void CreateShapeInternal(IFCImportShapeEditScope shapeEditScope, Transform lcs, bool forceSolid, string guid)
        {
            base.CreateShapeInternal(shapeEditScope, lcs, forceSolid, guid);

            // TODO: Apply scaling.
            Transform mappingTransform = MappingTarget.Transform;
            Transform newLCS = null;
            if (lcs == null)
                newLCS = mappingTransform;
            else if (mappingTransform == null)
                newLCS = lcs;
            else
                newLCS = lcs.Multiply(mappingTransform);

            // Pass in newLCS = null, use newLCS for instance.
            if (newLCS.IsConformal)
            {
                MappingSource.CreateShape(shapeEditScope, null, guid);
                IList<GeometryObject> instances = DirectShape.CreateGeometryInstance(shapeEditScope.Document, MappingSource.Id.ToString(), newLCS);
                foreach (GeometryObject instance in instances)
                    shapeEditScope.AddGeometry(IFCSolidInfo.Create(Id, instance));
            }
            else
            {
                MappingSource.CreateShape(shapeEditScope, newLCS, guid);
            }
        }

        protected IFCMappedItem(IFCAnyHandle item)
        {
            Process(item);
        }

        /// <summary>
        /// Create an IFCMappedItem object from a handle of type IfcMappedItem.
        /// </summary>
        /// <param name="ifcMappedItem">The IFC handle.</param>
        /// <returns>The IFCMappedItem object.</returns>
        public static IFCMappedItem ProcessIFCMappedItem(IFCAnyHandle ifcMappedItem)
        {
            if (IFCAnyHandleUtil.IsNullOrHasNoValue(ifcMappedItem))
            {
                IFCImportFile.TheLog.LogNullError(IFCEntityType.IfcMappedItem);
                return null;
            }

            IFCEntity mappedItem;
            if (!IFCImportFile.TheFile.EntityMap.TryGetValue(ifcMappedItem.StepId, out mappedItem))
                mappedItem = new IFCMappedItem(ifcMappedItem);
            return (mappedItem as IFCMappedItem);
        }
    }
}
