﻿//
// Revit IFC Import library: this library works with Autodesk(R) Revit(R) to import IFC files.
// Copyright (C) 2013  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
//

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.IFC;
using Revit.IFC.Common.Utility;
using Revit.IFC.Common.Enums;
using Revit.IFC.Import.Enums;
using Revit.IFC.Import.Geometry;
using Revit.IFC.Import.Utility;

namespace Revit.IFC.Import.Data
{
    /// <summary>
    /// This should be an abstract class, as it has no geometry in it.  The subtypes represent topological loops.
    /// </summary>
    public class IFCLoop : IFCRepresentationItem
    {
        // This is generated by the subentities of IFCLoop, if needed.
        CurveLoop m_Loop = null;

        public CurveLoop Loop
        {
            get
            {
                if (m_Loop == null)
                    m_Loop = GenerateLoop();
                return m_Loop;
            }
        }

        // In the case of a tessellated shape, the IFCLoop is defined by vertexes.
        IList<XYZ> m_LoopVertexes = null;

        public IList<XYZ> LoopVertexes
        {
           get
           {
              if (m_LoopVertexes == null)
                 m_LoopVertexes = GenerateLoopVertexes();
              return m_LoopVertexes;
           }
        }

        protected IFCLoop()
        {
        }

        override protected void Process(IFCAnyHandle ifcLoop)
        {
            base.Process(ifcLoop);
        }

        virtual protected CurveLoop GenerateLoop()
        {
            return null;
        }

        virtual protected IList<XYZ> GenerateLoopVertexes()
        {
           return null;
        }

       /// <summary>
        /// Create geometry for a particular representation item.
        /// </summary>
        /// <param name="shapeEditScope">The geometry creation scope.</param>
        /// <param name="lcs">Local coordinate system for the geometry.</param>
        /// <param name="forceSolid">True if we require a solid.</param>
        /// <param name="guid">The guid of an element for which represntation is being created.</param>
        protected override void CreateShapeInternal(IFCImportShapeEditScope shapeEditScope, Transform lcs, bool forceSolid, string guid)
        {
            base.CreateShapeInternal(shapeEditScope, lcs, forceSolid, guid);
        }

        protected IFCLoop(IFCAnyHandle ifcLoop)
        {
            Process(ifcLoop);
        }

        /// <summary>
        /// Create an IFCLoop object from a handle of type IfcLoop.
        /// </summary>
        /// <param name="ifcLoop">The IFC handle.</param>
        /// <returns>The IFCLoop object.</returns>
        public static IFCLoop ProcessIFCLoop(IFCAnyHandle ifcLoop)
        {
            if (IFCAnyHandleUtil.IsNullOrHasNoValue(ifcLoop))
            {
                IFCImportFile.TheLog.LogNullError(IFCEntityType.IfcLoop);
                return null;
            }

            IFCEntity loop;
            if (IFCImportFile.TheFile.EntityMap.TryGetValue(ifcLoop.StepId, out loop))
                return (loop as IFCLoop);
            
            if (IFCAnyHandleUtil.IsSubTypeOf(ifcLoop, IFCEntityType.IfcPolyLoop))
                return IFCPolyLoop.ProcessIFCPolyLoop(ifcLoop);

            IFCImportFile.TheLog.LogUnhandledSubTypeError(ifcLoop, IFCEntityType.IfcLoop, false);
            return null;
        }
    }
}
