import QtQuick 2.0
import com.autodesk.toolclips.qml 1.0
import "../Scripts/TCUtils.js" as TCUtils
import QtQuick.Layouts 1.2


/** \file ToolClipUI.qml

ToolClipUI.qml is the root UI for the full ToolClip user interface. 

*/

TCStyledRectangle {
    id : root
    property int borderMargin : 0 // TCUtils.dpiScale(5)
    property int borderTopMargin : borderMargin + ((arrowPos === Qt.TopEdge) ? arrowSize : 0)
    property int borderBottomMargin : borderMargin + ((arrowPos === Qt.BottomEdge) ? arrowSize : 0)
    property int borderLeftMargin : borderMargin + ((arrowPos === Qt.LeftEdge) ? arrowSize : 0)
    property int borderRightMargin : borderMargin + ((arrowPos === Qt.RightEdge) ? arrowSize : 0)
    property int contentHorzMargin : TCUtils.dpiScale(8)
    property int contentVertMargin : TCUtils.dpiScale(5)
    property int columnSpacing : TCUtils.dpiScale(4)
    property int shortcutSpacing : TCUtils.dpiScale(10)
    toolTipStyle : root.state !== "Level_3"
    arrowSize: ((targetRect.width > 0) && (targetRect.height > 0)) ? TCUtils.dpiScale(8) : 0
    radius : TCUtils.dpiScale(3)
    shadowAreaSize : TCUtils.dpiScale(6)
    shadowOffsetX: TCUtils.dpiFloatScale(3)
    shadowOffsetY: TCUtils.dpiFloatScale(3)
    width: (root.state !== "Level_3") ? (layout.width + borderLeftMargin + borderRightMargin + shadowAreaSize * 2) : layout.width
    height: (root.state !== "Level_3") ? (layout.height + borderTopMargin + borderBottomMargin + shadowAreaSize * 2) : layout.height

    TCPalette {
        id: hoverPal
        colorGroup: TCPalette.Hover
    }

    Connections {
        target: toolClipView
        onToolClipRestarted: {
            root.state = "Level_1"

            if ( toolclip.hasExtendedContent() )
            {
                // Restart the Level 2 waiting timer.
                timer_level_two.restart()
                // Restart the Level 2 waiting gif animation.
                waiting4level2_ani.visible = true
                waiting4level2_ani.currentFrame = 0
                waiting4level2_ani.paused = false
            }
            else
            {
                // Stop the Level 2 waiting timer.
                timer_level_two.stop()
                // Stop the Level 2 waiting gif animation.
                waiting4level2_ani.visible = false
                waiting4level2_ani.paused = true
            }
        }
        onToolClipClosed: {
            root.state = "Level_1"
            // Stop the Level 2 waiting timer.
            timer_level_two.stop()
            // Stop the Level 2 waiting gif animation.
            waiting4level2_ani.paused = true
        }
        onMouseLeftToolTipArea: {
            // Stop the Level 2 waiting timer since the tool clip will disappear anyway
            // after the short keep open time.
            timer_level_two.stop()
        }
    }

    onHeightChanged:
    {
        // when the tool clip is hosted in a window on Level 3 we sync the layout size manually here
        if ( state === "Level_3" )
        {
            layout.height = height
        }
    }

    onWidthChanged:
    {
        // when the tool clip is hosted in a window on Level 3 we sync the layout size manually here
        if ( state === "Level_3" )
        {
            layout.width = width
        }
    }

    Timer {
        id : timer_level_two
        interval: toolClipView.skipLevel1 ? 0 : 1750; running: false; repeat: false // to do api dependent
        onTriggered: root.state = "Level_2"
    }

    
ColumnLayout {
    id: layout
    x : (root.state !== "Level_3") ? (root.borderLeftMargin + shadowAreaSize) : 0
    y : (root.state !== "Level_3") ? (root.borderTopMargin + shadowAreaSize) : 0
    spacing: root.columnSpacing

    ColumnLayout {
        id: layout_constraints
        spacing: root.columnSpacing
        Layout.preferredWidth:  (root.state === "Level_2" || root.state === "Level_3") ? TCUtils.dpiScale(304) : -1
        Layout.maximumWidth: (root.state === "Level_1" || root.state === "Level_2") ? TCUtils.dpiScale(304) : -1



        // LEVEL 1 & 2 & 3 - Header (Title + Shortcut + Waiting 4 L2 Ani)
        Rectangle {
            id : header
            Layout.fillWidth: true // will expand the title and the shortcut is on the right
            Layout.preferredHeight:  title_shortcut.height
            Layout.preferredWidth: title_shortcut.getTitleWidth()
            Layout.leftMargin: root.contentHorzMargin
            Layout.rightMargin: root.contentHorzMargin + (waiting4level2_ani.visible ? waiting4level2_ani.width : 0) // we add some extra pix for the waiting animated gif
            Layout.topMargin: root.contentVertMargin
            Layout.bottomMargin: root.contentVertMargin
            color : "transparent"

            // Title & Shortcut
            TitleShortcutText {
                id: title_shortcut
                anchors.left: parent.left
                anchors.right: parent.right
            }

            // Waiting for Level 2 animation
            AnimatedImage {
                id: waiting4level2_ani
                anchors.left : parent.right
                anchors.leftMargin: (root.contentHorzMargin / 2 + width)
                width: TCUtils.dpiScale( sourceSize.width )
                height: TCUtils.dpiScale( sourceSize.height )
                smooth: true
                source: "waiting4level2.gif"
            }
        }


        // LEVEL 2 & 3 - Description Text
        DescriptionText {
            id: level2_description
            Layout.fillWidth: true
            Layout.leftMargin: root.contentHorzMargin
            Layout.rightMargin: root.contentHorzMargin
            textColor : root.textColor
            text: if ( toolclip.childElement("text") ) {
                toolClipView.interpolate( toolclip.childElement("text").content )
            } else {
                ""
            }
            visible: (root.state === "Level_2" || root.state === "Level_3") ? ((text == "") ? false : true) : false
        }

        // LEVEL 3 - Extended Description Text
        DescriptionText {
            id: level3_description
            Layout.fillWidth: true
            Layout.leftMargin: root.contentHorzMargin
            Layout.rightMargin: root.contentHorzMargin
            textColor : root.textColor
            text: if ( toolclip.childElement("expanded") && toolclip.childElement("expanded").childElement("text") ) {
                toolClipView.interpolate( toolclip.childElement("expanded").childElement("text").content )
            } else {
                ""
            }
            visible: (root.state === "Level_3") ? ((text == "") ? false : true) : false
        }

        // LEVEL 2 & 3
        MediaView {
            id: media_view
            Layout.fillWidth: true
            Layout.preferredHeight: media_stack.visible ? media_stack.height : 0
            visible: false
        }

        // LEVEL 3 - Resource panel with Video, Image and Help links
        ResourceList {
            id: resource_list
            Layout.leftMargin: root.contentHorzMargin
            Layout.rightMargin: root.contentHorzMargin
            Layout.bottomMargin: root.contentVertMargin
            Layout.fillWidth: true
            Layout.fillHeight: true
            Layout.minimumHeight: min_height
            model : if ( toolclip.childElement("expanded") ) {
                    toolclip.childElement("expanded").childElement("resourcelist").childElementList("");
                } else {
                    0
                }
            media : media_view
            visible: false

            text_color : root.textColor
        }

        // LEVEL 3 - Spacer when ResourceList is hidden on level 3, so that the rest of the UI gets pushed to the top.
        Rectangle {
            id : resource_list_spacer
            Layout.fillHeight: true
            visible: false
        }

        // LEVEL 2 - Footer Separator Line
        Rectangle {
            id: footer_line
            color: grayVal( root.backgroundColor ) > 0.5 ? Qt.darker( root.backgroundColor, 1.5 ) : Qt.lighter( root.backgroundColor, 1.5 )
            Layout.fillWidth: true
            Layout.leftMargin: root.contentHorzMargin
            Layout.rightMargin: root.contentHorzMargin
            height: 2
            visible: footer.visible
        }

        // LEVEL 2 - Footer
        Rectangle {
            id: footer
            Layout.fillWidth: true
            Layout.minimumWidth: more.width
            Layout.leftMargin: root.contentHorzMargin
            Layout.rightMargin: root.contentHorzMargin
            //Layout.topMargin: root.contentVertMargin
            Layout.bottomMargin: root.contentVertMargin
            // assume that more and help have the same sized icons.
            height : more.height /*+ TCUtils.dpiScale(4)*/
            color : "transparent"
            visible: (root.state === "Level_2") ? true : false

            LinkButton {
                id : more
                icon_source: theme() + "more.png"
                //icon_hover_source: theme() + "moreHover.png"
                anchors.verticalCenter : parent.verticalCenter
                anchors.right : parent.right
                text : toolClipView.translate(qsTr("More...", "ToolClipUI"), "ToolClipUI" )
                onClicked : root.state = 'Level_3'
            }

        }

    }

}

// convert R,G,B to gray 0..1
function grayVal(col) {
    // use the same conversion factors as Qt does
    return (col.r * 0.3437 + col.g * 0.5 + col.b * 0.1563);
}

// Returns the icon color scheme depending on the background color.
function theme() {
    return (grayVal( root.backgroundColor ) < 0.5) ? "Dark/" : "Light/";
}

// Returns the Level 2 media source, either video or an image.
function getLevel2MediaSource( ) {

    var mediaXmlElem = toolclip.childElement("video", false);
    if ( !mediaXmlElem )
        mediaXmlElem = toolclip.childElement("image", false);

    if ( mediaXmlElem ) {
        return mediaXmlElem;
    }

    return "";
}

// Returns the Level 3 media source index.
function defaultMediaIndex( ){
    var index;
    if ( toolclip.childElement("expanded")) {
        if ( toolclip.childElement("expanded").childElement("default-resource") &&
                toolclip.childElement("expanded").childElement("default-resource").attribute("index") ) {
            var result = toolclip.childElement("expanded").childElement("default-resource").attribute("index") - 1;
        } else {
            var list = toolclip.childElement("expanded").childElement("resourcelist").childElementList("");

            for ( var i = 0; i < list.length ; i++ ) {
                var tagname = list[i].tagName;
                if ( list[i].attribute("location") && (tagname=="video" || tagname=="image") ) {
                    return i;
                }
            }
        }
    }
    return 0;
}

// Returns the Level 3 media source.
function defaultMediaSource( ) {
    if ( toolclip.childElement("expanded") && toolclip.childElement("expanded").childElement("resourcelist") ) {
        var index = defaultMediaIndex();
        var result = toolclip.childElement("expanded").childElement("resourcelist").childElementList("")[index];
        return result;
    }
    else
    {
        return getLevel2MediaSource();
    }
}

states: [

    State {
        name: "Level_1"
        PropertyChanges {
            target: waiting4level2_ani
            visible: true
            paused: false
        }
        PropertyChanges {
            target: media_view
            videoState: 0
            visible: false
        }
        PropertyChanges {
            target: resource_list
            visible: false
        }
        PropertyChanges {
            target : toolClipView
            stayOpen : false
            reserved : false
        }
    },

    State {
        name: "Level_2"
        PropertyChanges {
            target: waiting4level2_ani
            visible: false
            paused: true
        }
        PropertyChanges {
            target: media_view
            videoState: 0
            visible: ( toolclip.childElement("video", false) || toolclip.childElement("image", false) ) ? true : false
        }
        PropertyChanges {
            target: resource_list
            visible: false
        }
        PropertyChanges {
            target : toolClipView
            stayOpen : false
            reserved : false
        }
        StateChangeScript
        {
            name: "Level_2_Script"
            script:{
                var mediaXmlElem = getLevel2MediaSource()
                if ( mediaXmlElem ) {
                    media_view.source = mediaXmlElem
                }
            }
        }
    },

    State {
        name: "Level_3"
        PropertyChanges {
            target: waiting4level2_ani
            visible: false
            paused: true
        }
        PropertyChanges {
            target: media_view
            visible: true
            videoState: 0
        }
        PropertyChanges {
            target: resource_list
            visible: (toolclip.childElement("expanded") && toolclip.childElement("expanded").childElement("resourcelist")) ? true : false
        }
        PropertyChanges {
            target : toolClipView
            stayOpen : true // set to true to hold expanded window open indefinitely
            reserved : true
        }
        PropertyChanges {
            target : media_view
            source : defaultMediaSource()
        }
        PropertyChanges {
            target : resource_list
            current_index : defaultMediaIndex()
        }
        PropertyChanges {
            target: resource_list_spacer
            visible: (toolclip.childElement("expanded") && toolclip.childElement("expanded").childElement("resourcelist")) ? false : true
        }
    }

]

}
