-- This manipulator creates a slider helper object that manipulates
-- itself.  

-- Written by Scott Morrison, July 25, 2000
-- Based on Peter Watje's 2d slider object

-- The slider uses a param block to store the slider value.  
-- This can be used in conjunction with parameter wiring to control
-- other objects in the scene.

plugin simpleManipulator sliderManipulator
    name:~NAME_SLIDER_CAPTION~
    classID:#(0x47db14ef, 0x4e9b5990)
    category:~MANIPULATORS_CATEGORY~
(
    -- Set the yellow and red colors for the gizmo
    local greenColor =  colorMan.getColor #manipulatorsActive
    local redColor =    colorMan.getColor #manipulatorsSelected
    local yellowColor = colorMan.getColor #manipulatorsInactive

    -- sliderOffset is the distance in pixels from the "+" to the slider bar
    local sliderOffset = 9

    -- textYOffset is the vertical distance from the slider to the text
    local textYOffset = 10

    -- textXOffset is the horizantal distance from the slider to the text
    local textXOffset = -5

    -- The minimum width a slider can have in pixels
    local minWidth = 10


    parameters main rollout:paramRollout
    ( 
         -- The value of the slider
         value      type:#float   animatable:true  ui:spnValue  default: 0.0
		 on value set val do
		 (
			-- Adjust the min and max accordingly
         	if ((val < this.minVal)) then this.minVal = val
			if ((val > this.maxVal)) then this.maxVal = val
	   	 )

         -- The min and max values for the slider
         minVal     type:#float   animatable:true  ui:spnMin    default: 0.0
         maxVal     type:#float   animatable:true  ui:spnMax    default: 100.0
 		 on minVal set val do
		 (
         	if ((val > this.maxVal)) then this.minVal = this.maxVal
			if (this.value < val) then this.value = val
		 )
		 on maxVal set val do
         (
		 	if ((val < this.minVal)) then this.maxVal = this.minVal
			if (this.value > val) then this.value = val
		 )

         -- The slider position as a percentage of screen space
         xPos       type:#float   animatable:false ui:spnXPos   default: 0.0
         yPos       type:#float   animatable:false ui:spnYPos   default: 0.0

         -- The width of the slider in pixels
         width      type:#float   animatable:false ui:spnWidth  default: 100.0

         -- When the hide flag is true, the slider bar and value are not draw
         hide       type:#boolean animatable:false ui:checkHide default: false

         -- The display name of the slider
         sldName   type:#string  animatable:false ui:labelEdit default: ""

        -- Snap toggle
         snapToggle type:#boolean animatable:false ui:checkSnap default: true

         -- snap value
         snapVal    type:#float   animatable:false  ui:spnSnapVal default: 0.01
    )

    -- The rollout for the slider parameters
    rollout paramRollout ~PARAMROLLOUT_PARAMETERS_CATEGORY~
    (
        Edittext labelEdit   ~LABELEDIT_LABEL_CAPTION~        text:""

       	Spinner spnValue ~SPNVALUE_CAPTION~ range:[-1.0e30, 1.0e30, 0.0] align:~SPNVALUE_ALIGN~ fieldwidth:~SPNVALUE_FIELDWIDTH~
       	Spinner spnMin ~SPNMIN_MINIMUM_CAPTION~ range:[-1.0e30, 1.0e30, 0.0] align:~SPNMIN_MINIMUM_ALIGN~ fieldwidth:~SPNMIN_MINIMUM_FIELDWIDTH~
       	Spinner spnMax ~SPNMAX_MAXIMUM_CAPTION~ range:[-1.0e30, 1.0e30, 100.0] align:~SPNMAX_MAXIMUM_ALIGN~ fieldwidth:~SPNMAX_MAXIMUM_FIELDWIDTH~
       	Spinner spnXPos ~SPNXPOS_CAPTION~ range:[0.0, 1.0, 0.0] scale:0.01 align:~SPNXPOS_ALIGN~ fieldwidth:~SPNXPOS_FIELDWIDTH~
       	Spinner spnYPos ~SPNYPOS_CAPTION~ range:[0.0, 1.0, 0.0] scale:0.01 align:~SPNYPOS_ALIGN~ fieldwidth:~SPNYPOS_FIELDWIDTH~
       	Spinner spnWidth ~SPNWIDTH_CAPTION~ range:[10.0, 1000.0, 100.0] align:~SPNWIDTH_CALIGN~ fieldwidth:~SPNWIDTH_FIELDWIDTH~
       	Checkbox checkSnap ~CHECKSNAP_CAPTION~ checked:false align:~CHECKSNAP_ALIGN~
       	Spinner spnSnapVal ~SPNSNAPVAL_CAPTION~ range:[0.0, 10000.0, 0.1] align:~SPNSNAPVAL_ALIGN~ fieldwidth:~SPNSNAPVAL_FIELDWIDTH~
        Checkbox checkHide  ~CHECKHIDE_CAPTION~        checked:false
    )

    -- This manipualtor manipulates itself
    on canManipulate target  return (classOf target) == sliderManipulator

    -- Creation mouse procedure
    tool create
    (
        on mousePoint click do
           case click of
           (
              1: 
              (
                -- Compute the percentage of of the mouse point in screen space
                this.xPos = viewPoint.x / gw.getWinSizeX()
                this.yPos = viewPoint.y / gw.getWinSizeY()

                #stop 
              )
           )

    )

    -- Create the manipulator gizmos.
    on updateGizmos do
    (
        -- Clear the current gizmo cache
        this.clearGizmos()

        -- If this is not a stand-alone manip, get values from the manip target
        
        local lsldName = this.sldName
        
        if (target != undefined) then 
        (
           this.value      = target.value
           this.minVal     = target.minVal
           this.maxVal     = target.maxVal
           this.xPos       = target.xPos
           this.yPos       = target.yPos
           this.width      = target.width
           this.hide       = target.hide
           this.snapToggle = target.snapToggle
           this.snapVal    = target.snapVal
           lsldName        = target.sldName
           unselColor  = greenColor
        ) 
        else
        (
           unselColor  = yellowColor
        )


        -- Compute the origin of the slider in screen space
        local x = xPos * gw.getWinSizeX()
        local y = yPos * gw.getWinSizeY()

        -- These flags are used for all the gizmos
        local flags = (gizmoUseScreenSpace + gizmoActiveViewportOnly)

        -- Gizmo 0
        -- This is the box that you use the move the slider
        local pos = [x, y, 0.0]
        this.addGizmoMarker #hollowBox pos flags unselColor redColor

        -- Gizmo 1
        -- This is the plus-sign used to toggle the hide flag
        pos =  [x + sliderOffset, y, 0.0]
        this.addGizmoMarker #plusSign pos flags unselColor redColor

        -- Compute the text label
        local str
        local relVal = value
		-- always draw the slider value within the min and max bounds
		if relVal < minVal do relVal = minVal
		if relVal > maxVal do relVal = maxVal

        if (hide) then str = lsldName
        else 
        (
			if lsldName.count > 0 then str = lsldName + ": " + (relVal as string)
            else str = relVal as string
        )

        local textLoc = [x + textXOffset, y - textYOffset, 0.0f]

        -- Gizmo 2
        -- The text with the name and value
        this.addGizmoText str textLoc flags unselColor redColor

        -- If the slider is hidden, don't draw any more
        if not hide then 
		(
	        -- Draw the slider bar
	        local barStart = x + 2 * sliderOffset
	        local barPos = [barStart, y, 0.0]
	        local barEnd = [barStart + this.width, y, 0.0]
	 
	        -- Create a polyline gizmo
	        local giz = manip.makeGizmoShape()
	        giz.AddPoint barPos
	        giz.AddPoint barEnd
	 
	        -- Gizmo 3
	        -- The slider bar
	        this.addGizmoShape giz flags unselColor unselColor
	        
	        -- Draw the slider "thumb" gizmo
			relVal = (relVal - minVal) / (maxVal - minVal)
	        local valPos = [barStart + relVal * width, y + 10, 0.0f]
	
	        -- Gizmo 4
	        -- The slider thumb triangle
	        this.addGizmoMarker #triangle valPos flags unselColor redColor
	
	        local resizePos = [barStart + width + sliderOffset, y, 0.0f]
	        
	        -- Gizmo 5
	        -- The slider resize diamond
	        this.addGizmoMarker #diamond resizePos flags unselColor redColor
		)
        -- return an empty string so no tool tip is displayed
        return ""
    )

    -- mouseMove is called on every mouse move when dragging the manip
    -- It needs to convert the mouse position 'm' into a new value for the slider
    on mouseMove m which do
    (
        local xSize = gw.getWinSizeX()
        local ySize = gw.getWinSizeY()

        if (which == 0)  then 
        (
            -- Gizmo 0

            -- Set the slider postition
            target.xPos = m.x / xSize
            target.yPos = m.y / ySize
        ) 
        else if (which == 4) then 
        (
            -- Gizmo 4

            -- Set the slider value
            local x = xPos * xSize
            local barStart = x + 2.0 * (sliderOffset as float)
            local newVal = ((m.x as float) - barStart) / width
			newVal = minVal + newVal * (maxVal - minVal)
		
            -- Clamp the value
            if (newVal < minVal) then newVal = minVal
            if (newVal > maxVal) then newVal = maxVal

            -- Snap, if needed
            if (snapToggle) then 
				if (snapVal != 0) then newVal = newVal - (mod newVal snapVal)

            -- Set the value in the target
            target.value = newVal
        ) 
        else if (which == 5) then 
        (
           -- Gizmo 4

            -- Set the slider width
            local x = xPos * xSize
            local barStart = x + 2.0 * sliderOffset
            local newWidth = m.x - sliderOffset - barStart

            -- Clamp the width
            if (newWidth < minWidth) then newWidth = 10.0

            -- Set the width in the target
            target.width = newWidth
        )
    )

    on mouseDown m which do
    (
      -- Toggle the value of the "Hide" state
      if (which == 1) then target.hide = not hide 
    )
)


-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEA5dK+WnG5bDemPmWVSBRBgw
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTcwODA0MDAwMDAwWhcNMTgwODA0MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBALPR50hy1FkrWOBmP+sGXfKWFUpFAKB9OLDlN3Uj
-- 94WBLdHje+wsBav/AOL1Ben4qOa74PWpJHTJd8jph4MSGhKZE3oFNPyAVXCVhUAj
-- qlLaYQXkHDWMeyz+y7FWX4oK1B1H+SNVcnc2+kAB0bEIT4VAIvQcyva41ThpVGzP
-- XZM/JKDDpA6tocMQ3935UAjHYuvoOADEkFt5O/lEWzPTz0aQkVLGiD18rgFxuSw+
-- Uz2jyuDZZ5lyNBQRF+K4cu8fle9uL2WqbaO7koHz76dkJrNW9wAmkdGCdfj3MQo+
-- OD4O5JjSMYHEcmjVbHyo+ZK/BIVykApxc0tfN2HRJSuHlG0CAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBALfg
-- FRNU3/Np7SJ5TRs8s8tPnOTd4D5We+stLCuQ0I1kjVIyiIY+Z3cQz2AB9x8VXuYF
-- LcXnT6Rc1cMYJtlTyB7Z7EZkfxQmFgc4chVfnguTpPqUtfo3QMT/S1+QIdYfIbk1
-- dSvFBmZwRGatmGbn2h7HGiIgNqQaO6TD7Fx9TEJPwIiiCK8F3b4ENpYQHlgH3OAd
-- CRLa1IWPfeA03yF3PIq8+NhLsngw1FNm9+C6UOM3mf3jHwxTrbt4ooIZstjPA4PU
-- G16FkhJg7l2RCDR6sE9iT7FMCsO6tAHX3pS8afFyNyEVfgJVKfzohdDOj+XQLkzp
-- c9v3Xoh1gTIPCte7VPsxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQDl0r5acblsN6Y+ZZVIFEGDANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQBnBKz9Fj45asH6HOyZFgkaBhZumXi5kps/
-- FD3sw3AcbFldAdkkx79/pmEzZO21SORZ8DdKG4gokm0HXuEmDTG93OmYvav47FEs
-- 4OvyghTrhLUAh6HOTIZthSJuUgUTMXdNsjOVB582N03H9ohAbaA36Xp2H/eIQh6a
-- ZhYXvtqiwDFyw9JoJ39UOV77prVPY86wsUgavM3q0XNBJaq9Z+B10CoNZxe9Z0vf
-- UDALLTitrEwU5eaIYdIGJRME8I3ovb4XecRG+C+/d1dXBq0omEkwOfRAZ8cCRsc0
-- ea0FrgaNfNJD6j+biSceglZR0nFlRBT+r1BonbjtF43AKLGI4aoY
-- -----END-SIGNATURE-----