/**********************************************************************
*<
FILE: 		    NET_TreeViewWrapper.ms

DESCRIPTION:    A wrapper for the .NET System.Windows.Forms.TreeView control

CREATED BY:     Chris P. Johnson
		
HISTORY:        Created April 18 2006

*>	Copyright Autodesdk (c) 2000, All Rights Reserved.
**********************************************************************/

struct TreeViewWraps
(
	m_dnColor = dotNetClass "System.Drawing.Color",
	m_iconClassType = dotnet.gettype "System.Drawing.Icon",
	m_bitmapClassType = dotnet.gettype "System.Drawing.Bitmap",
	m_bStyle    = dotNetClass "System.Windows.Forms.BorderStyle",

	function MXSColor_to_dotNetColor hColor = --New function
	(
		--[brief] Converts a maxscript color to a .NET color
		--[param] The maxscript color to convert
		--[return] A System.Drawing.Color object
		m_dnColor.fromARGB hColor.r hColor.g hColor.b
	),
	function IsIconFile stringFileName =
	(
		--[brief] Tells whether the filename is an icon or not.
		--[param] stringFileName - A maxscript string of the filename to query.
		--[return] true if the file is an icon (*.ico) file.
		local type = GetFileNameType stringFileName
		local result = false
		if type == ".ico" do ( result = true )
		result	
	),
	function InitImageList tv              \
					iconArray \
					pSize:0         \
					pTransparentColor:undefined =
	(
		--[brief] Add images to the treeNodes in a TreeView control. An imagelist
		--		is a structure that manages images for a control.
		--[param] tv - The System.Windows.Forms.TreeView class object
		--[param] iconArray - A maxscript array of string filenames for the images
		--		you want to add, System.Drawing.Icon instances, or System.Drawing.Bitmap instances
		--[param] pSize - The pixel dimension of the images you want to add. For instance passing
		--		in 16 will create 16x16 pixel images on your treenodes.
		--[param] pTransparentColor - The color to treat as transparent. The transparent
 		--		color is not rendered when the image is drawn
		--[return]The imagelist that was successfully created.
		--[Pre] 	Must pass the string file names as a symbolic pathname
		--     	i.e. "$maxSysIcons/mybitmap.bmp"
		--[Note] 	Icons and bitmaps are handled differently in .NET. This method
		--		only adds icons and bitmaps, to an imagelist.
		local hImgList = dotNetObject "System.Windows.Forms.ImageList"
		
		local image_size = 16 
		if pSize != 0 and (pSize > 2) then
			image_size = pSize 
		local imgSize 	= dotNetObject "System.Drawing.Size" image_size image_size
		hImgList.imagesize = imgSize

		if pTransparentColor != undefined and (classof pTransparentColor == color) then
		(	
			hImgList.transparentColor = m_dnColor.fromARGB pTransparentColor.r pTransparentColor.g pTransparentColor.b
		) 
		else
		(
			hImgList.transparentColor = m_dnColor.fromARGB 125 125 125	
		)
		
		for item in iconArray do
		(
			if ((classof item) == dotNetObject) then
			(
				local itemType = item.GetType()
				if (itemType == m_iconClassType or itemType == m_bitmapClassType) then
					hImgList.images.add item
				else
					throw "bad iconArray item type"
			)
			else
			(
				fName = (symbolicPaths.expandFileName item)
				if (IsIconFile fName ) then --Add icons
				(
					local icon  = dotNetObject "System.Drawing.Icon" fName
					hImgList.images.add icon
				)
				else --or Add bitmaps
				(
					local dnBitmap = dotNetObject "System.Drawing.Bitmap" fName
					hImgList.images.add dnBitmap
				)
			)
		)
		tv.ImageList = hImgList
	),
	---------------------------------------
	-- Misc Methods
	---------------------------------------
	function getHitNode arg tv =
	(
		--[brief] Finds the TreeNode that was hit, given a point
		--[param] arg - System.Windows.Forms.MouseEventArgs
		--[param] tv - The TreeView object that was hit
		--[return] Returns an object of the System.Windows.Forms.TreeNode class
		local position = dotNetObject "System.Drawing.Point" arg.x arg.y
		tv.GetNodeAt position
	),
	function ClearTvNodes tv =
	(
		--[brief] Deletes all treenodes in the treeview control.
		tv.nodes.clear()	
	),
	function CollapseAll tv =
	(
		--[brief] Collapses all treenodes in the treeview control.
		tv.collapseAll()
	),
	function ExpandAll tv =
	(
		--[brief] Expands all treenodes in the treeview control.
		tv.ExpandAll()	
	),
	function GetHitCoordinates tv =
	(
		--[brief] Use this method to get the hit coordinates on the treeview.
		--[param] tv - A System.Windows.Forms.TreeNode class object
		--[return] Returns the coordinates of the mouse as a .NET System.Drawing.Point object
		local screenHitPoint = tv.MousePosition
		tv.PointToClient screenHitPoint
	),
	---------------------------------------
	-- The Grand Initialization Method
	---------------------------------------
	function InitTreeView tv                 \
						pLabelEdit:    true  \
						pCheckBoxes:   false \ 
						pFullRowSel:   false \
						pAllowDrop:    false \
						pScrollable:   true  \
						pShowLines:    true  =
	(
		--[brief] Initialization Function to set general properties of a listview control
		------------------------------------
		-- user defined options
		------------------------------------
		-- Allow the user to edit item text.
		tv.LabelEdit     = pLabelEdit;
		-- Display check boxes
		tv.CheckBoxes    = pCheckBoxes;
		-- Select the item and subitems when selection is made.
		tv.FullRowSelect = pFullRowSel;
		-- Set Drag and Drop
		tv.allowDrop     = pAllowDrop
		-- Sort the items in the list in ascending order.
		tv.Sorted        = true
		-- Whether to display scroll bars.
		tv.scrollable    = pScrollable
		-- Show lines or not
		tv.showLines     = pShowLines
		
		tv.hideSelection 	= false
		
		tv.BorderStyle 	= m_bStyle.Fixed3D
		------------------------------------
		-- other options
		------------------------------------	
		-- Set Listview to System Colors
		local textColor   = MXSColor_to_dotNetColor (((colorman.getColor #text  )*255) as color)
		local windowColor = MXSColor_to_dotNetColor (((colorman.getColor #window)*255) as color)
			
		tv.backColor = windowColor
		tv.foreColor = textColor
	)	
)

global tvops = TreeViewWraps()
-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEA5dK+WnG5bDemPmWVSBRBgw
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTcwODA0MDAwMDAwWhcNMTgwODA0MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBALPR50hy1FkrWOBmP+sGXfKWFUpFAKB9OLDlN3Uj
-- 94WBLdHje+wsBav/AOL1Ben4qOa74PWpJHTJd8jph4MSGhKZE3oFNPyAVXCVhUAj
-- qlLaYQXkHDWMeyz+y7FWX4oK1B1H+SNVcnc2+kAB0bEIT4VAIvQcyva41ThpVGzP
-- XZM/JKDDpA6tocMQ3935UAjHYuvoOADEkFt5O/lEWzPTz0aQkVLGiD18rgFxuSw+
-- Uz2jyuDZZ5lyNBQRF+K4cu8fle9uL2WqbaO7koHz76dkJrNW9wAmkdGCdfj3MQo+
-- OD4O5JjSMYHEcmjVbHyo+ZK/BIVykApxc0tfN2HRJSuHlG0CAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBALfg
-- FRNU3/Np7SJ5TRs8s8tPnOTd4D5We+stLCuQ0I1kjVIyiIY+Z3cQz2AB9x8VXuYF
-- LcXnT6Rc1cMYJtlTyB7Z7EZkfxQmFgc4chVfnguTpPqUtfo3QMT/S1+QIdYfIbk1
-- dSvFBmZwRGatmGbn2h7HGiIgNqQaO6TD7Fx9TEJPwIiiCK8F3b4ENpYQHlgH3OAd
-- CRLa1IWPfeA03yF3PIq8+NhLsngw1FNm9+C6UOM3mf3jHwxTrbt4ooIZstjPA4PU
-- G16FkhJg7l2RCDR6sE9iT7FMCsO6tAHX3pS8afFyNyEVfgJVKfzohdDOj+XQLkzp
-- c9v3Xoh1gTIPCte7VPsxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQDl0r5acblsN6Y+ZZVIFEGDANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQBMVICFXYykc1BWV7s76N4VE3YmvhVqKh/F
-- cmUv89gwo4iryZXbTyuyOf2n93WuG5quY8CpsFemGIIWuwIjn56uAWJj1HS4FHaY
-- tnjgtgR8sXeVaxwBhID/Ja1aq1llE1sFOOFVxVzh22pJ5h4dvDJ4a+XFq+Qw/m3Y
-- f3etMK1liG9gmZ6ehEf1Oey0Hzb7VwiF3oJn1qxA1ZLB2dQ//VKa4cZr3F+WXcs8
-- uCLtS/9NEJhhEyuIUS3a/MDCOuM75I3E5Gx53DIIynHfPwnBolbe0Fzk1AxF2kiI
-- ggNOMk6W4udXGR28XuvNnSFmQlXd56KlgoM50JbGswFND6sEC5np
-- -----END-SIGNATURE-----