/*
Scripts File
Purposes:  
    
	Functions that perform Daylight Simulation Analysis-related actions.	

Revision History

	November 5, 2007: David Cunningham
	
*/

fn CreateDefaultDaylightSimulationMaterial =
(
	-- either replace this return value here, or define your 
	-- own CreateDefaultDaylightSimulationMaterial function in the
	-- Listener window
	Arch___Design__mi()  
)


fn FixInvalidMaterialsForDaylightSimulation =
(
	-- fix selected nodes with a standard Architectural and Design Material
	for i in $selection do
	(
		-- Change this material if you are not happy with the default assignment
		i.material = CreateDefaultDaylightSimulationMaterial()
		i.material.refl_weight = 0.0
		i.material.name = ~MATERIAL_NAME_LIGHTING_ANALYSIS_DEFAULT_MATERIAL~
	)
)

fn CreateValidDaylightSimulationLightCallback n = 
(
	if classof n  != TargetObject do
	(
		n.baseObject.castShadows = true
		n.shadowGenerator = raytraceShadow()
	)
)

struct DontShowAgainDialog
(
	titleStr = undefined,
	messageStr = undefined,
	retval = undefined,
	notAgain = undefined,
	default = undefined,
	
	fn DoDialog =
	(
		if GetQuietMode() then
		(
			if retval==undefined do retval = default
		)
		else if notAgain!=true do
		(
			-- HACK:  MaxScript has no general mechanism to "Don't Show This Again" dialogs.
			-- And it has no means to change the title and labels within a dialog before displaying it.
			-- So, dialog handling here is manual.
			-- To support arbitrary titles and labels, the dialog is defined as a string, and the values are printed into it.
			local rolloutTemplate =
				"rollout DontShowAgainRollout \"%\" width:564 height:202
				(
					--LOC_NOTES:localize the three message below, but don't change the unusual quote marks
					button bNo ~BNO_CAPTION~ pos:[286,144] width:90 height:24
					button bYes ~BYES_CAPTION~ pos:[188,144] width:90 height:24
					checkbox cNotAgain ~CNOTAGAIN_CAPTION~ pos:[8,176] width:360 height:15 checked:false
					label eLabel \"%\" pos:[8,10] width:532 height:130
					local retval = undefined
					local notAgain = undefined

					on bNo pressed do
					( 
						retval = false
						notAgain = cNotAgain.checked
						destroydialog DontShowAgainRollout
					)
					on bYes pressed do
					(
						retval = true
						notAgain = cNotAgain.checked
						destroydialog DontShowAgainRollout
					)
				)"
			
			-- Print title and labels into the rollout definition
			local rolloutStringStream = StringStream ""
			format rolloutTemplate (titleStr as string) (messageStr as string) to:rolloutStringStream
			
			-- Execute the rollout definition
			seek rolloutStringStream 0
			local rolloutValue = execute rolloutStringStream
			
			-- Display the dialog
			CreateDialog rolloutValue modal:true
			
			-- Get the return values
			notAgain = rolloutValue.notAgain
			retval = rolloutValue.retval
		)
		retval
	)
)

-- Checks if an appropriate exposure control is currently applied, for this light type.
-- If not, the user is prompted with the option to automatically apply an approriate exposure control.
global ValidateExposureControlForLight_Dialog = DontShowAgainDialog()

fn ValidateExposureControlForLight lightClass = 
(

	local isDaylight = (lightClass == Daylight)
	local isPhotometric = ((lightClass == Target_Light) or (lightClass == Free_Light))

	local isMentalRay = (classof renderers.current == mental_ray_renderer)
	local isIray = (classof renderers.current == mental_ray_iray_renderer)
	local isScanline = (classof renderers.current == Default_Scanline_Renderer)
	local isHardwareRenderer = (classof renderers.current == Quicksilver_Hardware_Renderer)
	local isRapidRT = (classof renderers.current == ART_Renderer)

	local isApplicableLight = (isDaylight or isPhotometric)
	local isApplicableRenderer = (isMentalRay or isIray or isScanline or isHardwareRenderer or isRapidRT)
	if ((not isApplicableLight) or (not isApplicableRenderer)) do
	( -- Not creating a relvant light type, or not relevant renderer.  No changes needed.
		return()
	)
	
	local currentExpCtrl = sceneExposureControl.exposureControl
	local isNullExpCtrl = (currentExpCtrl==undefined)
	local compatibleExpCtrl = (if (isMentalRay or isIray or isHardwareRenderer) then mr_Photographic_Exposure_Control else Logarithmic_Exposure_Control)

	if(isRapidRT) then
	(
		compatibleExpCtrl = Physical_Camera_Exposure_Control
	)

	local isCompatibleExpCtrl = ((not isNullExpCtrl) and ((classof currentExpCtrl) == compatibleExpCtrl))
	local isOtherExpCtrl = ((not isNullExpCtrl) and (not isCompatibleExpCtrl))
	
		

	-- If the user agreed, update the exposure control.
	local createExposure = GetQuietMode()
	
	--always create the exposure control if we are in quiet mode.
	if (not createExposure) do
	(
		local titleStr
		local messageStr

		-- Validate the current exposure control.
		if isDaylight then
		(
			if (isMentalRay or isIray ) then
			(
				-- Don't prompt the user for changes if (1) the standard exposure control is applied and perfectly configured, or (2) the user applied their own type of exposure control.
				local isDesiredExpCtrl = (isCompatibleExpCtrl and (currentExpCtrl.exposure_value == 15))
				if (isDesiredExpCtrl or isOtherExpCtrl) do return() -- No changes needed.
				--LOC_NOTES:localize this. Use && for an ampersand character
				messageStr = ~MSGSTR_RECOMMENDED_MR_PHOTOGRAPHIC_EXPOSURE_CONTROL~ +
					~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_EE_DIALOG~				
			)
			else if isScanline then
			(
				-- Don't prompt the user for changes if (1) the standard exposure control is applied and perfectly configured, or (2) the user applied their own type of exposure control.
				local isDesiredExpCtrl = (isCompatibleExpCtrl and (currentExpCtrl.exteriordaylight == true))
				if (isDesiredExpCtrl or isOtherExpCtrl) do return() -- No changes needed
				--LOC_NOTES:localize this.  Use && for an ampersand character
				messageStr = ~MSGSTR_RECOMMENDED_USE_LOGARITHMIC_EXPOSURE_CONTROL~+
					~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_ENV_EFFECTS_DIALOG~
			)
			else if (isRapidRT  or isHardwareRenderer) then
			(
			-- Don't prompt the user for changes if (1) the standard exposure control is applied and perfectly configured, or (2) the user applied their own type of exposure control.
				local isDesiredExpCtrl = (isCompatibleExpCtrl and (currentExpCtrl.global_ev == 15))
				if (isDesiredExpCtrl or isOtherExpCtrl) do return() -- No changes needed.
				--LOC_NOTES:localize this. Use && for an ampersand character
				messageStr = ~MSGSTR_RECOMMENDED_PHYSICAL_EXPOSURE_CONTROL~ +
					~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_EE_DIALOG~		
			)
			else return() -- Not any applicable renderer
			--LOC_NOTES:localize this
			titleStr = ~TITLESTR_DAYLIGHT_SYSTEM_CREATION~
		)
		else if isPhotometric then
		(
			if not isNullExpCtrl do return () -- Not creating a Daylight, and exposure control is already applied.  No changes needed.
			if (isMentalRay or isIray) then
			(
				--LOC_NOTES:localize this. Use && for an ampersand character
				messageStr = ~MSGSTR_PHOTOMETRIC_LIGHT_RECOMMENDED_MR_PHOTOGRAPHIC_EXPOSURE_CONTROL~ +
					~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_ENV_AND_EFFECTS_DIALOG~			
			)		
			else if isScanline then
			(
				--LOC_NOTES:localize this. Use && for an ampersand character
				messageStr = ~MSGSTR_CREATING_PHOTOMETRIC_LIGHT_RECOMMENDED_LOGARITHMIC_EXPOSURE_CONTROL~ +
					~EXPOSURE_CONTROL_SETTINGS_CHANGE_NOW~			
			)
			else if (isRapidRT  or isHardwareRenderer) then
			(
				--LOC_NOTES:localize this. Use && for an ampersand character
				messageStr = ~MSGSTR_PHOTOMETRIC_LIGHT_RECOMMENDED_PHYSICAL_CAMERA_EXPOSURE_CONTROL~ +
					~EXPOSURE_CONTROL_SETTINGS_ADJUSTED_FROM_ENV_AND_EFFECTS_DIALOG~
			)
			else return() -- Not any applicable renderer
			--LOC_NOTES:localize this
			titleStr = ~TITLESTR_PHOTOMETRIC_LIGHT_CREATION~
		)
		else return() -- Not any applicable light type

		-- Checks failed.  Prompt user.
		ValidateExposureControlForLight_Dialog.titleStr = titleStr
		ValidateExposureControlForLight_Dialog.messageStr = messageStr
		ValidateExposureControlForLight_Dialog.default = false
		ValidateExposureControlForLight_Dialog.DoDialog()
		createExposure = ValidateExposureControlForLight_Dialog.retval
	)
	
	if createExposure do
	(
		-- Create a new exposure control if there is no current one
		if (isNullExpCtrl) do 
		(
			sceneExposureControl.exposureControl = compatibleExpCtrl()
			currentExpCtrl = sceneExposureControl.exposureControl
		)
		
		if (isMentalRay or isIray ) then
		(
			if isDaylight then
			(
				currentExpCtrl.exposure_mode = 0
				currentExpCtrl.exposure_value = 15
			)
			else if isPhotometric do
			(
				currentExpCtrl.exposure_mode = 0
				currentExpCtrl.exposure_value = 2
			)
		)
		if (isRapidRT  or isHardwareRenderer) then
		(
			if isDaylight then
			(
				currentExpCtrl.global_ev = 15
			)
			else if isPhotometric do
			(
				currentExpCtrl.global_ev = 6
			)
		)	
		else
		(
			if isDaylight then
				currentExpCtrl.exteriordaylight = true
			else if IsPhotometric do
				currentExpCtrl.exteriordaylight = false
		)
	)
)


-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEA5dK+WnG5bDemPmWVSBRBgw
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTcwODA0MDAwMDAwWhcNMTgwODA0MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBALPR50hy1FkrWOBmP+sGXfKWFUpFAKB9OLDlN3Uj
-- 94WBLdHje+wsBav/AOL1Ben4qOa74PWpJHTJd8jph4MSGhKZE3oFNPyAVXCVhUAj
-- qlLaYQXkHDWMeyz+y7FWX4oK1B1H+SNVcnc2+kAB0bEIT4VAIvQcyva41ThpVGzP
-- XZM/JKDDpA6tocMQ3935UAjHYuvoOADEkFt5O/lEWzPTz0aQkVLGiD18rgFxuSw+
-- Uz2jyuDZZ5lyNBQRF+K4cu8fle9uL2WqbaO7koHz76dkJrNW9wAmkdGCdfj3MQo+
-- OD4O5JjSMYHEcmjVbHyo+ZK/BIVykApxc0tfN2HRJSuHlG0CAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBALfg
-- FRNU3/Np7SJ5TRs8s8tPnOTd4D5We+stLCuQ0I1kjVIyiIY+Z3cQz2AB9x8VXuYF
-- LcXnT6Rc1cMYJtlTyB7Z7EZkfxQmFgc4chVfnguTpPqUtfo3QMT/S1+QIdYfIbk1
-- dSvFBmZwRGatmGbn2h7HGiIgNqQaO6TD7Fx9TEJPwIiiCK8F3b4ENpYQHlgH3OAd
-- CRLa1IWPfeA03yF3PIq8+NhLsngw1FNm9+C6UOM3mf3jHwxTrbt4ooIZstjPA4PU
-- G16FkhJg7l2RCDR6sE9iT7FMCsO6tAHX3pS8afFyNyEVfgJVKfzohdDOj+XQLkzp
-- c9v3Xoh1gTIPCte7VPsxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQDl0r5acblsN6Y+ZZVIFEGDANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQAwg3ismd0KnqQqHEbvTohAW7kYWhzLza4O
-- 7cZih62bwsPrALQggRb1N/nqIY70YQLBYCMGb2RnzugxkiJPyGanDx4ErJIdFh5Q
-- BGM87sDBVmX4Rr9e7yaigrnWv72enku0DB3o1320hGZQIyU7BW2D9vRBUVfTDlHW
-- oRq1tCAhZfDpCP4yxop7zuu/CvxcfyEw7Ut+HK9O2SK4CRRnG9W2BFlpq4Ls0wLu
-- sgzU5EyzNmcB9DB5nURAqQPG0Cc020IUoRWknjbIlUD3HGNdEtd0QqpfbJbbZZuP
-- tHXkFmuDxngcaoJAOsXSplg8aRSDUMRy3UUmQ06sxix5KxTYHi03
-- -----END-SIGNATURE-----