--------------------------------------------------------------------------------
-- BVH parser written in MaxScript
-- by Geoff Bolton
-- Revised by Phil
--
--------------------------------------------------------------------------------
global rImportBVH;
rollout rImportBVH ~RIMPORTBVH_CAPTION~ width:405 height:150
(
	checkbox chkAnimation ~CHKANIMATION_CAPTION~ pos:[30,24] width:~CHKANIMATION_WIDTH~ height:16 checked:true
	checkbox chkCapAnim ~CHKCAPANIM_CAPTION~ pos:[30,45] width:~CHKCAPANIM_WIDTH~ height:16 checked:true
	button btnCamFile ~BTNCAMFILE_CAPTION~ pos:[30,70] width:120 height:24	
	
	spinner spnScale ~SPNSCALE_CAPTION~ pos:[256,48] width:120 height:16 enabled:true range:[0.0001,10000,1] type:#float scale:0.1 fieldwidth:50
	spinner spnStartTime ~SPNSTARTTIME_CAPTION~ pos:[264,24] width:112 height:16 type:#integer fieldwidth:50
	spinner spnBoneSize ~SPNBONESIZE_REL_CAPTION~ pos:[224,72] width:152 height:16 range:[0,100,20] fieldwidth:50
	
	progressBar pb ~PROGRESSBAR_CAPTION~ pos:[159,108] width:144 height:24
	label lblStatus ~LBLSTATUS_CAPTION~ pos:[8,108] width:144 height:23
	button btnOk ~BTNOK_CAPTION~ pos:[312,108] width:80 height:24	
	
	fn SetStatus txt = 
	(
		lblStatus.text = txt;
	)
	
	-----------------------------------------------------------
	local AXIS_ORDER_XYZ         = 1
	local AXIS_ORDER_XZY         = 2
	local AXIS_ORDER_YZX         = 3
	local AXIS_ORDER_YXZ         = 4 
	local AXIS_ORDER_ZXY         = 5
	local AXIS_ORDER_ZYX         = 6

/*	local axis_orders = #(
	#nZXY,
	#nZYX,
	#nYXZ,
	#nUnsupported);
*/	
	local bone_types = #(
	#nNothing,
	#nRoot,
	#nJoint,
	#nEndSite);
	
	struct Skeleton (
		hierarchy     = #(),
		totalchannels = 0
	)
		
	struct SkelNode (
		label         = "",
		type          = #nNothing,
		localpos      = [0.0, 0.0, 0.0],
		wspos         = [0.0, 0.0, 0.0],
		localrot      = (matrix3 1),
		ws_tm         = (matrix3 1),
		animrot       = (matrix3 1),
		wsanimrot     = (matrix3 1),
		length        = 0.0,
		numchannels   = 0,
		channels      = #(),
		axis_order    = #ZXY,
		parent        = 0,
		children      = #(),
		maxobject     = undefined
	--	bones         = #()
	)
	
	struct SceneMotion (
		frames        = 0,
		frametime     = 0.04,
		totalchannels = 0,
		data          = #()
	)
	
	
	-- Define globals here but do not initialise - this is done by initMotionData()
	-- New skeletons are added by parseMotionDataFile()
	local ini_file = ((getDir #plugcfg_ln) + "\\CAT\\importbvh.ini");
	
	local rootnode;
	local skeletons
	local motion
	
	local catparentnode;
	local fhandle
	local filename;
	local camfile;
	-- predeclare the function as we'll need it soon
	local parsehierarchy;
	local preTransform;
	local postTransform;
	
	fn initMotionData =
	(
		skeletons = #()
		motion = SceneMotion frames:0
		
		preTransform = Matrix3 1
		postTransform = Matrix3 1
		RotateX preTransform 90
		RotateX postTransform -90
		
		OK
	)
	
	fn getSplitLine handle = (
		local line;
		-- skip over short lines
		do(
			if eof handle then return undefined;
			line = readLine handle 
		)while ((line.count==0) and (not (eof handle)))
		filterString line " \t"
	)
	
	fn sliceArray array begin num = (
		local arr = #(), end = begin + num - 1
		for i = begin to end do append arr (array[i] as number)
		arr
	)
	
	fn sliceStringArray array begin num = (
		local arr = #(), end = begin + num - 1
		for i = begin to end do append arr array[i]
		arr
	)
	--------------------------------------------------------------------------------
	-- Parse an entire BVH file, specified by 'filename' or queried from the user
	-- if not specified.  Multiple ROOT hierarchies are supported, though probably
	-- will never be encountered.  Each hierarchy tree is entered into the global
	-- array 'skeletons'.  Motion data is stored in the global struct 'motion'.
	--------------------------------------------------------------------------------
	fn parseBVHFile filename scale &loadanimation =
	(
		local line
		local split
		local i
		local importing = true;

		if (filename == undefined) do return OK
	
		fhandle = openFile filename
		if fhandle == undefined then
		(
			messagebox(~MSGBOX_UNABLE_TO_OPEN_FILE~+filename);
			return OK;
		)
		
		-- make sure the first line reads "HIERARCHY" and assume
		-- this is the criterion for a vaild BVH file.
		split = getSplitLine(fhandle)
		if split[1] != ~HIERARCHY~ then throw(~THROW_FILE_NOT_VALID_BVH~)
	
		-- parse root nodes until no more
		while (split = getSplitLine(fhandle))!=undefined and (split[1] == ~ROOT~) do
		(
			s = Skeleton totalchannels:0
			parsehierarchy s split[2] scale
			append skeletons s
			motion.totalchannels += s.totalchannels
		) 
		
		if not loadanimation then return OK;
	
		-- parse motion
		if split==undefined or split[1] != ~MOTION~ then 
		(
			MessageBox(~MSGBOX_NO_MOTION_DATA~);
			loadanimation = false;
			return OK;
		)
	
		split = getSplitLine(fhandle)
		if split[1] != ~FRAMES~ then throw(~MISSING_FRAME_COUNT_INFO~)
		motion.frames = split[2] as number
		
		split = getSplitLine(fhandle)
		if (split[1]+" "+split[2]) != ~FRAME_TIME~ then throw(~MISSING_FRAME_TIME_INFORMATION~)
		-- Based on our current framerate, calculate how far appart the BVH keys should be
		motion.frametime =  framerate / (1/(split[3] as number)) 
	
		SetStatus (~READING~ + (motion.frames as string) + ~MOTION_FRAMES~)
		for i = 1 to motion.frames do (
			if importing and keyboard.escPressed then(
				SetStatus ~ABORTING_IMPORT~
				importing = false;
			) 
			
			split = getSplitLine(fhandle)
			motion.data[i] = slicearray split 1 motion.totalchannels
		)
		close fhandle
		if importing then (
			SetStatus ~DONE~
		)
		return importing;
	)
	
	
	--------------------------------------------------------------------------------
	-- Parses a ROOT hierarchy called 'label' into 'skel'
	--------------------------------------------------------------------------------
	fn parsehierarchy skel label scale =
	(
		local split = #(~SPLIT_ROOT~, label)
		local bvh_node
		local numnodes = 0
		local parentnode = 0
		local stop = false
		local level = 0
		
		while not stop do
		(
			if split[1] == ~SPLIT_ONE_ROOT~ or split[1] == ~SPLIT_ONE_JOINT~ or
				(split[1] == ~SPLIT_ONE_END~ and split[2] == ~SPLIT_TWO_SITE~) then
			(
				--node = SkelNode label:split[2]
				skel.hierarchy[numnodes+1] = SkelNode label:split[2]
				bvh_node = skel.hierarchy[numnodes+1]
				
				if split[1] == ~IF_SPLIT_ONE_ROOT~ then bvh_node.type = #nRoot
				else if split[1] == ~ELSE_IF_SPLIT_ONE_JOINT~ then bvh_node.type = #nJoint
				else bvh_node.type = #nEndSite
				
				-- sort out parent/child shit (careful of the code ordering here!!!)
				bvh_node.parent = parentnode
				numnodes += 1
				if (bvh_node.type != #nRoot) then
					append skel.hierarchy[parentnode].children numnodes
				parentnode = numnodes
	
				-- fix node label if it is an end site
				if bvh_node.type == #nEndSite then
					bvh_node.label = skel.hierarchy[bvh_node.parent].label+~END_SITE~
				SetStatus (~SETSTATUS_NODE~+bvh_node.label)
	
				-- parse open curly brace
				split = getSplitLine(fhandle)
				if split[1] == "{" then level += 1
				else throw(~THROW_ERROR_IN_HIERARCHY_TREE~)
	
				-- parse OFFSET keyword
				split = getSplitLine(fhandle)
				if split[1] == ~SPLIT_ONE_OFFSET~ then (
				--	bvh_node.localpos = scale * (point3 (split[2] as number) -(split[4] as number) (split[3] as number))
					bvh_node.localpos = scale * ((point3 (split[2] as number) (split[3] as number) (split[4] as number)) * preTransform);
					if bvh_node.type != #nRoot then bvh_node.wspos = skel.hierarchy[bvh_node.parent].wspos + bvh_node.localpos
					else bvh_node.wspos = bvh_node.localpos
				) else throw(~THROW_NO_LOCALPOS_IN_HIERARCHY~)
	
				-- parse CHANNELS keyword
				if bvh_node.type != #nEndSite then
				(
					split = getSplitLine(fhandle)
					if split[1] == ~SPLIT_ONE_CHANNELS~ then (
						bvh_node.numchannels = split[2] as number
						bvh_node.channels = sliceStringArray split 3 bvh_node.numchannels
						
						if (bvh_node.numchannels==0) then 
						(
							bvh_node.axis_order = AXIS_ORDER_XYZ
						)else 
						if (bvh_node.numchannels==1) then 
						(
							bvh_node.axis_order = AXIS_ORDER_XYZ
						)else 
						if (bvh_node.numchannels==2 and	bvh_node.channels[1]=="Xrotation"  and	bvh_node.channels[2]=="Yrotation" )  then
						(
							bvh_node.axis_order = AXIS_ORDER_XYZ
						)else
						if (bvh_node.numchannels==2 and	bvh_node.channels[1]=="Zrotation"  and	bvh_node.channels[2]=="Yrotation" )  then
						(
							bvh_node.axis_order = AXIS_ORDER_ZYX
						)else
						if (bvh_node.numchannels==3 and	bvh_node.channels[1]=="Xrotation"  and	bvh_node.channels[2]=="Yrotation"  and	bvh_node.channels[3]=="Zrotation" ) or 
						   (bvh_node.numchannels==6 and	bvh_node.channels[4]=="Xrotation"  and	bvh_node.channels[5]=="Yrotation"  and	bvh_node.channels[6]=="Zrotation" ) then
						(
							bvh_node.axis_order = AXIS_ORDER_XYZ
						)else 
						if (bvh_node.numchannels==3 and	bvh_node.channels[1]=="Xrotation"  and	bvh_node.channels[2]=="Zrotation"  and	bvh_node.channels[3]=="Yrotation" ) or 
						   (bvh_node.numchannels==6 and	bvh_node.channels[4]=="Xrotation"  and	bvh_node.channels[5]=="Zrotation"  and	bvh_node.channels[6]=="Yrotation" ) then
						(
							bvh_node.axis_order = AXIS_ORDER_XZY
						)else 
						if (bvh_node.numchannels==3 and	bvh_node.channels[1]=="Yrotation"  and	bvh_node.channels[2]=="Xrotation"  and	bvh_node.channels[3]=="Zrotation" ) or 
						   (bvh_node.numchannels==6 and	bvh_node.channels[4]=="Yrotation"  and	bvh_node.channels[5]=="Xrotation"  and	bvh_node.channels[6]=="Zrotation" ) then
						(
							bvh_node.axis_order = AXIS_ORDER_YXZ
						)else
						if (bvh_node.numchannels==3 and	bvh_node.channels[1]=="Yrotation"  and	bvh_node.channels[2]=="Zrotation"  and	bvh_node.channels[3]=="Xrotation" ) or 
						   (bvh_node.numchannels==6 and	bvh_node.channels[4]=="Yrotation"  and	bvh_node.channels[5]=="Zrotation"  and	bvh_node.channels[6]=="Xrotation" ) then
						(
							bvh_node.axis_order = AXIS_ORDER_YZX
						)else
						if (bvh_node.numchannels==3 and	bvh_node.channels[1]=="Zrotation"  and	bvh_node.channels[2]=="Xrotation"  and	bvh_node.channels[3]=="Yrotation" ) or 
						   (bvh_node.numchannels==6 and	bvh_node.channels[4]=="Zrotation"  and	bvh_node.channels[5]=="Xrotation"  and	bvh_node.channels[6]=="Yrotation" ) then
						(
							bvh_node.axis_order = AXIS_ORDER_ZXY
						)
						else 
						if (bvh_node.numchannels==3 and	bvh_node.channels[1]=="Zrotation"  and	bvh_node.channels[2]=="Yrotation"  and	bvh_node.channels[3]=="Xrotation" ) or 
						   (bvh_node.numchannels==6 and	bvh_node.channels[4]=="Zrotation"  and	bvh_node.channels[5]=="Yrotation"  and	bvh_node.channels[6]=="Xrotation" ) then
						(
							bvh_node.axis_order = AXIS_ORDER_ZYX
						)
						else
						(
							bvh_node.axis_order = #nUnsupported;
							print (bvh_node.channels as string);
							throw(~THROW_UNSUPPORTED_AXIS_ORDER~)
						)
					) else throw(~NO_CHANNELS_IN_HIERARCHY~)
				)
			
				skel.totalchannels += bvh_node.numchannels
			)
			else if split[1] == "}" then
			(
				level -= 1
				parentnode = skel.hierarchy[parentnode].parent
				if level == 0 then stop = true
			)
			if not stop then (
				split = getSplitLine(fhandle)
			)
		)	
	)
	
	
	--------------------------------------------------------------------------------
	-- Creates joints of the skeleton in its default pose, using a cube for the
	-- root, a sphere for a joint and a sphere for an end site.
	--------------------------------------------------------------------------------
	fn createJoints skel bonesize =
	(
		for bvh_node in skel.hierarchy do (
		
			bvh_node.maxobject = box width:bonesize length:bonesize height:bonesize name:bvh_node.label
			-- set position and rotation controllers for keyframing
			bvh_node.maxobject.position.controller = bezier_position()
			bvh_node.maxobject.rotation.controller = TCB_rotation()
			
			if (bvh_node.type == #nRoot) then (
				-- attach our pelvis node to the root nod so that we can move the entire bone hierarchy easily
				attachObjects rootnode bvh_node.maxobject move:false
				bvh_node.localrot = bvh_node.ws_tm;
				bvh_node.ws_tm.translation = bvh_node.wspos;
			) else (
				local pnode = skel.hierarchy[bvh_node.parent]
				
				if ((bvh_node.type == #nJoint) and (bvh_node.children.count > 0)) then (
					local childvec = skel.hierarchy[bvh_node.children[1]].localpos
					bvh_node.length = Length(childvec)
					
					-- Calculate a rotation that will make us look at our 1st child bone
					childvec = Normalize(childvec)
					bvh_node.ws_tm = matrixFromNormal childvec
					
			/*		
					local parentvec = pnode.ws_tm.row3;
					local childvec = skel.hierarchy[bvh_node.children[1]].wspos - bvh_node.wspos;
					bvh_node.length = Length(childvec)
					childvec = normalize childvec;
					
					bvh_node.ws_tm = copy pnode.ws_tm;
					rotate bvh_node.ws_tm (Quat(angleaxis (acos (dot childvec parentvec)) (normalize (cross childvec parentvec))));
					
					local temp = (normalize (cross bvh_node.ws_tm.row3 rootnode.transform.row3))
				--	local angle = acos (dot bvh_node.ws_tm.row1 temp);
					local angle = 90 - acos (dot bvh_node.ws_tm.row1 rootnode.transform.row2);
					print angle;
					preRotateZ bvh_node.ws_tm -angle
					
			*/		
					
					bvh_node.maxobject.height = bvh_node.length
				)
				else (
					bvh_node.ws_tm = copy pnode.ws_tm;
				)
				attachObjects pnode.maxobject bvh_node.maxobject move:false	
				
				-- Calculate a new local rotation offset
				bvh_node.localrot = bvh_node.ws_tm * Inverse(pnode.ws_tm)
				
				-- Calculate a new local position offset
				bvh_node.ws_tm.translation = bvh_node.wspos;
				bvh_node.localpos = bvh_node.wspos * Inverse(pnode.ws_tm);
				
			--	print bvh_node.label
			--	print  bvh_node.wspos
			)
			
		--	bvh_node.maxobject.position.controller.value = bvh_node.localpos
		--	bvh_node.maxobject.rotation.controller.value = bvh_node.localrot
		--	bvh_node.maxobject.rotation = bvh_node.ws_tm
		--	bvh_node.maxobject.position = bvh_node.wspos
			bvh_node.maxobject.transform = bvh_node.ws_tm
		)
		OK
	)
	
	--------------------------------------------------------------------------------
	-- Animate from 'startframe' to 'endframe'
	--------------------------------------------------------------------------------
	fn CreateKeyframes quiet starttime:0 scalevalue:1.0 =
	(
		local startframe = 	0;
		local endframe =	motion.frames-1;
		
		if not quiet then(
			pb.value = 0
			SetStatus ~CREATING_KEYFRAMES~
		)
	
		for frame = startframe to endframe do (
			if keyboard.escPressed then(
				SetStatus ~SETSTATUS_ABORTING_IMPORT~
				exit;
			) 
			
			local motiondata = motion.data[frame+1]
			local datapos = 1
			for skel in skeletons do (
				for bvh_node in skel.hierarchy do (
					local position = #(0,0,0,0)   -- #(channels, x, y, z)
					local rotation = #(0,0,0,0)   -- #(order, x, y, z)
					for channel in bvh_node.channels do (
						if (datapos > motion.totalchannels) then
							throw(~TOO_MANY_CHANNELS_IN_HIERARCHY~)
							
						case channel of (
							"Xposition": (
								position[1] += 1;
								position[2] = motiondata[datapos]
								)
							"Yposition": (
								position[1] += 1;
								position[3] = motiondata[datapos]
								)
							"Zposition": (
								position[1] += 1;
								position[4] = motiondata[datapos]
								)
							"Xrotation": (
								rotation[1] += 1;
								rotation[2] = motiondata[datapos]
								)
							"Yrotation": (
								rotation[1] += 1;
								rotation[3] = motiondata[datapos]
								)
							"Zrotation": (
								rotation[1] += 1;
								rotation[4] = motiondata[datapos]
								)
						)
						datapos += 1
					)
					if (position[1] > 0) then (
					--	at time frame animate on bvh_node.maxobject.position.controller.value = Point3 position[2] position[3] position[4]
						bvh_node.wspos = scalevalue * ((Point3 position[2] position[3] position[4]) * preTransform);
					)
					if (rotation[1] > 0) then (
					
						local rot;
						-- Files that come with different axis orders can be a big problem. 
						-- we are trying to support every kind of format available, but
						-- this poses a big maths problem.
						-- A note about co-ordinate systems...
						--   BVH uses a LH system with Y-axis up
						--   Max uses a RH system with Z-axis up
						--   so I map (x,y,z) in BVH to (x,z,-y) in Max
						
						-- [PT 01/05/08] Note : converting coordinate systems...
						case bvh_node.axis_order of
						(
							--  Map (z,x,y) -> (
							
							-- \\lobby\Clients_Databases\Databases\CAT\Motion Capture\Mocap Files\www.giantstudios.com\soccer_freestyle.bvh
						--	AXIS_ORDER_XYZ:(	rot = ((eulerToQuat (eulerAngles rotation[2] -rotation[4] rotation[3]) order:AXIS_ORDER_XYZ) as matrix3)			);
							AXIS_ORDER_XYZ:(	rot = (postTransform  * ((eulerToQuat (eulerAngles rotation[4] rotation[3] rotation[2]) order:6) as matrix3) * preTransform));
						
							-- credo files
							-- \\lobby\Clients_Databases\Databases\CAT\Motion Capture\Mocap Files\www.credo-interactive.com\02 Walk and RunPOSERBVH\b_prowl_wlk.bvh
							AXIS_ORDER_XZY:(	rot = (postTransform  * ((eulerToQuat (eulerAngles rotation[3] rotation[4] rotation[2]) order:3) as matrix3) * preTransform));
							
							-- files from Irrational
							-- \\lobby\Clients_Databases\Databases\CAT\CAT3\Irrational_Games\BVH_Import_Crash\SessionD-001.bvh
						--	AXIS_ORDER_YXZ:(	rot = ((eulerToQuat (eulerAngles -rotation[4] rotation[2] rotation[3]) order:AXIS_ORDER_YXZ) as matrix3)			);
							AXIS_ORDER_YXZ:(	rot = (postTransform  * ((eulerToQuat (eulerAngles rotation[2] rotation[4] rotation[3]) order:2) as matrix3) * preTransform));
							
							-- credo files
						--	AXIS_ORDER_YZX:(	rot = ((eulerToQuat (eulerAngles -rotation[4] rotation[3] rotation[2]) order:AXIS_ORDER_YZX) as matrix3)			);
							-- \\lobby\Clients_Databases\Databases\CAT\Motion Capture\Mocap Files\www.credo-interactive.com\02 Walk and RunPOSERBVH\b_prowl_wlk.bvh
							AXIS_ORDER_YZX:(	rot = (postTransform * ((eulerToQuat (eulerAngles rotation[2] rotation[3] rotation[4]) order:1) as matrix3) * preTransform)	);
							
							
							-- CS BVH file
							-- \\lobby\Clients_Databases\Databases\CAT\Motion Capture\Geoffs old scripts\scripts from geoff's 3dsmax4 dir\talchum.bvh
						--	AXIS_ORDER_ZXY:(	rot = ((eulerToQuat (eulerAngles rotation[3] rotation[2] -rotation[4]) order:AXIS_ORDER_ZXY) as matrix3)			);
							AXIS_ORDER_ZXY:(	rot = (postTransform * ((eulerToQuat (eulerAngles rotation[3] rotation[4] rotation[2]) order:3) as matrix3) * preTransform)	);
							
							
							AXIS_ORDER_ZYX:(	rot = (postTransform * ((eulerToQuat (eulerAngles rotation[2] rotation[3] rotation[4]) order:1) as matrix3) * preTransform)	);
						)

						if (bvh_node.type != #nRoot) then (
							local pnode = skel.hierarchy[bvh_node.parent]
							-- build a world space matrix which represents our animation
							bvh_node.animrot = rot * pnode.animrot;
							-- apply the transform that aligns our bones with our chidren
							bvh_node.wsanimrot = bvh_node.ws_tm * bvh_node.animrot;
							-- subtract off our parent rotation to give us a local matrix
						--	local localrot = bvh_node.wsanimrot * Inverse(pnode.wsanimrot);
							at time (starttime + (frame * motion.frametime)) animate on(
								bvh_node.maxobject.transform = bvh_node.wsanimrot;
								bvh_node.maxobject.position.controller.value = bvh_node.localpos;
							)
						)
						else(
							bvh_node.animrot = rot
							rot.translation = bvh_node.wspos
						--	at time frame animate on bvh_node.maxobject.rotation.controller.value = rot;
							at time (starttime + (frame * motion.frametime)) animate on bvh_node.maxobject.transform = rot;
						)
					)
				)
			)
			if not quiet then(
				local percent = (((frame-startframe) as float)/((endframe - startframe) as float)) * 100.0;
				pb.value = percent
			)
		)
		OK
	)
	
	
	--------------------------------------------------------------------------------
	-- This sets up a scene and animates the first frame
	--------------------------------------------------------------------------------
	fn ImportBVH filename startt scale:1.0 loadanimation:true bonesize:3.0 quiet:false =
	(
		if filename == undefined then return OK;

		-- for some wird reason the importer causes havok with the motion panel rollout
		local panel = getCommandPanelTaskMode();
		setCommandPanelTaskMode mode:#display
		
		with redraw off (		
			with undo off (
				with animate off (			   
					initMotionData();
					parseBVHFile filename scale &loadanimation;
				
					--	Start a new undo block
					undo ~BVH_FILE_LOADED~ on(
						rootnode = Point axistripod:on Box:on name:(GetFileNameFile filename)
						for skel in skeletons do createJoints skel bonesize
						if loadanimation then
							CreateKeyframes quiet starttime:startt scalevalue:scale
					)
				)
			)
		)
		
		if(catparentnode != undefined and chkCapAnim.checked) then
		(
			local timerange = Interval startt (startt + motion.frames-1)
			CaptureAnimation catparentnode rootnode CamFile:camfile quiet:quiet delsrc:true StartTime:timerange.start EndTime:timerange.end
		)
		
		-- put us back on the we were in before import
		setCommandPanelTaskMode mode:panel
		return rootnode;	
	)
	
	fn Refresh =
	(
		if fname != undefined then
			rImportBVH.title = (~IMPORTING~+ (getFilenameFile fname))
			
		if catparentnode==undefined then
		(
			chkCapAnim.enabled = false;
			btnCamFile.enabled = false; 
		)
		else
		(
			if camfile != undefined and (doesFileExist camfile) then
				 btnCamFile.text = (getFilenameFile camfile)
			else btnCamFile.text = ~CAM_FILE_BUTTONTEXT~
		)
	)
	
	fn openDialog catparent:undefined fname:undefined cam:undefined scale:1.0 quiet:false = 
	(
		if fname == undefined then (
			local lastfileloaded = getIniSetting ini_file ~INI_FILE_FILES_CAPTION~ ~LASFILELOADED_CAPTION~
			if(lastfileloaded.count == 0)then lastfileloaded = ((getDir #plugcfg_ln) + "\\CAT\\*.bvh")
			
			fname = getOpenFileName caption:~FNAME_CAPTION~ \
					                   types:~BIOVISION_HIERARCHY_FILE_TYPES~ \
									   filename:lastfileloaded
		)
		
		if fname != undefined then
			setIniSetting ini_file ~SETINISETTING_FILES~ ~SETINISETTING_LASTFILELOADED~ fname
		else return OK;
	
		if quiet then(
		
			createDialog rImportBVH style:#(#style_border, #style_titlebar, #style_sysmenu) width:310 height:40 escapeEnable:true
			
			filename = fname;
			catparentnode = catparent;
			camfile = cam;
			
			chkAnimation.visible = false;
			spnScale.visible = false;
			spnStartTime.visible = false;
			spnBoneSize.visible = false;
			btnOk.visible = false;	
			
			lblStatus.pos = [10, 10]
			pb.pos = [160, 10]
			
			str = getIniSetting ini_file ~INI_FILE_PARAMS~ ~INI_FILE_BONESIZE~
			if (str.count > 0) then 	spnBoneSize.value = str as number		
			
			chkCapAnim.checked = true;
			
			ImportBVH filename 0 scale:scale loadanimation:true bonesize:spnBoneSize.value quiet:true
		
			DestroyDialog rImportBVH;

			
		)else(
			local pos = execute (getIniSetting ini_file #general #position)
			if pos == ok then pos = [100, 100]
			createDialog rImportBVH pos:pos style:#(#style_border, #style_titlebar, #style_sysmenu)
			filename = fname;
			catparentnode = catparent;
			camfile = cam;
			
			local str = getIniSetting ini_file ~LOCAL_STR_PARAMS~ ~LOCAL_STR_LOADANIMATION~
			if (str.count > 0) then 	chkAnimation.checked = execute str
			
			str = getIniSetting ini_file ~STR_PARAMS~ ~STR_SCALEFACTOR~
			if (str.count > 0) then 	spnScale.value = str as number
	
			str = getIniSetting ini_file ~GETINI_PARAMS~ ~GETINI_STARTTIME~
			if (str.count > 0) then 	spnStartTime.value = str as number
			
			str = getIniSetting ini_file ~INI_FILE_PARAMS_CAPTION~ ~INI_FILE_BONESIZE_CAPTION~
			if (str.count > 0) then 	spnBoneSize.value = str as number			
			
			str = getIniSetting ini_file ~STR_PARAMS_CAPTION~ ~STR_CAPTUREANIMATION_CAPTION~
			if (str.count > 0) then 	chkCapAnim.checked = execute str		
			
			camfile = getIniSetting ini_file ~STR_PARAMS_CAMFILE~ "CAMFile"
			
			Refresh();
		)
		
	)

	on btnCamFile pressed do
	(
		if camfile==undefined then(
			camfile = getIniSetting ini_file ~GETINI_PARAMS_CAMFILE~ "CAMFile"
			if camfile==undefined then(
				camfile = ((getDir #plugcfg_ln) + "\\CAT\\*.cam")
			)
		)
		camfile = getOpenFileName caption:~CAMFILE_CAPTION~ \
				                   types:~CAM_FILE_TYPES~ \
								   filename:camfile		
		Refresh()
	)
	
	on btnOk pressed do
	(
		ImportBVH filename spnStartTime.value scale:spnScale.value loadanimation:chkAnimation.checked bonesize:spnBoneSize.value
		
		setIniSetting ini_file ~INI_FILE_PARAMS_LOADANIMATION~ ~LOADANIMATION~ 	(chkAnimation.checked as string)
		setIniSetting ini_file ~PARAMS_SCALEFACTOR~ ~SCALEFACTOR~ 		(spnScale.value as string)
		setIniSetting ini_file ~PARAMS_STARTTIME~ ~STARTTIME_CAPTION~ 		(spnStartTime.value as string)
		setIniSetting ini_file ~PARAMS_BONESIZE~ ~BONESIZE_CAPTION~ 			(spnBoneSize.value as string)
		setIniSetting ini_file ~PARAMS_CAPTUREANIMATION~ ~CAPTUREANIMATION_CAPTION~ 	(chkCapAnim.checked as string)
		if camfile!=undefined then setIniSetting ini_file ~PARAMS_CAMFILE~ "CAMFile" 	camfile
		
		DestroyDialog rImportBVH;
	)
	on pb clicked val do
	(
	)
)
/*
fn CATImportBVH catparentnode fname CamFile:undefined quiet:false = 
(
	if quiet then
		 rImportBVH.ImportBVH fname 0
	else rImportBVH.openDialog catparent:catparentnode  fname:fname cam:CamFile
)
*/
fn CATImportBVH catparentnode bvhfile CamFile:undefined quiet:false scale:1.0 timeoffset:0 = 
(
	rImportBVH.openDialog catparent:catparentnode  fname:bvhfile cam:CamFile scale:scale quiet:quiet
)




-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEA5dK+WnG5bDemPmWVSBRBgw
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTcwODA0MDAwMDAwWhcNMTgwODA0MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBALPR50hy1FkrWOBmP+sGXfKWFUpFAKB9OLDlN3Uj
-- 94WBLdHje+wsBav/AOL1Ben4qOa74PWpJHTJd8jph4MSGhKZE3oFNPyAVXCVhUAj
-- qlLaYQXkHDWMeyz+y7FWX4oK1B1H+SNVcnc2+kAB0bEIT4VAIvQcyva41ThpVGzP
-- XZM/JKDDpA6tocMQ3935UAjHYuvoOADEkFt5O/lEWzPTz0aQkVLGiD18rgFxuSw+
-- Uz2jyuDZZ5lyNBQRF+K4cu8fle9uL2WqbaO7koHz76dkJrNW9wAmkdGCdfj3MQo+
-- OD4O5JjSMYHEcmjVbHyo+ZK/BIVykApxc0tfN2HRJSuHlG0CAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBALfg
-- FRNU3/Np7SJ5TRs8s8tPnOTd4D5We+stLCuQ0I1kjVIyiIY+Z3cQz2AB9x8VXuYF
-- LcXnT6Rc1cMYJtlTyB7Z7EZkfxQmFgc4chVfnguTpPqUtfo3QMT/S1+QIdYfIbk1
-- dSvFBmZwRGatmGbn2h7HGiIgNqQaO6TD7Fx9TEJPwIiiCK8F3b4ENpYQHlgH3OAd
-- CRLa1IWPfeA03yF3PIq8+NhLsngw1FNm9+C6UOM3mf3jHwxTrbt4ooIZstjPA4PU
-- G16FkhJg7l2RCDR6sE9iT7FMCsO6tAHX3pS8afFyNyEVfgJVKfzohdDOj+XQLkzp
-- c9v3Xoh1gTIPCte7VPsxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQDl0r5acblsN6Y+ZZVIFEGDANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQA5IBE8nzsb+9lCxoqf7yLrkmlrJ+ptzNV1
-- iNeu8OI59NAcOpKzag+7bf9jzNnrS/TmI6mqPfx5Oh/BQ6MAAaPGaHFNk95C/yt/
-- OTVZ0Bmy3MmfTAtu/F8DFB+p/DXqmHz79UjCpq1nC1MbhE8V+LBek2y5on505P3V
-- vKrxje9O21QUFr7osNBYgaymRFZXi9iOvatP7rNOVB6/PoODWEtEVv3jnrr2NgbA
-- 4IzxMCvRjYCt38dUnaDDfVWd2lDtXnfOW0IiuyK64wpCKF3v44BWp9lRPR38Kty/
-- XuC8b+3yFwZNDREKJ2Mj9MRnQ6kCNtI7jF9muqBZnXxKQ6eBXNX8
-- -----END-SIGNATURE-----