/*******************************************************************************************\
-- 	ApplyIKChain()										
--	This function applies one of Max's standard IK		
--	chains to a selected CAT hierarchy.					
--														
--	By: Stephen Taylor (aka Taystee) 					
--	On:	29-06-05(ish)									
--
-- What we are doing is...
-- 1. Creating a hierarchy of point helpers that matches our CAT Hierarchy exactly
-- 2. Assiging the max IK solver to this hierarchy. 
-- 3. We then constrain our CAT bones to follow the point helpers using orientation constraints											
--
\*******************************************************************************************/

filein "CATMacroUtilFunctions.ms"

applymaxIKCA = attributes applymaxIKCADef
(
	parameters main rollout:params
	(
		ik_blend type:#float ui:sldIKBlend default:0
	)
	rollout params ~IK_PARAMETERS_CAPTION~
	(
		slider sldIKBlend ~IK_BLEND_CAPTION~ type:#float range:[0,1, 0] 
	)
)


global rApplyIKChain;
rollout rApplyIKChain ~APPLY_MAX_IK_SYSTEMS_TO_CAT_BONES~ width:280
(
	local catchain;
	local catparentnode;
--	local targetLayer = 0;
	
	fn GetCATChainFromNode node = 
	(
		local ctrl = getTMController node
		case (classOf ctrl) of
		(
			TailTrans:				return ctrl.TailData
			CATBoneSegTrans:	return ctrl.BoneData.LimbData.controller
			palmTrans:				return ctrl.LimbData
			CATDigitSegTrans:	return ctrl.DigitData.controller
		)
		return undefined;
	)
	
	fn catHeirFilt o = (GetCATChainFromNode o != undefined)
	fn ShapeFilt o = ( return superClassOf o == Shape )
	
	
	pickbutton btnPickHierarchy ~BTNPICKHIERARCHY_BUTTONTEXT~ pos:[ ~BTNPICKHIERARCHY_POSITION~,12+30-38 ] width:144 height:24 filter:catHeirFilt
	label lbIKSolvers ~LBIKSOLVERS~ pos:[ ~LBIKSOLVERS_POSITION~,48+30-38 ] width:~LBIKSOLVERS_WIDTH~ height:13
	dropdownList lstIKSolvers "" pos:[ ~LSTIKSOLVERS_POSITION~,43+30-38 ] width:144 height:21
	label lbLayers ~LBLAYERS_CAPTION~ pos:[ ~LBLAYERS_POSITION~,71+30-38 ] width:~LBLAYERS_WIDTH~ height:13
	dropdownList lstLayers "" pos:[ ~LSTLAYERS_POSITION~,68+30-38 ] width:144 height:21 items:#(~USE_SETUP_CONTROLLER~, ~CREATE_NEW_LAYER~)

	
--	group ~GENERAL_IK_SETTINGS~ 
--	-(
--		checkbox chkConfigureIKBlendParam " : Setup IK Blend Slider on First Bone" align:#left checked:true
--	--	checkbox chkCreateExtraBoneIKHandles " : Create Extra Bones for IK Handles" align:#left 
--	)	
	
	group ~SPLINE_IK_SETTINGS~ 
	(
	--	label lbKnots ~NUM_KNOTS~width:55 height:13 -- pos:[ 26, 146-38 ] 
		spinner spnKnots ~SPNKNOTS_NUM_KNOTS~ enabled:false range:[ 2,10,3 ] type:#integer fieldwidth:50 align:#left --pos:[ 95-10, 146-38 ] 
	)
	button btnOk ~BTNOK_APPLY~ width:88 height:24 enabled:false align:#right --pos:[ 152-10, 166-38 ] 
	
	fn Refresh =
	(
		btnOk.enabled = (catparentnode!=undefined);
	)
	
	-----------------------------------------------------------------------------------------------
	-- function: ApplyOrientationConst
	-- Does: Applies an Orientation Constratin controller to the given
	-- 		layerTrans controller, and set up its parameters
	-- Arguments:
	--		layerTrans: layerTrans Controller. Assign created
	--					  to this controllers active layer
	--		node: Node to look at
	--		rotation: Possible Rotation offset.
	-- Returns: nothing
	-----------------------------------------------------------------------------------------------
	fn ApplyOrientationConst tabCATNodeCtrls catboneid targetnode targetlayer bDoConversion tmConvertZoX = 
	(	
		local layerTrans = tabCATNodeCtrls[ catboneid ].layertrans.controller;
		
		-- Create the Orentation Constraint controller 
		-- and set the target as the point helper
		local ctrlIKConstraint = Orientation_Constraint();
		ctrlIKConstraint.appendTarget targetnode 100;
		
		-- We must assign the controller immediately, becaue when it is assigned, 
		-- the currently assigned controller is always appended to the end of controllers
		if( targetlayer > 0) then
		(
			layerTrans.SelectedLayerCtrl.rotation.controller = ctrlIKConstraint;
		)
		
		if bDoConversion then (
			local ctrlOffsetList = RotationList();
			
			if( targetlayer > 0) then
			(
				-- Assign the new list controller. The Constraint automaticly goes into the 1st channel
				layerTrans.SelectedLayerCtrl.rotation.controller = ctrlOffsetList;
			)
			else
			(
				ctrlOffsetList.available.controller = ctrlIKConstraint;
			)
			
			-- We add another channel to the list controller to hold the CoordinateSystem Conversion
			ctrlOffsetList.available.controller = Euler_XYZ();
			ctrlOffsetList[ 2 ].value = ((Inverse tmConvertZoX) as quat);
			ctrlOffsetList.SetName 1 "IK_Constraint"
			ctrlOffsetList.SetName 2 "Z_To_X_Conversion"
			
			-- We will use this List controller as the constraint controller
			ctrlIKConstraint = ctrlOffsetList;
		)
		
		-- Assign our controller to the layerTrans
	--	print (~PRINT_USE_SETUP_CONTROLLER~ + ( ( targetlayer == 0) as string ));
		if( targetlayer == 0) then
		(
			local ikblend_controller = undefined;
			
			if( catboneid == 1 ) then
			(
				custAttributes.add tabCATNodeCtrls[ 1 ].node applymaxIKCA;
				local ikblend_custattributes = (custAttributes.get tabCATNodeCtrls[ 1 ].node applymaxIKCA BaseObject:true);
				ikblend_custattributes[ 1 ].controller = tabCATNodeCtrls[ catboneid ].CreateLayerFloat()
				ikblend_controller = ikblend_custattributes[ 1 ].controller;
				--  By default, we want the spring controllers turned off.
				ikblend_controller.SetupVal = 0.0;
			)
			else
			(
				ikblend_controller = (custAttributes.get tabCATNodeCtrls[ 1 ].node applymaxIKCA BaseObject:true)[ 1 ].controller;
			)
			
			layerTrans.UseSetupController = true;
			layerTrans.AdditiveToSetupValue = true;
			
			layerTrans.SetupController.rotation.controller = RotationList();
			local ctrlRotationList = layerTrans.SetupController.rotation.controller;
			
			-- Add a new channel at the bottom of the list controller. 
			-- The first slot is reserved for IK, and the second will be constrained to the IK Chain;
			ctrlRotationList.available.controller = ctrlIKConstraint;
			
			-- Update the names that will appear in the track view and assign controller dialog
			ctrlRotationList.SetName 1 "FK";
			ctrlRotationList.SetName 2 "IK_Constraint";
			
			-- Now set up the scripts whcih will control the weights.
			-- Put script controllers on each of the weights parameters
			ctrlRotationList[4][2].controller = float_script();
			ctrlRotationList[4][2].controller.AddTarget "CATLayerFloatIKBlendController" ikblend_controller;
			ctrlRotationList[4][2].controller.SetExpression "CATLayerFloatIKBlendController";
		)
	)

	-----------------------------------------------------------------------------------------------
	-- function: CreateHelperHierarchy	
	-- Does: Creates a hierarchy of point helpers at the joints in the
	--		layerTrans controllers. Creates extra point helper at end of
	--		final element, as each node/ctrl looks at the NEXT joint. Assigns
	--		point helpers to pointArray
	-- Arguments:
	--		ctrlTab: table of controllers controlling nodes. NOT nodeTab[ i ].3 though
	--		initialParentNode: Parent of the hierarchy as a whole
	--		last_bone_length: length of final element in hierarchy
	-- Returns: nothing
	-----------------------------------------------------------------------------------------------
	fn CreateHelperHierarchy tabCATNodeCtrls initialParentNode last_bone_length bDoConversion tmConvertZoX =
	(
		-- Init local variables
		local numhelpers = tabCATNodeCtrls.count + 1;
		local parentNode = initialParentNode;
		local point_helper_node;
		local point_helper_wire_color;
		local point_helper_transform;
		
		-- our array of point helper nodes
		local tabPointHelpers = #();
		
		for i = 1 to numhelpers do
		(
			-- Create one point helper for each bone in the CAT chain, and the one extra for the chain tip, or 'nub'
			if i==1 then
			(
				point_helper_node = Point()
			--	point_helper_node.size = 10
				point_helper_node.centermarker= false;
				point_helper_node.cross = true;
				point_helper_node.axistripod = false;
				point_helper_node.box = false;
				point_helper_node.constantscreensize = false;
				point_helper_node.drawontop = false;
			)
			else
 			(
				-- instance all the point helpers by sharing the object
				local newpt = Point();
				newpt.baseobject = point_helper_node.baseobject
				point_helper_node = newpt;
			)
			
			-- What we are doing is
			-- creating a hierarchy of point_helper_node helpers, and assiging our
			-- IK system to this hierarchy. We then constrain our CAT bones
			-- to follow the point helpers using orientation constraints
			if(i < numhelpers) then
			(
				point_helper_node.name = tabCATNodeCtrls[ i ].node.name + "IKPt"
				point_helper_wire_color = tabCATNodeCtrls[ i ].node.wirecolor;
				
				-- the transform for the new point we create. 
				point_helper_transform = tabCATNodeCtrls[ i ].node.transform;
				if bDoConversion then point_helper_transform = tmConvertZoX * point_helper_transform;
			)
			else
			(
				point_helper_node.name = tabCATNodeCtrls[ numhelpers-1 ].node.name + "Nub";
				
				local translationVect;
				-- In this section we wish to place the point_helper_node helper at the end of the last link
				if catparentnode.lengthaxis == "X" or bDoConversion then
					 translationVect = [ last_bone_length, 0, 0 ];
				else translationVect = [ 0, 0, last_bone_length ];

				point_helper_transform = transMatrix(translationVect) * parentNode.transform;
			)
			
			point_helper_node.wirecolor = point_helper_wire_color;
			point_helper_node.parent = parentNode;
			point_helper_node.transform = point_helper_transform;
			point_helper_node.size = catparentnode.catunits * 3.0;
			
			-- Now we have created and linked our point_helper_node helper, add it to the array.
			append tabPointHelpers point_helper_node;
			
			if( i < numhelpers ) then
			(
				-- Save the newly created node as our new parent node for the next node in the chain
				parentNode = point_helper_node
			)
		)
		return tabPointHelpers;
	)
	
	fn ConstrainCATChainToPointHelpers tabCATNodeCtrls tabPointHelpers targetlayer bDoConversion tmConvertZoX =
	(
		for i = 1 to tabCATNodeCtrls.count do
		(
			ApplyOrientationConst tabCATNodeCtrls i tabPointHelpers[i] targetlayer bDoConversion tmConvertZoX;
		)
	)
	
	-----------------------------------------------------------------------------------------------
	-- function: CreateHelperHierarchyByChainType 	
	-- Does: Creates parameters for CreateHelperHierarchy. Extracts the necessary information
	--		depending on what we are trying to apply our IK to, and then 
	--		calls CreateHelperHierarchy to do the actual work
	-- Arguments: none
	-- Returns: nothing
	-----------------------------------------------------------------------------------------------
	fn CreateHelperHierarchyByChainType catchain bDoConversion tmConvertZoX = 
	(	
		local parentNode;
		local last_bone_length;
		
		case (classOf catchain) of
		(
			TailData2:
			(
				parentNode = catchain.hub.Node
				last_bone_length = catchain.bones[ catchain.bones.count ].node.baseobject.Length * catparentnode.catunits;
			)
			CATLimbData2:
			(
				local parentNode = catchain.hub.Node
				local last_bone_length = catchain.Bones[ catchain.NumBones ].Length * catparentnode.catunits;
				
				-- We have our point helpers, our lookats are set up, 
				-- the last thing we have do to put them in use is set our
				-- limb to be in FK (and override CAT's built in IK)
				-- in FK, the palm and ankle offset thier parents transform with thier own SeupTM before using it
				-- here we are calculating an FK matrix that will keep the Ankle on the same rotation as it is in IK
		--		for j = 1 to catchain.palm.Digits.count do(
		--			local digit = catchain.palm.Digits[ j ];
		--			local tm = digit.bones[ 1 ].Node.transform * (Inverse catchain.palm.Node.transform);
		--			tm.translation = [ 0,0,0 ];
		--			digit.bones[ 1 ].layerTrans.controller.SelectedLayerCtrl.value = tm;
		--		)
		--		catchain.palm.layerTrans.value = catchain.palm.Node.transform * Inverse(catchain.Bones[ catchain.Bones.count ].node.transform);
		--		catchain.layerikfkratio.controller.SelectedLayerCtrl.value = 1.0;
			)
			DigitData:
			(
				parentNode = catchain.palm.Node;
				last_bone_length = catchain.Bones[ catchain.NumBones ].Length * catparentnode.catunits;
			)
		)
		
		-- Create all the point helpers. 
		-- The point helpers are aligned perfectly with the CATBone chain
		return CreateHelperHierarchy  catchain.Bones parentNode last_bone_length bDoConversion tmConvertZoX;
	)
	
	-----------------------------------------------------------------------------------------------
	-- function: ApplyIKToPtHelpers 	
	-- Does: Apply an IK system to the linked chain of Pt Helpers
	-- 		that should have been created by now, stored in pointArray
	-- Arguments: none
	-- Returns: nothing
	-----------------------------------------------------------------------------------------------
	fn ApplyIKToPtHelpers tabPointHelpers solvertype = 
	(
		if(tabPointHelpers.count < 2) then return false;

		if tabPointHelpers[ 1 ] == undefined or tabPointHelpers[ tabPointHelpers.count ] == undefined then(
			throw ~THROW_ERROR_CREATING_IK_SYSTEM~
			return OK;
		)

		case ( solvertype ) of
		(
			-- Apply HI Soliver
			1:
			(
				iksys.ikchain  tabPointHelpers[ 1 ] tabPointHelpers[ tabPointHelpers.count ] "IKHISolver"
			)
			-- Apply HD Solver
			2:
			(
				HDiksys.ikchain tabPointHelpers[ 1 ] tabPointHelpers[ tabPointHelpers.count ] True
			)
			-- IK Limb Solver
			3: 
			(
				if(tabPointHelpers.count > 3) do
				(
					messagebox ~MSGBOX_IKLIMB_SYSTEM_ONLY_APPLIED~
					return false
				)
				iksys.ikchain tabPointHelpers[ 1 ] tabPointHelpers[ tabPointHelpers.count ] "IKLimb"
			)
			-- Spline IK
			4:
			(
				-- First thing to do, take the
				-- first nodes parent, store it, and unlink it.
				pthlpParent = tabPointHelpers[ 1 ].parent;
				tabPointHelpers[ 1 ].parent = undefined;
				
				-- start applying IK.
				NIK = iksys.IKChain  tabPointHelpers[ 1 ] tabPointHelpers[ tabPointHelpers.count ] "SplineIKSolver"
				
				local create_spline = querybox ~QUERYBOX_WANT_SPLINE_CREATED~ title:~QUERYBOX_WANT_SPLINE_CREATED_TITLE~
				
				EC_SplineOBJ = undefined
				if create_spline then
				(
					try(
						-- Create a Spline Object, matching the 
						-- existing hierarchy closely
						EC_SplineOBJ = splineShape()
						EC_SplineOBJ.name = (catchain.name + "IKSpline")
						EC_SplineOBJ.pos = tabPointHelpers[ 1 ].pos
						--EC_SplineOBJ.parent = tabPointHelpers[ 1 ].parent
						
						addNewSpline EC_SplineOBJ
	
						addKnot EC_SplineOBJ 1 #smooth #curve tabPointHelpers[ 1 ].pos
						numKnots = spnKnots.value - 1;
						for i = 1 to numKnots do
						(
							local ratio = ((i as float)/numKnots) * tabPointHelpers.count;
							local id = (floor ratio);
							local pos = tabPointHelpers[ id ].pos;
							if(id!=ratio)then (
								if id < tabPointHelpers.count then(
									pos = pos + ((tabPointHelpers[ id+1 ].pos - pos) * (ratio - (floor ratio)));
								)
							)
							addKnot EC_SplineOBJ 1 #smooth #curve pos
						)
						updateShape EC_SplineOBJ;
					)catch( ShowMessage ~SHOWMSG_FAILED_TO_CREATE_SPLINE~ )
				)
				else
				(
					-- Pick an existing spline to use.
					EC_SplineOBJ = PickObject count:1 select:true filter:ShapeFilt count:#Multiple Message:~PICK_SPLINE_TO_ASSIGN_SPLINEIK~ Rubberband:tabPointHelpers[ tabPointHelpers.count ].pos ForceListenerFocus:False
				)
				-- Mostly Copied from MAX macro scripts
				if EC_SplineOBJ != undefined and EC_SplineOBJ != ~NONE~ then
				(
					-------------------------------------------------------------------------------------------
					-- Make the Spline the goal shape of the SplineIK
					-------------------------------------------------------------------------------------------
					NIK.transform.controller.pickShape = EC_SplineOBJ		
	
					-------------------------------------------------------------------------------------------			
					-- Add SplinIKControl Modifier and Create Helpers
					-------------------------------------------------------------------------------------------		
					mod = Spline_IK_Control()
					AddModifier EC_SplineOBJ (mod)
					mod.createHelper(0)
					EC_HelperOBJ = mod.helper_list[ 1 ]
					
					-- Set the upnode to the first helper
					NIK.controller.upnode = EC_HelperObj
	
					-------------------------------------------------------------------------------------------
					-- Add List Controller
					-------------------------------------------------------------------------------------------
					local cont = AddListController tabPointHelpers[ 1 ] "Pos" Position_List
					
					-------------------------------------------------------------------------------------------
					-- Add Constraint
					-------------------------------------------------------------------------------------------
					If classof cont[ listCtrl.GetActive cont ].object != Position_Constraint then constraint = AddConstraint tabPointHelpers[ 1 ] "Pos" Position_Constraint true
					else constraint = cont[ listCtrl.GetActive cont ].object
												
					----------------------------	---------------------------------------------------------------
					-- Add Position Constraint Object Target as the first helper on the spline
					-------------------------------------------------------------------------------------------		
					constraint.AppendTarget EC_HelperOBJ 50
								
					-------------------------------------------------------------------------------------------
					-- Set Active Controller
					-------------------------------------------------------------------------------------------
					SetActiveController cont constraint
								
					--Select EC_OBJ
					EC_HelperOBJ.parent = pthlpParent;
					
				--	NIK.controller.autoEnable = false;
				--	NIK.controller.autoEnable = true;
				)
				-- Reassign the original parent, but to the spline now.
				--EC_SplineObj.parent = pthlpParent;
				--tabPointHelpers[ 1 ].parent=pthlpParent
			)
		)
	)
	
	fn ApplyMaxIKtoCATChain catchain targetlayer iksolvertype =
	(		
		local catparentnode =  catchain.catparent;
		local oldpanel = getCommandPanelTaskMode()
		setCommandPanelTaskMode #display
		
		undo ~APPLY_MAX_IK~ on 
		(
			local current_selection = ($Selection as array);
			-- a flag telling us whether we do the conversion
			local bDoConversion = false;
			-- A matrix that converse bones from being Z-Aligned to X-Aligned
			local tmConvertZoX = Matrix3 1;
			-- if the user has selected the SplineIK system in the ISolvers list, 
			-- then we NEED X forward, not Z. 
			if(iksolvertype == 4) do
			(
				if catparentnode.lengthaxis == "Z" then
				(
					bDoConversion = true;
					-- Spin from Z forward Y up to X forward Y up?
					tmConvertZoX = rotateYMatrix(-90)
				)
			)
			-- Createand return  a table of point helpers
			local tabPointHelpers = CreateHelperHierarchyByChainType catchain bDoConversion tmConvertZoX;
			
			-- Now Constrain the CATBone Chain to the Point helpers
			ConstrainCATChainToPointHelpers catchain.Bones tabPointHelpers targetlayer bDoConversion tmConvertZoX;
			
			-- Now apply the 3dsmax IK Solver to the CAT bone chain
			ApplyIKToPtHelpers tabPointHelpers iksolvertype;
			
			-- close the dialog
			DestroyDialog rApplyIKChain;
			Select current_selection;
		)
		setCommandPanelTaskMode oldpanel	
	)

	-----------------------------------------------------------------------------------------------
	-- UI Functions for managing the Dialog
	-----------------------------------------------------------------------------------------------	
	-----------------------------------------------------------------------------------------------
	-- function: FillLayerList 	
	-- Does: Fills the drop down list with layers it finds on the CATParent
	-- Arguments: None
	-- Returns: zilch
	-----------------------------------------------------------------------------------------------	
	fn FillLayerList =
	(
		newLayerList = #(~SETUP_CONTROLLER~)
		if(catparentnode == undefined) do 
		(	
			return undefined
		)
		for i = 1 to catparentnode.Layers.controller.numlayers do
		(
			append newLayerList ((i as string)+": "+(catparentnode.Layers.controller[ i ].name))
		)
		append newLayerList ~CREATE_NEW_LAYER_PLUS_L~;
		lstLayers.items = newLayerList
	)
	
	fn UpdateUIWithTargetNode obj =
	(
		btnPickHierarchy.text = obj.name;
		catchain = GetCATChainFromNode obj;
		catparentnode = GetCatParent catchain;

		Refresh();
		FillLayerList();
	)

	-----------------------------------------------------------------------------------------------
	-- Boring Max dialogue maintenance stuff from here on
	-----------------------------------------------------------------------------------------------	
	fn openDialog = 
	(
		createDialog rApplyIKChain style:#(#style_border, #style_titlebar, #style_sysmenu)
	)

	on rApplyIKChain open do
	(
		lstIKSolvers.items = #(~HI_SOLVER~, ~HD_SOLVER~, ~IK_LIMB_SOLVER~, ~SPLINE_IK_SOLVER~);

		if selection.count>0 and (catHeirFilt selection[ 1 ]) then 
			UpdateUIWithTargetNode selection[ 1 ];
	)
	on btnPickHierarchy picked obj do
	(
		UpdateUIWithTargetNode obj;
	)
	on lstIKSolvers selected i do
	(
		spnKnots.enabled = (i==4)
	)
	on btnOk pressed do
	(
		local targetlayer;
	
		-- The first option in the list is the Create New option.	
		if( lstLayers.selection == 1) then
		(
			targetlayer = 0;
		)
		else if( lstLayers.selection == lstLayers.items.count ) then
		(
			-- Add the new layer now, and return its index
			catparentnode.AppendLayer lstIKSolvers.selected #RelativeLocal;
			targetlayer = catparentnode.numLayers;
		)
		else 
		(
			-- Otherwise, the index of the list item is the index of layer
			catparentnode.selectedlayer = lstLayers.selection-1;
			targetlayer = lstLayers.selection-1;
		)
		
		ApplyMaxIKtoCATChain catchain targetlayer lstIKSolvers.selection;
	)
		
	on rApplyIKChain close do
	(
		if toolMode.CommandMode == #pick do toolMode.CommandMode = #move
	)
)



-- When evaluating this script, this line tells us to
-- open the dialogue, otherwise nothing would happen






-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEA5dK+WnG5bDemPmWVSBRBgw
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTcwODA0MDAwMDAwWhcNMTgwODA0MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBALPR50hy1FkrWOBmP+sGXfKWFUpFAKB9OLDlN3Uj
-- 94WBLdHje+wsBav/AOL1Ben4qOa74PWpJHTJd8jph4MSGhKZE3oFNPyAVXCVhUAj
-- qlLaYQXkHDWMeyz+y7FWX4oK1B1H+SNVcnc2+kAB0bEIT4VAIvQcyva41ThpVGzP
-- XZM/JKDDpA6tocMQ3935UAjHYuvoOADEkFt5O/lEWzPTz0aQkVLGiD18rgFxuSw+
-- Uz2jyuDZZ5lyNBQRF+K4cu8fle9uL2WqbaO7koHz76dkJrNW9wAmkdGCdfj3MQo+
-- OD4O5JjSMYHEcmjVbHyo+ZK/BIVykApxc0tfN2HRJSuHlG0CAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBALfg
-- FRNU3/Np7SJ5TRs8s8tPnOTd4D5We+stLCuQ0I1kjVIyiIY+Z3cQz2AB9x8VXuYF
-- LcXnT6Rc1cMYJtlTyB7Z7EZkfxQmFgc4chVfnguTpPqUtfo3QMT/S1+QIdYfIbk1
-- dSvFBmZwRGatmGbn2h7HGiIgNqQaO6TD7Fx9TEJPwIiiCK8F3b4ENpYQHlgH3OAd
-- CRLa1IWPfeA03yF3PIq8+NhLsngw1FNm9+C6UOM3mf3jHwxTrbt4ooIZstjPA4PU
-- G16FkhJg7l2RCDR6sE9iT7FMCsO6tAHX3pS8afFyNyEVfgJVKfzohdDOj+XQLkzp
-- c9v3Xoh1gTIPCte7VPsxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQDl0r5acblsN6Y+ZZVIFEGDANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQCds9BLbOgJk4QgE8barsfxns0IAHX8Bngf
-- 0Vckk5tegftpLCcL08zUqGlXvqEGhKJJuoJg2m8RyTb5zLuU9RQ0ps+VRGh3v+l5
-- qdvGnzqLNfJhZyM+Y2sS6EENwHibLhJ3xWWS3vfBA1TMJywSLtCYTu2WJnBl28MA
-- KsBL2txUQkL9DHqEOIzz59sxHOJvXRlG0ikmnOcTkreJx6GLE6rClwJZIZE9lXoZ
-- kNSDmscdFobU55n3kDjAH+rG9WM6VtYwB5IY1D+GfSTxeg/xcKzyjz7dBHrbpvRt
-- goYHOdruPnOIbrNXtqpM2/PZZaIVYlVkIvKvZol7G2Pm4QG6W8w9
-- -----END-SIGNATURE-----