
-- This structure wraps operations specific to the Procedural Content feature
struct ProceduralContentOps
(
	private
	--  Loads the assemblies (DLLs) needed by the procedural content code
	fn LoadAssemblies = 
	(
		-- Load the PC-3dsMax bridge assembly
		local bridgePath = (symbolicPaths.getPathValue "$max") + @"\Viper3dsMaxBridge.dll"
		dotNet.loadAssembly bridgePath returnPassFail:true
		
		-- Load the MaxPlus assembly
		local maxPlusPath = (symbolicPaths.getPathValue "$max") + @"\MaxPlusDotNet.dll"
		dotNet.loadAssembly maxPlusPath returnPassFail:true
	),
	
	-- Creates the user profile folder if not existent and copies the CSS file
	fn SetupUserProfileFolder =
	(
		local bridge = dotNetClass "Viper3dsMaxBridge.Main"
		try
		(
			-- This function can throw a .NET exception if folders cannot be created.
			-- This will happen when 3dsMax runs in low integrity mode because folders
			-- are not created in %USER PROFILE%\AppData\LocalLow.
			bridge.CopyProceduralContentToUserProfile()
		)
		catch()
	),
	
	-- Compiles and registers the standard (built-in) operators, all compounds, and all tools
	-- to make them available to the current user
	fn ReloadOperatorsAndCompileTools =
	(
		local bridge = dotNetClass "Viper3dsMaxBridge.Main"    
		--  Loads all compounds from the users profile folder and exports the operators.
		bridge.ReloadOperators()
		-- Compile all graphs from all the registered folders
		bridge.CompileAllToolsAndPackages()
	),

	-- Adds directories from the specified section of the ini file to the list of directory
	fn AddDirs AddDirFunc iniFile iniSection = 
	(
		if (hasINISetting iniFile iniSection) then
		(
			local iniKeys = getINISetting iniFile iniSection
			for key in iniKeys do
			(
				local dir = getINISetting iniFile iniSection key
				AddDirFunc dir
			)
		)
	),

	-- Registers with Viper the custom graph directories specified in the startup ini file
	fn RegisterCustomGraphPaths =
	(
		local iniFile = getMAXIniFile()
		local settings = dotNetClass "Viper3dsMaxBridge.Settings"
		
		-- MCG directory evaluation order: Max, user, custom directories if any
		AddDirs settings.AddCompoundDirectory iniFile "MCG Compound Directories"
		AddDirs settings.AddToolDirectory iniFile "MCG Tools Directories"
		AddDirs settings.AddPackageDirectory iniFile "MCG Package Directories"
		AddDirs settings.AddAssemblyDirectory iniFile "MCG Assembly Directories"
	),
	
	-- Register a call back on file open 
	fn RegisterForPreLoadMaxFile = 
	(
		callbacks.removeScripts id:#pcPreFileOpen
		callbacks.addScript #filePreOpenProcess "ProceduralContentOps.OnPreFileOpen()" id:#pcPreFileOpen	
	),

	public
	
	-- Call back registered for file opens  
	fn OnPreFileOpen = 
	(
		-- This callback is effective only when Max is in network rendering mode
		if (IsNetServer()) then
		(
			local np = callbacks.notificationParam()
			--format "CompileProceduralContentGraphs: param1=%, param2=%\n" (np[1]) (np[2])
			local path = (getFilenamePath np[2])
			local bridge = dotNetClass "Viper3dsMaxBridge.Main"
			local replaceExistentCompounds = true
			-- Load the compounds from the folder where the Max file is loaded from
			-- Any new compounds that have the same name as already registered ones will overwrite them
			bridge.LoadViperCompounds path replaceExistentCompounds
			-- Load and compile the tools from the folder where the Max file is loaded from
			local loadToolsFromSubFolders = true
			local PrintToConsole = true
			bridge.CompileGraphsByFolder path
		)
	),

	-- This sets up the user currently logged in and running 3ds Max to use
	-- the Procedural Content features. It registers built in operators, 
	-- loads and registers compounds and tools 
	fn SetUpCurrentUser = 
	(
		LoadAssemblies()
		SetupUserProfileFolder()
		RegisterCustomGraphPaths()
		ReloadOperatorsAndCompileTools()
		RegisterForPreLoadMaxFile()
	)
)

-- A helper function for generating a MaxPlus INode from a MAXScript node
fn MaxPlusINodeFromNode n = 
(
	if not (IsValidNode n) do return undefined
	local maxPlusINodeClass = dotNetClass "Autodesk.Max.MaxPlus.INode"    
	if (maxPlusINodeClass == undefined) then (
		format "Error: failed to load Autodesk.Max.MaxPlus.INode\n"
		return undefined
	)
	return maxPlusINodeClass._CreateWrapper n
)

-- Turns on a log file that profiles mesh updating in modifiers and geometry. 
fn TurnOnMCGProfiling =
(
	local profiler = dotNetClass "Viper3dsMaxBridge.Profiler"    
	profiler.TurnOnProfiling()
)

-- Turns off the profiler 
fn TurnOffMCGProfiling = 
(
	local profiler = dotNetClass "Viper3dsMaxBridge.Profiler"    
	profiler.TurnOffProfiling()
)

-- Create the procedural content structure 
thePcOps = ProceduralContentOps()

-- The main entry function 
thePcOps.SetUpCurrentUser()

fn GetFileNameForMCGUserTool theToolName = (
	-- @todo: there should be symbolic paths for MCG directories
	viperMaxBridge = dotNetClass "Viper3dsMaxBridge.Main"
	local toolPath = viperMaxBridge.UserToolsDirectory 
	append toolpath ("\\" + theToolName + ".maxtool")
)

fn GenerateMeshGraphCompoundForSelection addUV compoundName: = 
(
	local res = false
	if (selection.count == 1) then (
		createAction = dotNetClass "ProceduralContentActions.PCActionCreateCompoundFromMesh"
		if (createAction != undefined) do (
			if (compoundName == unsupplied) do
				compoundName = selection[1].name
			res = createAction.GenerateMeshGraphCompoundForSelection addUV compoundName
		)
	)
	res
)

-- Generate a new tool file from a tool template file
-- Optionally compile it and open it in the graph editor
fn GenerateMCGToolFromTemplate templateFilename toolName compileIt openIt =
(
	local res = false
	local templateStream = openFile templateFilename
	if (templateStream != undefined) then (
		-- read the content of the template file to a string
		local template = readChars templateStream -1 errorAtEOF:false
		close templateStream 

		local guidMarker1 = "uuid=\""
		local guidMarker2 = "guid=\""
		local identifierMarker = "<identifier>"
		local displayNameMarker = "<displayName>"
		local guidPattern = "????????-????-????-????-????????????"
		-- Replace the 2 existing guids in the template file with new ones, dynamically created
		local ndx = findString template guidMarker1
		if (ndx != undefined) do (
			local newGuid = genGuid noBrackets:true
			template = replace template (ndx + guidMarker1.count) guidPattern.count newGuid 
			ndx = findString template guidMarker2
			if (ndx != undefined) do (
				newGuid = genGuid noBrackets:true
				template = replace template (ndx + guidMarker2.count) guidPattern.count newGuid
				ndx = findString template identifierMarker
				if (ndx != undefined) do (
					local identifier = toolName
					template = replace template (ndx + identifierMarker.count) 0 identifier
					ndx = findString template displayNameMarker
					if (ndx != undefined) do (
						local displayName = toolName
						template = replace template (ndx + displayNameMarker.count) 0 displayName
						
						-- Now save that string to <toolName>.maxtool in the user MCG Tools folder
						local toolPath = GetFileNameForMCGUserTool toolName
						local toolStream = openFile toolpath mode:"w"
						if (toolStream != undefined) do (
							format "%" template to:toolStream
							close toolStream

							res = true
							if (compileIt) do 
							(
								-- compile this tool, and then create a selected instance of the tool object
								local viperMaxBridge = dotNetClass "Viper3dsMaxBridge.Main"
								res = viperMaxBridge.CompileGraph toolpath
							)

							if (openIt) do
							(
								-- Open the MCG Editor with that template graph
								geBridge = dotNetClass "ProceduralContentActions.PCGraphEditorBridge"
								geBridge.LoadGraph toolpath
							)
						)
					)	
				)
			)
		)
	)
	res
)
-------BEGIN-SIGNATURE-----
-- 4wYAADCCBt8GCSqGSIb3DQEHAqCCBtAwggbMAgEBMQ8wDQYJKoZIhvcNAQELBQAw
-- CwYJKoZIhvcNAQcBoIIE3jCCBNowggPCoAMCAQICEA5dK+WnG5bDemPmWVSBRBgw
-- DQYJKoZIhvcNAQELBQAwgYQxCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRl
-- YyBDb3Jwb3JhdGlvbjEfMB0GA1UECxMWU3ltYW50ZWMgVHJ1c3QgTmV0d29yazE1
-- MDMGA1UEAxMsU3ltYW50ZWMgQ2xhc3MgMyBTSEEyNTYgQ29kZSBTaWduaW5nIENB
-- IC0gRzIwHhcNMTcwODA0MDAwMDAwWhcNMTgwODA0MjM1OTU5WjCBijELMAkGA1UE
-- BhMCVVMxEzARBgNVBAgMCkNhbGlmb3JuaWExEzARBgNVBAcMClNhbiBSYWZhZWwx
-- FzAVBgNVBAoMDkF1dG9kZXNrLCBJbmMuMR8wHQYDVQQLDBZEZXNpZ24gU29sdXRp
-- b25zIEdyb3VwMRcwFQYDVQQDDA5BdXRvZGVzaywgSW5jLjCCASIwDQYJKoZIhvcN
-- AQEBBQADggEPADCCAQoCggEBALPR50hy1FkrWOBmP+sGXfKWFUpFAKB9OLDlN3Uj
-- 94WBLdHje+wsBav/AOL1Ben4qOa74PWpJHTJd8jph4MSGhKZE3oFNPyAVXCVhUAj
-- qlLaYQXkHDWMeyz+y7FWX4oK1B1H+SNVcnc2+kAB0bEIT4VAIvQcyva41ThpVGzP
-- XZM/JKDDpA6tocMQ3935UAjHYuvoOADEkFt5O/lEWzPTz0aQkVLGiD18rgFxuSw+
-- Uz2jyuDZZ5lyNBQRF+K4cu8fle9uL2WqbaO7koHz76dkJrNW9wAmkdGCdfj3MQo+
-- OD4O5JjSMYHEcmjVbHyo+ZK/BIVykApxc0tfN2HRJSuHlG0CAwEAAaOCAT4wggE6
-- MAkGA1UdEwQCMAAwDgYDVR0PAQH/BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMD
-- MGEGA1UdIARaMFgwVgYGZ4EMAQQBMEwwIwYIKwYBBQUHAgEWF2h0dHBzOi8vZC5z
-- eW1jYi5jb20vY3BzMCUGCCsGAQUFBwICMBkMF2h0dHBzOi8vZC5zeW1jYi5jb20v
-- cnBhMB8GA1UdIwQYMBaAFNTABiJJ6zlL3ZPiXKG4R3YJcgNYMCsGA1UdHwQkMCIw
-- IKAeoByGGmh0dHA6Ly9yYi5zeW1jYi5jb20vcmIuY3JsMFcGCCsGAQUFBwEBBEsw
-- STAfBggrBgEFBQcwAYYTaHR0cDovL3JiLnN5bWNkLmNvbTAmBggrBgEFBQcwAoYa
-- aHR0cDovL3JiLnN5bWNiLmNvbS9yYi5jcnQwDQYJKoZIhvcNAQELBQADggEBALfg
-- FRNU3/Np7SJ5TRs8s8tPnOTd4D5We+stLCuQ0I1kjVIyiIY+Z3cQz2AB9x8VXuYF
-- LcXnT6Rc1cMYJtlTyB7Z7EZkfxQmFgc4chVfnguTpPqUtfo3QMT/S1+QIdYfIbk1
-- dSvFBmZwRGatmGbn2h7HGiIgNqQaO6TD7Fx9TEJPwIiiCK8F3b4ENpYQHlgH3OAd
-- CRLa1IWPfeA03yF3PIq8+NhLsngw1FNm9+C6UOM3mf3jHwxTrbt4ooIZstjPA4PU
-- G16FkhJg7l2RCDR6sE9iT7FMCsO6tAHX3pS8afFyNyEVfgJVKfzohdDOj+XQLkzp
-- c9v3Xoh1gTIPCte7VPsxggHFMIIBwQIBATCBmTCBhDELMAkGA1UEBhMCVVMxHTAb
-- BgNVBAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMR8wHQYDVQQLExZTeW1hbnRlYyBU
-- cnVzdCBOZXR3b3JrMTUwMwYDVQQDEyxTeW1hbnRlYyBDbGFzcyAzIFNIQTI1NiBD
-- b2RlIFNpZ25pbmcgQ0EgLSBHMgIQDl0r5acblsN6Y+ZZVIFEGDANBgkqhkiG9w0B
-- AQsFADANBgkqhkiG9w0BAQEFAASCAQADdj2KH9ZdTNUZorAGe7g8TTHGdTlJcCjT
-- LjQ5epe+9GS6RJVWMX9ti8YNU5TMfSe9QXjBDTPcHtPXv0hqXnr4za7+wIZZDU59
-- 5ZRHk4AruCs712CAw0KW3r4c16WHgH5yQ+mhvxSDSwuIhfU2sM//Aj08kyPsvtlN
-- KYL/IxTFFR9wW2h96jE0qSNBEqP3q7z7zsB1ldA56g1QN2ljE7LFFF7Q1jltjGN2
-- n5kIt7hHOuhFHXWChu9JjOj1jAoj2DnqzlkawODYVrnUBSasznjMip7+r0hNR70Y
-- VgUpP75Ok6YYNhnAcce4GMrqT46r/121+/rg/4B43cuHoyH74oYf
-- -----END-SIGNATURE-----