import QtQuick 2.0
import QtMultimedia 5.6
import "../Scripts/TCUtils.js" as TCUtils

/** \file VideoPlayer.qml

Provides a Video component along with a progress bar and play/pause button. The progress bar
can be scrubbed back and forth to reposition the play head.

*/

Rectangle {
    id : video_comp
    property variant video_frame : video_frame
    property alias source: video_frame.source
    property alias autoPlay:        video_frame.autoPlay
    property alias hasVideo:        video_frame.hasVideo
    property alias error:        video_frame.error
    color : "black"
    
    Rectangle { 
        id : video_layout
        anchors.left : parent.left
        anchors.top : parent.top
        anchors.right : parent.right
        anchors.bottom: parent.bottom
        color : "transparent"

        VideoExt {
            id : video_frame
            anchors.left : parent.left
            anchors.top : parent.top
            anchors.right : parent.right
            anchors.bottom: parent.bottom
            autoPlay: true;


            onBufferProgressChanged:  {
                scrub_bar.width = (progress.width - thumb.width) * bufferProgress
            }

            onVisibleChanged: {
                // when video gets hidden we stop it,
                // when it gets shown we auto play
                if ( visible == true ) {
                    play()
                } else {
                    stop()
                }
            }

            MouseArea {
                anchors.fill: parent
                hoverEnabled : true

                onClicked: {
                    if ( video_frame.playbackState == MediaPlayer.PlayingState  ) {
                        video_frame.pause()
                    } else {
                        video_frame.play()
                    }
                }

                onEntered: {
                    timer_hide_playbk_ctrls.stop()
                    overlay.visible = true
                }

                onExited: {
                    timer_hide_playbk_ctrls.restart() // overlay.visible = false
                }

                Timer {
                    id : timer_hide_playbk_ctrls
                    interval: 750; running: false; repeat: false
                    onTriggered: overlay.visible = false
                }


                // Playback controls
                Rectangle {
                    id : overlay
                    anchors.left : parent.left
                    anchors.right : parent.right
                    anchors.bottom : parent.bottom
                    height : TCUtils.dpiScale(15)
                    visible : false
                    opacity : 0.9
                    gradient : Gradient {
                        GradientStop { position : 0; color : Qt.hsla( 0,0, 0.2, 1) }
                        GradientStop { position : 0.1; color : Qt.hsla( 0,0, 0.2, 1) }
                        GradientStop { position : 0.5; color : Qt.hsla( 0,0, 0.1, 1) }
                        GradientStop { position : 0.51; color : Qt.hsla( 0,0, 0.05, 1) }
                        GradientStop { position : 1.0; color : Qt.hsla( 0,0, 0, 1) }
                    }

                    // Play / Pause Button
                    Image {
                        id : play_pause
                        source :
                        {
                            if ( video_frame.playbackState == MediaPlayer.PlayingState  ) {
                                playpause_ma.containsMouse ? "highlightPause.png" : "pause.png";
                            } else {
                                playpause_ma.containsMouse ? "highlightPlay.png" : "play.png";
                            }
                        }
                        width: {playpause_ma.containsMouse ? TCUtils.dpiScale(14) : TCUtils.dpiScale(10)}
                        height: {playpause_ma.containsMouse ? TCUtils.dpiScale(14) : TCUtils.dpiScale(10)}
                        fillMode: Image.PreserveAspectFit
                        smooth : true
                        anchors.verticalCenter: parent.verticalCenter
                        x : (parent.height - width ) / 2.0

                        MouseArea {
                            id : playpause_ma
                            anchors.fill : parent
                            hoverEnabled : true
                            onClicked : {
                                if ( video_frame.playbackState == MediaPlayer.PlayingState  ) {
                                    video_frame.pause()
                                } else {
                                    video_frame.play()
                                }
                            }
                        }
                    }
                    Rectangle {
                        id : separator
                        x : parent.height
                        anchors.top : parent.top
                        anchors.bottom : parent.bottom
                        width : TCUtils.dpiScale(2)
                        clip : true
                        // There's no way in QML right now to draw a gradient at an angle
                        // Since we want this one to go left to right instead of up and down,
                        // we clip the drawing of children and draw a large rotated rectangle inside
                        Rectangle {
                            rotation : 90
                            width : parent.height
                            height : parent.height
                            gradient : Gradient {
                                GradientStop { position : 0; color : Qt.hsla( 0,0, 0.3, 1) }
                                GradientStop { position : 0.5; color : Qt.hsla( 0,0, 0.2, 1) }
                                GradientStop { position : 1.0; color : Qt.hsla( 0,0, 0, 1) }
                            }
                        }
                    }
                    // Progress / Scrubbing Bar
                    Rectangle {
                        id : progress
                        anchors.left : separator.right
                        anchors.top : parent.top
                        anchors.bottom : parent.bottom
                        anchors.right : parent.right
                        color : "transparent"
                        BorderImage {
                            id : scrub_bar
                            source : "scrubBar.png"
                            x : thumb.width / 2
                            width : (parent.width - thumb.width) * 1
                            anchors.verticalCenter : parent.verticalCenter
                            border.left : TCUtils.dpiScale(5)
                            border.right : TCUtils.dpiScale(5)
                            border.top : 0
                            border.bottom : 0
                        }
                        Image {
                            id : thumb
                            x : (video_frame.position / video_frame.duration) * scrub_bar.width

                            anchors.verticalCenter : parent.verticalCenter
                            source : "thumb.png"
                            width: TCUtils.dpiScale(12)
                            height: TCUtils.dpiScale(11)
                            fillMode: Image.PreserveAspectFit
                            smooth: true
                        }

                        Component.onCompleted:
                        {
                            toolClipView.setItemCursor(overlay, "PointingHandCursor", "progress");
                        }

                        MouseArea {
                            id : overlay_ma
                            anchors.fill : parent
                            onPositionChanged : {
                                video_frame.seek( Math.max(0, Math.min( 1,( mouse.x / width ))) * video_frame.duration )

                                // When scrubbing to the end, the video stops and no scrubbing updates are visible,
                                // so we set it to pause again which gives us frame updates.
                                if ( video_frame.playbackState == MediaPlayer.StoppedState )
                                {
                                    video_frame.pause()
                                }
                            }
                            onPressed : {
                                video_frame.seek( Math.max(0, Math.min( 1, ( mouse.x / width ))) * video_frame.duration )
                            }
                        }
                    }
                }

            }

        }       
	}
}
