import QtQuick 2.0
import com.autodesk.toolclips.qml 1.0
import "../Scripts/TCUtils.js" as TCUtils

/** \file TitleShortcutText.qml

A tool tip header item that consists of a tool name and a keyboard shortcut.

*/

Item {
    id: title_shortcut
    height: Math.max( title.height, shortcut.height )
    property int maxTitleWidth: TCUtils.dpiScale(304)
    property int shortcutWidth: ((shortcut.text == "") ? 0 : (shortcut.width + root.shortcutSpacing))

    function getTitleWidth( ) {

        // Use our title text measure method for determining the proper contentWidth.
        // For plaintext that has manual line breaks set by the dev, we don't restrict the width
        // to our maxTitleWidth for wrapping and use the width of the manual wrap instead.
        if ( (title.textFormat == Text.PlainText && title.text.indexOf('\n') != -1) )
        {
            // Measure wrapped text and give it some space to do a default layout with the manual line breaks.
            var w = toolClipView.measureWidth( title.font, title.textFormat, 4000, title.text );

            // Return the wrapped title width + shortcut width.
            return w + shortcutWidth;
        }
        else
        {
            // Measure wrapped text within the given max width constraints.
            var w = toolClipView.measureWidth( title.font, title.textFormat, maxTitleWidth - shortcutWidth, title.text );

            // For rich text we need to take the implicitWidth into account,
            // cause html breaks doesn't seem to affect the contentWidget.
            w = (title.textFormat == Text.RichText) ? Math.min( title.implicitWidth, w ) : w;

            // add shortcut width to title
            w = w + shortcutWidth;

            // clip to our max tool clip width
            w = Math.min( w, maxTitleWidth );

            return w;
        }
    }

    Text {
        id: title
        text: toolClipView.interpolate( toolclip.attribute("title") )
        font.pixelSize: TCUtils.dpiScale(11)
        horizontalAlignment: Text.AlignLeft
        color : root.textColor
        anchors.left: parent.left
        anchors.right: parent.right
        anchors.rightMargin: shortcutWidth
        textFormat : { toolClipView.mightBeRichText( text ) ? Text.RichText : Text.PlainText }
        wrapMode : Text.WordWrap
    }

    Text {
        id: shortcut
        text: if ( toolclip.hasAttribute("shortcut") ) { "(" + toolclip.attribute("shortcut") + ")" } else { "" }
        font: title.font
        horizontalAlignment: Text.AlignRight
        color : root.textColor
        anchors.right : parent.right
    }
}
