﻿//
// Revit IFC Import library: this library works with Autodesk(R) Revit(R) to import IFC files.
// Copyright (C) 2013  Autodesk, Inc.
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
//

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Autodesk.Revit.DB;
using Autodesk.Revit.DB.IFC;
using Revit.IFC.Common.Utility;
using Revit.IFC.Common.Enums;
using Revit.IFC.Import.Enums;
using Revit.IFC.Import.Utility;

namespace Revit.IFC.Import.Data
{
    /// <summary>
    /// Represents an IfcObject.
    /// </summary>
    public abstract class IFCObject : IFCObjectDefinition
    {
        private string m_ObjectType = null;

        private IDictionary<string, IFCPropertySetDefinition> m_IFCPropertySets = null;

        private HashSet<IFCTypeObject> m_IFCTypeObjects = null;

        /// <summary>
        /// The object type.
        /// </summary>
        public string ObjectType
        {
            get { return m_ObjectType; }
        }

        /// <summary>
        /// The property sets.
        /// </summary>
        public IDictionary<string, IFCPropertySetDefinition> PropertySets
        {
            get 
            {
                if (m_IFCPropertySets == null)
                    m_IFCPropertySets = new Dictionary<string, IFCPropertySetDefinition>();
                return m_IFCPropertySets; 
            }
        }

        /// <summary>
        /// The type objects.
        /// </summary>
        /// <remarks>IFC Where rule for IfcObject states that we expect at most 1 item in this set.</remarks>
        public HashSet<IFCTypeObject> TypeObjects
        {
            get 
            {
                if (m_IFCTypeObjects == null)
                    m_IFCTypeObjects = new HashSet<IFCTypeObject>();
                return m_IFCTypeObjects; 
            }
        }

        /// <summary>
        /// Creates or populates Revit elements based on the information contained in this class.
        /// </summary>
        /// <param name="doc">The document.</param>
        protected override void Create(Document doc)
        {
            base.Create(doc);
        }

        /// <summary>
        /// Cleans out the IFCEntity to save memory.
        /// </summary>
        public override void CleanEntity()
        {
            base.CleanEntity();
        
            m_ObjectType = null;

            m_IFCPropertySets = null;

            m_IFCTypeObjects = null;
        }

        /// <summary>
        /// Gets the shape type from the IfcObject, depending on the file version and entity type.
        /// </summary>
        /// <param name="ifcObjectDefinition">The associated handle.</param>
        /// <returns>The shape type, if any.</returns>
        protected override string GetShapeType(IFCAnyHandle ifcObjectDefinition)
        {
            string shapeTypeName = "PredefinedType";

            // Not all entity types have any predefined type; for IFC2x3, some have a "ShapeType" instead of a "PredefinedType".
            if ((EntityType == IFCEntityType.IfcProject) ||
                (EntityType == IFCEntityType.IfcSite) ||
                (EntityType == IFCEntityType.IfcBuilding) ||
                (EntityType == IFCEntityType.IfcBuildingStorey))
                return null;

            if (IFCImportFile.TheFile.SchemaVersion < IFCSchemaVersion.IFC4)
            {
                // The following have "PredefinedType", but are out of scope for now:
                // IfcCostSchedule, IfcOccupant, IfcProjectOrder, IfcProjectOrderRecord, IfcServiceLifeFactor
                // IfcStructuralAnalysisModel, IfcStructuralCurveMember, IfcStructuralLoadGroup, IfcStructuralSurfaceMember
                if ((EntityType == IFCEntityType.IfcRamp) ||
                    (EntityType == IFCEntityType.IfcRoof) ||
                    (EntityType == IFCEntityType.IfcStair))
                    shapeTypeName = "ShapeType";
                else if ((EntityType != IFCEntityType.IfcCovering) &&
                    (EntityType != IFCEntityType.IfcElementAssembly) &&
                    (EntityType != IFCEntityType.IfcFooting) &&
                    (EntityType != IFCEntityType.IfcPile) &&
                    (EntityType != IFCEntityType.IfcRailing) &&
                    (EntityType != IFCEntityType.IfcSlab) &&
                    (EntityType != IFCEntityType.IfcTendon))
                    return null;
            }

            try
            {
                return IFCAnyHandleUtil.GetEnumerationAttribute(ifcObjectDefinition, shapeTypeName);
            }
            catch
            {
            }

            return null;
        }

        /// <summary>
        /// Processes IfcObject attributes.
        /// </summary>
        /// <param name="ifcObject">The IfcObject handle.</param>
        protected override void Process(IFCAnyHandle ifcObject)
        {
            base.Process(ifcObject);

            m_ObjectType = IFCAnyHandleUtil.GetStringAttribute(ifcObject, "ObjectType");

            HashSet<IFCAnyHandle> isDefinedByHandles = IFCAnyHandleUtil.GetAggregateInstanceAttribute
                <HashSet<IFCAnyHandle>>(ifcObject, "IsDefinedBy");

            if (isDefinedByHandles != null)
            {
                IFCPropertySetDefinition.ResetCounters();
                foreach (IFCAnyHandle isDefinedByHandle in isDefinedByHandles)
                {
                    if (IFCAnyHandleUtil.IsSubTypeOf(isDefinedByHandle, IFCEntityType.IfcRelDefinesByProperties))
                    {
                        ProcessIFCRelDefinesByProperties(isDefinedByHandle);
                    }
                    else if (IFCAnyHandleUtil.IsSubTypeOf(isDefinedByHandle, IFCEntityType.IfcRelDefinesByType))
                    {
                        ProcessIFCRelDefinesByType(isDefinedByHandle);
                    }
                    else
                        IFCImportFile.TheLog.LogUnhandledSubTypeError(isDefinedByHandle, IFCEntityType.IfcRelDefines, false);
                }
            }
        }

        /// <summary>
        /// Processes IfcRelDefinesByProperties.
        /// </summary>
        /// <param name="ifcRelDefinesByProperties">The IfcRelDefinesByProperties handle.</param>
        void ProcessIFCRelDefinesByProperties(IFCAnyHandle ifcRelDefinesByProperties)
        {
            IFCAnyHandle propertySetDefinition = IFCAnyHandleUtil.GetInstanceAttribute(ifcRelDefinesByProperties, "RelatingPropertyDefinition");

            if (IFCAnyHandleUtil.IsNullOrHasNoValue(propertySetDefinition))
            {
                IFCImportFile.TheLog.LogNullError(IFCEntityType.IfcPropertySetDefinition);
                return;
            }

            IFCPropertySetDefinition ifcPropertySet = IFCPropertySetDefinition.ProcessIFCPropertySetDefinition(propertySetDefinition);

            if (ifcPropertySet != null)
            {
                int propertySetNumber = 1;
                while (true)
                {
                    string name = (propertySetNumber == 1) ? ifcPropertySet.Name : ifcPropertySet.Name + " " + propertySetNumber.ToString();
                    if (PropertySets.ContainsKey(name))
                        propertySetNumber++;
                    else
                    {
                        PropertySets[name] = ifcPropertySet;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// Processes IfcRelDefinesByType.
        /// </summary>
        /// <param name="ifcRelDefinesByType">The IfcRelDefinesByType handle.</param>
        void ProcessIFCRelDefinesByType(IFCAnyHandle ifcRelDefinesByType)
        {
            IFCAnyHandle typeObject = IFCAnyHandleUtil.GetInstanceAttribute(ifcRelDefinesByType, "RelatingType");

            if (IFCAnyHandleUtil.IsNullOrHasNoValue(typeObject))
            {
                IFCImportFile.TheLog.LogNullError(IFCEntityType.IfcTypeObject);
                return;
            }

            if (!IFCAnyHandleUtil.IsSubTypeOf(typeObject, IFCEntityType.IfcTypeObject))
            {
                IFCImportFile.TheLog.LogUnhandledSubTypeError(typeObject, IFCEntityType.IfcTypeObject,false);
                return;
            }

            IFCTypeObject ifcTypeObject = IFCTypeObject.ProcessIFCTypeObject(typeObject);

            if (ifcTypeObject != null)
                TypeObjects.Add(ifcTypeObject);
        }

        /// <summary>
        /// Processes IfcObject handle.
        /// </summary>
        /// <param name="ifcObject">The IfcObject handle.</param>
        /// <returns>The IfcObject object.</returns>
        public static IFCObject ProcessIFCObject(IFCAnyHandle ifcObject)
        {
            if (IFCAnyHandleUtil.IsNullOrHasNoValue(ifcObject))
            {
                IFCImportFile.TheLog.LogNullError(IFCEntityType.IfcObject);
                return null;
            }

            IFCEntity cachedObject;
            if (IFCImportFile.TheFile.EntityMap.TryGetValue(ifcObject.StepId, out cachedObject))
                return (cachedObject as IFCObject); 
            
            if (IFCAnyHandleUtil.IsSubTypeOf(ifcObject, IFCEntityType.IfcProduct))
            {
                return IFCProduct.ProcessIFCProduct(ifcObject);
            }
            else if (IFCAnyHandleUtil.IsSubTypeOf(ifcObject, IFCEntityType.IfcProject))
            {
                return IFCProject.ProcessIFCProject(ifcObject);
            }

            IFCImportFile.TheLog.LogUnhandledSubTypeError(ifcObject, IFCEntityType.IfcObject, true);
            return null;
        }

        /// <summary>
        /// Creates or populates Revit element params based on the information contained in this class.
        /// </summary>
        /// <param name="doc">The document.</param>
        /// <param name="element">The element.</param>
        protected override void CreateParametersInternal(Document doc, Element element)
        {
            base.CreateParametersInternal(doc, element);

            if (element != null)
            {
                // Set "ObjectTypeOverride" parameter.
                string objectTypeOverride = ObjectType;
                if (!string.IsNullOrWhiteSpace(objectTypeOverride))
                    IFCPropertySet.AddParameterString(doc, element, "ObjectTypeOverride", objectTypeOverride, Id);
            }
        }

        /// <summary>
        /// Create property sets for a given element.
        /// </summary>
        /// <param name="doc">The document.</param>
        /// <param name="element">The element being created.</param>
        /// <param name="propertySetsCreated">A concatenated string of property sets created, used to filter schedules.</returns>
        public override void CreatePropertySets(Document doc, Element element, string propertySetsCreated)
        {
            if (PropertySets != null && PropertySets.Count > 0)
            {
                IFCParameterSetByGroup parameterGroupMap = IFCParameterSetByGroup.Create(element);
                foreach (IFCPropertySetDefinition propertySet in PropertySets.Values)
                {
                    string newPropertySetCreated = propertySet.CreatePropertySet(doc, element, parameterGroupMap);
                    if (propertySetsCreated == null)
                        propertySetsCreated = newPropertySetCreated;
                    else
                        propertySetsCreated += ";" + newPropertySetCreated;
                }

                Parameter propertySetList = element.LookupParameter("IfcPropertySetList");
                if (propertySetList != null)
                    propertySetList.Set(propertySetsCreated);
            }
        }
    }
}
